/* Masque des bits du registre SERCTL */
#define	PAREVEN 	(1 << 0)
#define TXBRK 		(1 << 1)
#define TXOPEN  	(1 << 2)
#define RESETERR	(1 << 3)
#define PAREN		(1 << 4)
#define	RXINTEN		(1 << 6)
#define	TXINTEN		(1 << 7)
#define PARBIT		(1 << 0)
#define	RXBRK 		(1 << 1)
#define	FRAMERR 	(1 << 2)
#define	OVERRUN 	(1 << 3)
#define PARERR 		(1 << 4)
#define	TXEMPTY 	(1 << 5)
#define	RXRDY 		(1 << 6)
#define	TXRDY 		(1 << 7)


#define		Comm_TailleMessage  	4   			/* Taille en octets des messages émis et reçus */


#define		Comm_SyncEmissMsg		0x81		    /* Valeur de l'octet de synchronisation pour l'émission d'un message */
#define		Comm_SyncRecepMsg		0x80		    /* Valeur de l'octet de synchronisation pour la réception d'un message */
#define		Comm_SyncLectureMsg	    0x82		    /* Valeur de l'octet de synchronisation pour la demande de lecture d'un message */

/* États possibles du protocole série */
#define		Comm_EtatInactif		0
#define		Comm_EtatRecepMsg		1
#define		Comm_EtatEmissMsg		2
#define 	Comm_EtatDmdLectMsg		3
#define 	Comm_Etat	 			uint8_t			

uint8_t     Comm_BufMsgEmiss[Comm_TailleMessage];   /* Buffer pour le message à émettre */
uint8_t     Comm_BufMsgRecep[Comm_TailleMessage];  	/* Buffer pour le message à émettre */
bool        Comm_MsgEmissEnAtt = false;             /* Indique s'il y a un message à émettre non intégralement traité */
bool        Comm_MsgRecepEnAtt = false;             /* Indique s'il y a un message reçu non intégralement traité */
Comm_Etat   Comm_EtatCourant = Comm_EtatInactif;    /* État courant du protocole série */
uint8_t    	Comm_NumOctetMsg;                       /* Numéro du prochain octet du message à émettre/recevoir */
bool		Comm_SyncEnvoye;				        /* L'octet de synchronisation a été envoyé */
bool        Comm_EmissionActivee = false;           /* Il y a des octets à émettre */
uint8_t		Comm_ValSERCTL = 0x5D;					/* Dernière valeur écrite dans SERCTL */


/* Écriture des bits du registre SERCTL */
/* BitsOn est le masque de bits à activer, BitsOff le masque des bits à désactiver */
void Comm_DefBitsSERCTL(BitsOn, BitsOff)
uint8_t BitsOn;
uint8_t BitsOff;
{
	Comm_ValSERCTL |= BitsOn;
	Comm_ValSERCTL &= ~BitsOff;
	serctl = Comm_ValSERCTL;
}


/* Remise à zéro des indicateurs d'erreur de communication */
void Comm_RazErreurs(void)
{
	serctl = Comm_ValSERCTL | RESETERR;
}


/* Initialisation de la communication */
void Comm_Init(void)
{
	/* 9600 bps */
	/* Horloge 1 MHz, rechargement auto activé, timer activé */
	/* Parité paire, mode collecteur ouvert, interruptions désactivées, reset des erreurs, break désactivé */
	timer4.reload = 12;			
	timer4.control = 0x18;		
	Comm_DefBitsSERCTL(TXOPEN | PAREN | PAREVEN, TXINTEN | RXINTEN | TXBRK | RESETERR);
		
	/* Remise à zéro des indicateurs d'erreur de communication */	
	Comm_RazErreurs();
	
	/* Effacement du buffer de réception */
	while (serctl & RXRDY) 
	{
		unsigned char c;
		c = serdat;
	}
}


/* Définition de l'état courant du protocole série */
void Comm_DefEtatCourant(Etat)
Comm_Etat Etat;
{
	Comm_EtatCourant = Etat;
	Comm_NumOctetMsg = 0;
	Comm_SyncEnvoye = false;

    /* Active ou non l'émission suivant l'état */
	switch (Etat)
	{
	    /* Communication inactive ou réception d'un message en cours : émission désactivée */
        case Comm_EtatInactif:
        case Comm_EtatRecepMsg:
        Comm_EmissionActivee = false;
        break;

        /* Émission d'un message ou demande de lecture d'un message : émission activée */
        case Comm_EtatEmissMsg:
        case Comm_EtatDmdLectMsg:
        Comm_EmissionActivee = true;
        break;
    }
}


/* Gestion de l'émission des octets */
void Comm_GestionEmission(void)
{
	while (Comm_EmissionActivee)
	{
		uint8_t OctetEmis;	
		
        switch (Comm_EtatCourant)
        {
            /* Émission d'un message */
            case Comm_EtatEmissMsg:
            {
                if (Comm_SyncEnvoye)
                {
                    /* Émission d'un octet de données du message */
                    OctetEmis = Comm_BufMsgEmiss[Comm_NumOctetMsg++];

                    if (Comm_NumOctetMsg == Comm_TailleMessage)
                    {
                        /* Message complètement envoyé : mise à jour des états */
                        Comm_MsgEmissEnAtt = false;
						if (!Comm_MsgRecepEnAtt)
						{					
							Comm_DefEtatCourant(Comm_EtatDmdLectMsg);
						}
						else
						{
							Comm_DefEtatCourant(Comm_EtatInactif);
						}
                    }
                }
                else
                {
                    /* Émission de l'octet de synchronisation */
                    OctetEmis = Comm_SyncEmissMsg;
                    Comm_SyncEnvoye = true;
                }
            }
            break;

            /* Émission d'une demande de lecture de message */
            case Comm_EtatDmdLectMsg:
            OctetEmis = Comm_SyncLectureMsg;
            Comm_DefEtatCourant(Comm_EtatInactif);
            break;

            default:
            /* Autre état : ne doit pas se produire -> désactivation de l'émission */
            Comm_EmissionActivee = false;
            break;
        }
		
		/* Émission de l'octet */
		while (!(serctl & 0x80));		/* Attend que l'émetteur soit prêt */
		serdat = OctetEmis;
    }
}


/* Gestion de la réception des octets */
void Comm_GestionReception(OctetRecu, ErreurReception)
uint8_t OctetRecu;
uint8_t ErreurReception;
{
	if (!ErreurReception)
	{
		/* Octet reçu sans erreur */
		if (OctetRecu & (1 << 7))
		{
			/* Octet de synchronisation */
			switch (OctetRecu)
			{
				/* Octet de synchronisation de réception d'un message */
				case Comm_SyncRecepMsg:
				if ((Comm_EtatCourant == Comm_EtatInactif) && (!Comm_MsgRecepEnAtt))
				{
					/* Pas de message déjà reçu non traité -> réception du message */
					Comm_DefEtatCourant(Comm_EtatRecepMsg);
				}
				break;

				/* Autre cas : octet ignoré */
				default:
				break;
			}
		}
		else
		{
			/* Octet de données */
			switch (Comm_EtatCourant)
			{
				/* Réception d'un message en cours */
				case Comm_EtatRecepMsg:
				{
					/* Écriture de l'octet reçu dans le message */
					Comm_BufMsgRecep[Comm_NumOctetMsg++] = OctetRecu;

					if (Comm_NumOctetMsg == Comm_TailleMessage)
					{
						/* Message complètement reçu : mise à jour des états */
						Comm_MsgRecepEnAtt = true;
						Comm_DefEtatCourant(Comm_EtatInactif);
					}
				}
				break;

				/* Autre cas : octet ignoré */
				default:
				break;
			}
		}
	}
	else
	{
		/* Erreur de communication */
		Comm_RazErreurs();
		
		switch (Comm_EtatCourant)
		{
			/* Émission d'un message : réémission du message depuis le début */
			case Comm_EtatEmissMsg:
			Comm_DefEtatCourant(Comm_EtatEmissMsg);
			break;

			/* Autre cas : arrêt de la communication en cours */
			default:
			Comm_DefEtatCourant(Comm_EtatInactif);
			break;
		}
	}
}


/* Émission/réception d'un message */
/* Renvoie true si un message a été reçu */
bool Comm_TransferMsg(BufMsgEmiss, BufMsgRecep)
uint8_t *BufMsgEmiss;
uint8_t *BufMsgRecep;
{
    bool MsgRecu;

    if ((BufMsgRecep != NULL) && Comm_MsgRecepEnAtt)
    {
		
		/* Copie du message reçu dans le buffer */
		{	
			uint8_t i1;
			for (i1 = 0; i1 < Comm_TailleMessage; i1++) BufMsgRecep[i1] = Comm_BufMsgRecep[i1];
		}
        Comm_MsgRecepEnAtt = false;

        MsgRecu = true;
    }
    else
    {
        /* Aucun message n'a été reçu */
        MsgRecu = false;
    }

    /* Émission du message et récupération du message suivant */
    if ((BufMsgEmiss != NULL) && (!Comm_MsgEmissEnAtt))
    {
        /* Copie du message dans le buffer d'émission */
        {
			uint8_t i2;
			for (i2 = 0; i2 < Comm_TailleMessage; i2++) Comm_BufMsgEmiss[i2] = BufMsgEmiss[i2];	
		}
        Comm_MsgEmissEnAtt = true;
    }

    if (Comm_MsgEmissEnAtt)
    {
        /* Émission du message (suivie automatiquement par la récupération du message suivant) */
        Comm_DefEtatCourant(Comm_EtatEmissMsg);
    }
    else
    {
        if (!Comm_MsgRecepEnAtt)
        {
            /* Pas de message reçu non traité : demande de lecture du message suivant */
            Comm_DefEtatCourant(Comm_EtatDmdLectMsg);
        }
    }
	
	Comm_GestionEmission();

    return MsgRecu;
}


/* Interruption série */
SER() interrupt
{
	unsigned char input;

	DisableIRQ(4);
	POKE(0x10,0xFD80);	/* INTRST */

	// Réception d'un octet
	if (serctl & 0x40)  
	{
		Comm_GestionReception(serdat, false);
	}

	EnableIRQ(4);
}
