#include <Gamebuino-Meta.h>

#define MAP_WIDTH   128
#define MAP_HEIGHT  64

#define TILE_WIDTH          8
#define TILE_HEIGHT         8
#define TILES_PASSABLE_END  26
#define TILE_GRASS          3
#define TILE_CAVE           7
#define TILE_WATER_START    17
#define TILE_WATER_END      26
#define TILE_ELEMENT        101

#define WALK_DELAY              5
#define BUTTON_DELAY            5
#define ARROW_DELAY             5
#define TEXT_DELAY              40
#define TEXT_BLINK_DELAY        70
#define TEXT_BUTTON_DELAY       1
#define FADE_DELAY              70
#define BATTLE_TEXT_BLINK_DELAY 40
#define BATTLE_DAMAGES_DELAY    30

#define BATTLE_FREQUENCY  20 // percent

#define NO_RESET    true
#define WITH_RESET  false

#define STATS_MENU_MAP    0
#define STATS_MENU_BATTLE 1
#define STATS_MENU_ITEM   2

Color pico_8_palette[] = {
  (Color)0x0000,  // BLACK
  (Color)0x194a,  // DARK-BLUE
  (Color)0x792a,  // DARK-PURPLE
  (Color)0x042a,  // DARK-GREEN
  (Color)0xaa86,  // BROWN
  (Color)0x5aa9,  // DARK-GRAY
  (Color)0xc618,  // LIGHT-GRAY
  (Color)0xff9d,  // WHITE
  (Color)0xf809,  // RED
  (Color)0xfd00,  // ORANGE
  (Color)0xff64,  // YELLOW
  (Color)0x0726,  // GREEN
  (Color)0x2d7f,  // BLUE
  (Color)0x83b3,  // INDIGO
  (Color)0xfbb5,  // PINK
  (Color)0xfe75   // PEACH
};

Color pico_8_palette_dark[] = {
  (Color)0x0000,  // BLACK
  (Color)0x0004,  // DARK-BLUE
  (Color)0x4804,  // DARK-PURPLE
  (Color)0x02a3,  // DARK-GREEN
  (Color)0x7900,  // BROWN
  (Color)0x2923,  // DARK-GRAY
  (Color)0x9492,  // LIGHT-GRAY
  (Color)0xcdf6,  // WHITE
  (Color)0xc803,  // RED
  (Color)0xCB80,  // ORANGE
  (Color)0xcdc0,  // YELLOW
  (Color)0x0580,  // GREEN
  (Color)0x03d9,  // BLUE
  (Color)0x522d,  // INDIGO
  (Color)0xCa2e,  // PINK
  (Color)0xcccf   // PEACH
};

Color pico_8_palette_darker[] = {
  (Color)0x0000,  // BLACK
  (Color)0x0000,  // DARK-BLUE
  (Color)0x1800,  // DARK-PURPLE
  (Color)0x0000,  // DARK-GREEN
  (Color)0x4000,  // BROWN
  (Color)0x0000,  // DARK-GRAY
  (Color)0x5aec,  // LIGHT-GRAY
  (Color)0x9C70,  // WHITE
  (Color)0x9800,  // RED
  (Color)0x99e0,  // ORANGE
  (Color)0x9c40,  // YELLOW
  (Color)0x0400,  // GREEN
  (Color)0x0253,  // BLUE
  (Color)0x1887,  // INDIGO
  (Color)0x9888,  // PINK
  (Color)0x9b48   // PEACH
};

// debug
byte player_x = 7;
byte player_y = 9;

//byte player_x = 36;
//byte player_y = 7;
byte player_direction = 3;
byte player_animation = 0;
short player_kills = 0;
int camera_x;
int camera_y;
byte tmp;

#include "maps.h"
#include "sprites.h"
#include "strings.h"
#include "stats.h"
#include "scripts.h"
#include "graphics.h"

const byte zone00[] = {
  SWOOTY, PIGGY, GROLDO, FEUXDINO, WATAWAMP, ROWPIM, BUZZCOR, PURBIRB, TINKERELLE, BOBERL,
  PLARADI, HATCELL, MOGMINE, CRUB, FLIPA, SNEG, PILO, LIMEGOO, DOGSHADE, PUKESUN, HUEVO
};
const byte zone01[] = { ROWPIM, PIGGY, PURBIRB };
const byte zone02[] = { GROLDO, FEUXDINO, TINKERELLE };
const byte zone03[] = { LIMEGOO };
const byte zone04[] = { MOGMINE, HATCELL, PLARADI };
const byte zone05[] = { DOGSHADE, BUZZCOR, SNEG };
const byte zone06[] = { WATAWAMP, PILO };
const byte zone07[] = { FLIPA, CRUB };

const byte map_menu[] = {
  // x, y, w, h, option id
  45, 0, 35, 27,
  M_MONSTER, M_ITEM, M_SAVE, M_RESET
};

const byte first_release_menu[] = {
  // x, y, w, h, option id
  22, 24, 35, 15,
  M_FIRST, M_RELEASE
};

const byte use_drop_menu[] = {
  // x, y, w, h, option id
  28, 24, 23, 15,
  M_USE, M_DROP
};

const byte no_yes_menu[] = {
  // x, y, w, h, option id
  14, 29, 15, 11,
  M_NO, M_YES
};

void setup() {
  gb.begin();
  gb.display.colorIndex = pico_8_palette;
  player_sprite_set.setTransparentColor((Color)0xff9d);
}

void loop() {
  boolean object_collision = false;
  update_camera();
  update_map();
  while (true) {
    if (gb.update()) {
      int x_direction = -gb.buttons.repeat(BUTTON_LEFT, 1) * (player_x > 0) + gb.buttons.repeat(BUTTON_RIGHT, 1) * (player_x < MAP_WIDTH - 1);
      int y_direction = -gb.buttons.repeat(BUTTON_UP, 1) * (player_y > 0) + gb.buttons.repeat(BUTTON_DOWN, 1) * (player_y < MAP_HEIGHT - 1);
      if (!x_direction != !y_direction) {
        player_direction = (1 + x_direction) * (x_direction != 0);
        player_direction = (2 + y_direction) * (y_direction != 0 || player_direction == 0);

        // check for objects to collide
        object_collision = false;
        for (byte i = 0; i < sizeof(action_triggered_scripts); i += ACTION_SCRIPTS_LENGTH) {
          if (action_triggered_scripts[i + 3] > 0 &&
              flag[action_triggered_scripts[i + 2]] == 0 &&
              player_x + x_direction == action_triggered_scripts[i] &&
              player_y + y_direction == action_triggered_scripts[i + 1]) {
            object_collision = true;
            break;
          }
        }
 
        if (TILES_PASSABLE_END - map_[(player_y + y_direction)*MAP_WIDTH + player_x + x_direction] > 0 && !object_collision) {
          for (byte i = 1; i <= 8; i++) {
            player_animation += i % 2;
            player_animation *= player_animation < 2 && i < 8;
            camera_x = (player_x * TILE_WIDTH - gb.display.width() / 2 + 4 + i * x_direction);
            camera_x = camera_x * (camera_x > 0) + (MAP_WIDTH * TILE_WIDTH - gb.display.width() - camera_x) * (camera_x > MAP_WIDTH * TILE_WIDTH - gb.display.width());
            camera_y = player_y * TILE_HEIGHT - gb.display.height() / 2 + 4 + i * y_direction;
            camera_y = camera_y * (camera_y > 0) + (MAP_HEIGHT * TILE_HEIGHT - gb.display.height() - camera_y) * (camera_y > MAP_HEIGHT * TILE_HEIGHT - gb.display.height());
            draw_map(camera_x, camera_y);
            draw_player(player_x * TILE_WIDTH - camera_x + i * x_direction, player_y * TILE_WIDTH - camera_y + i * y_direction);
            gb.update();
            delay(WALK_DELAY);
          }
          player_x += x_direction;
          player_y += y_direction;

          // check for warps
          for (byte i = 0; i < sizeof(warps); i += WARPS_LENGTH) {
            if (player_x == warps[i] && player_y == warps[i + 1]) {
              fade_in(FADE_DELAY);
              player_x = warps[i + 2];
              player_y = warps[i + 3];
              update_camera();
              fade_out(FADE_DELAY);
              break;
            }
          }

          // check for walk-triggered events
          for (byte i = 0; i < sizeof(walk_triggered_scripts); i += WALK_SCRIPTS_LENGTH) {
            if (player_x == walk_triggered_scripts[i] & player_y == walk_triggered_scripts[i + 1]) {
              script(walk_triggered_scripts[i + 2]);
              break;
            }
          }

          // check for battles
          if(rand()%100 <= BATTLE_FREQUENCY){
            if(map_[player_y * MAP_WIDTH + player_x] == TILE_GRASS){
              if(player_x < 19 && player_y < 31){
                battle(zone01, sizeof(zone01));
              }else if(player_x < 35 && player_y < 56){
                battle(zone02, sizeof(zone02));
              }else if(player_x < 35){
                battle(zone03, sizeof(zone03));
              }else if(player_x > 108 && player_y > 38){
                battle(zone04, sizeof(zone04));
              }else if(player_x > 60 && player_y > 40){
                battle(zone05, sizeof(zone05));
              }else{
                battle(zone00, sizeof(zone00));
              }
            }else if(map_[player_y * MAP_WIDTH + player_x] == TILE_CAVE){
              battle(zone06, sizeof(zone06));
            }else if(map_[player_y * MAP_WIDTH + player_x] >= TILE_WATER_START && map_[player_y * MAP_WIDTH + player_x] <= TILE_WATER_END){
              battle(zone07, sizeof(zone07));
            }
          }

        }
        update_map();
      } else if (gb.buttons.repeat(BUTTON_A, BUTTON_DELAY)) {
        for (byte i = 0; i < sizeof(action_triggered_scripts); i += ACTION_SCRIPTS_LENGTH) {
          if (flag[action_triggered_scripts[i + 2]] == 0 &&
              player_x + (player_direction == 0) - (player_direction == 2) == action_triggered_scripts[i] &&
              player_y + (player_direction == 3) - (player_direction == 1) == action_triggered_scripts[i + 1]) {
            switch (action_triggered_scripts[i + 4]) {
              case TEXT :
                text((char*) strings[action_triggered_scripts[i + 5]], WITH_RESET);
                break;
              case SCRIPT :
                script(action_triggered_scripts[i + 5]);
                break;
            }
          }
        }
      }else if (gb.buttons.repeat(BUTTON_MENU, BUTTON_DELAY)) {
        tmp = 0;
map_menu_loop:
        gb.update();
        update_map();
        switch(menu(map_menu, sizeof(map_menu), true, tmp)){
          case M_MONSTER:
            stats_menu(STATS_MENU_MAP);
            tmp = 0;
            goto map_menu_loop;
          case M_ITEM:
            item_menu(false);
            tmp = 1;
            goto map_menu_loop;
          case M_SAVE:
          case M_RESET:
            break;
        }
        update_map();
      }
      
    }
  }
}

void text_frame() {
  gb.display.setColor(WHITE);
  gb.display.fillRect(0, 37, 80, 27);
  gb.display.setColor(BLACK);
  gb.display.drawRect(0, 37, 80, 27);
  gb.display.setCursor(2, 39);
}

void text_pause() {
  byte i = 0;
  byte quit = 0;
  while (!quit) {
    if (gb.update()) {
      i++;
      if (i % 2) {
        gb.display.setColor(WHITE);
      }
      gb.display.drawChar(gb.display.getCursorX(), gb.display.getCursorY(), 21, 1);
      gb.display.setColor(BLACK);
      gb.update();
      delay(TEXT_BLINK_DELAY);
      if (gb.buttons.repeat(BUTTON_A, TEXT_BUTTON_DELAY)) quit = 1;
    }
  }
}

byte text(const char string[], boolean no_reset) {
  if(!no_reset) text_frame();
  for (byte i = 0; i < strlen(string); i++) {
    char c = (char) string[i];
    switch ((byte) c) {
      case 10:
        gb.display.setCursor(2, gb.display.getCursorY()+6);
        break;
      case 112:
        text_pause();
        text_frame();
        break;
      default:
        gb.display.print(c);
        gb.update();
        delay(TEXT_DELAY);
    }
  }
  if(!no_reset){
    text_pause();
    update_map();
  }
}

byte handle_vertical_cursor(byte options_number, byte x, byte y, byte cursor_pos){
  byte prev_pos = (cursor_pos == 0);
  do{
    if (gb.update()) {
      if (cursor_pos != prev_pos) {
        gb.display.setColor(WHITE);
        gb.display.drawChar(x, y + 6 * (prev_pos), 16, 1);
        gb.display.setColor(BLACK);
        gb.display.drawChar(x, y + 6 * (cursor_pos), 16, 1);
        prev_pos = cursor_pos;
      }
      cursor_pos -= gb.buttons.repeat(BUTTON_UP, ARROW_DELAY) * (cursor_pos > 0);
      cursor_pos += gb.buttons.repeat(BUTTON_DOWN, ARROW_DELAY) * (cursor_pos < options_number-1);
      if(gb.buttons.repeat(BUTTON_B, BUTTON_DELAY)){ return M_CANCEL; }
    }
  }while(!gb.buttons.repeat(BUTTON_A, BUTTON_DELAY));
  return cursor_pos;
}

word menu(const byte data[], byte size, boolean frame, byte cursor_pos){
  if(frame){
    gb.display.setColor(WHITE);
    gb.display.fillRect((int) data[0], (int) data[1], (int) data[2], (int) data[3]);
    gb.display.setColor(BLACK);
    gb.display.drawRect((int) data[0], (int) data[1], (int) data[2], (int) data[3]);
  }
  for (byte i = 4; i < size; i++) {
    gb.display.setCursor(data[0]+2*frame+4, data[1]+2*frame+6*(i-4));
    gb.display.print(menu_strings[data[i]]);
  }
  byte option = handle_vertical_cursor(size-4, data[0]+2*frame, data[1]+2*frame, cursor_pos);
  if(option == M_CANCEL) return option;
  return data[4+option];
}

byte script(byte script_id) {
  byte i;
  switch (script_id) {
    case S_PHONE:
      if(!flag[F_PICOBALL]){
        text(str_phone, WITH_RESET);
        break;
      }
      text(str_no_messages, WITH_RESET);
      break;
    case S_PICOBALL:
      flag[F_PICOBALL] = 1;
      add_picomon(0, SWOOTY, 2, calc_stat(SWOOTY, 2, HP), 0, MO_TACKLE, MO_LEER, NO_MOVE, NO_MOVE);
      text(str_starter, WITH_RESET);
      break;
    case S_LEAVE_HOUSE:
      if(!flag[F_PICOBALL]){
        text(str_take_picoball, WITH_RESET);
        break;
      }
      fade_in(FADE_DELAY);
      player_x = 12;
      player_y = 7;
      update_camera();
      fade_out(FADE_DELAY);
      break;
    case S_PICOSTOP:
      for(i = 0; i < sizeof(player_picomons)/2; i += PLAYER_PICOMONS_LENGTH){
        if(player_picomons[i+ID] == MISSING_NO) break;
        player_picomons[i+CUR_HP] = calc_stat(player_picomons[i+ID], player_picomons[i+LVL], HP);
      }
      //save_game();
      text("YOUR GAME HAS\nAUTOMATICALLY BEEN\nSAVED!", WITH_RESET);
      text("ALL YOUR MONSTERS\nARE HEALED...", WITH_RESET);
      if(player_kills < 2){
        text("DEFEAT A COUPLE OF\nWILD MONSTERS TO\nGET A PRIZE FROM\nTHIS PICOSTOP!", WITH_RESET);
      }else{
        player_kills -= 2;
        i = search_item(NO_ITEM);
        if(i != NO_ITEM){
          player_items[i] = rand()%3;
          player_items[i] = 3*(rand()%100 < 5);
          text("YOU PRESS THE\nBUTTON.\nTHE PICOSTOP DROPS\nA...", WITH_RESET);
          text("...", NO_RESET);
          text(item_names[player_items[i]], NO_RESET);
          text("!\p", NO_RESET);
          update_map();
        }else{
          text("YOUR INVENTORY IS\nFULL!", WITH_RESET);
        }
      }
      break;
  }
}

void add_picomon(byte position, byte id, byte lvl, unsigned short hp, unsigned short xp, byte move_1, byte move_2, byte move_3, byte move_4){
  player_picomons[position+ID] = (unsigned short) id;
  player_picomons[position+LVL] = (unsigned short) lvl;
  player_picomons[position+CUR_HP] = hp;
  player_picomons[position+XP] = xp;
  player_picomons[position+MOVE_1] = (unsigned short) move_1;
  player_picomons[position+MOVE_2] = (unsigned short) move_2;
  player_picomons[position+MOVE_3] = (unsigned short) move_3;
  player_picomons[position+MOVE_4] = (unsigned short) move_4;
  return;
}

unsigned short tmp_array[]={ 0, 0, 0, 0, 0, 0, 0, 0, 0 };
void swap_picomons(byte first_id, byte second_id){
  memcpy(&tmp_array, &player_picomons[first_id*PLAYER_PICOMONS_LENGTH], PLAYER_PICOMONS_LENGTH);
  memcpy(&player_picomons[first_id*PLAYER_PICOMONS_LENGTH], &player_picomons[second_id*PLAYER_PICOMONS_LENGTH], PLAYER_PICOMONS_LENGTH);
  memcpy(&player_picomons[second_id*PLAYER_PICOMONS_LENGTH], &tmp_array, PLAYER_PICOMONS_LENGTH);
}

void sort_picomons(){
  for(byte i = 0; i < sizeof(player_picomons)/2/PLAYER_PICOMONS_LENGTH-1; i++){
    if(player_picomons[i*PLAYER_PICOMONS_LENGTH+ID] == MISSING_NO && player_picomons[(i+1)*PLAYER_PICOMONS_LENGTH+ID] != MISSING_NO){
      swap_picomons(i, i+1);
    }
  }
  return;
}

unsigned short calc_stat(byte id, byte lvl, byte stat){
  return floor((picomon_stats[id*PICOMON_STATS_LENGTH+stat]*2*lvl/100)+lvl+10);
}

byte stats_menu(byte option){
  if(player_picomons[ID] == MISSING_NO) return M_CANCEL;
  byte picomon_count, cursor_pos = 0, prev_pos = 1;
  while(true){
    if(gb.update()){
      do{
        if (gb.update()) {
          if (cursor_pos != prev_pos) {
            gb.display.fill(WHITE);
            gb.display.setColor(BLACK);
            gb.display.drawRect(0, 0, 43, 64);
            gb.display.drawRect(37, 37, 43, 27);
            gb.display.setColor(WHITE);
            gb.display.fillRect(38, 38, 41, 25);
            for(picomon_count = 0; picomon_count < sizeof(player_picomons)/2; picomon_count += PLAYER_PICOMONS_LENGTH){
              if(player_picomons[picomon_count+ID] == MISSING_NO) break;
              tile_set.setFrame(picomon_stats[player_picomons[picomon_count+ID]*PICOMON_STATS_LENGTH+SPRITE_ID]);
              gb.display.drawImage(48, 1+picomon_count/PLAYER_PICOMONS_LENGTH*9, tile_set);
              draw_HP_box(57, 4+picomon_count/PLAYER_PICOMONS_LENGTH*9, 22, 4, (float) player_picomons[picomon_count+CUR_HP]/calc_stat(player_picomons[picomon_count+ID], player_picomons[picomon_count+LVL], HP));
            }
            picomon_count = picomon_count/PLAYER_PICOMONS_LENGTH;
            
            gb.display.setColor(WHITE);
            gb.display.drawChar(44, 3 + 9 * (prev_pos), 16, 1);
            gb.display.setColor(BLACK);
            gb.display.drawChar(44, 3 + 9 * (cursor_pos), 16, 1);
            prev_pos = cursor_pos;

            gb.display.setCursor(2, 2);
            gb.display.print(picomon_names[player_picomons[(cursor_pos*PLAYER_PICOMONS_LENGTH)+ID]]);
            tile_set.setFrame(TILE_ELEMENT+picomon_stats[player_picomons[(cursor_pos*PLAYER_PICOMONS_LENGTH)+ID]*PICOMON_STATS_LENGTH+ELEMENT]);
            gb.display.drawImage(2, 8, tile_set);
            gb.display.setCursor(11, 10);
            gb.display.print(element_names[picomon_stats[player_picomons[(cursor_pos*PLAYER_PICOMONS_LENGTH)+ID]*PICOMON_STATS_LENGTH+ELEMENT]]);
            gb.display.setCursor(2, 17);
            gb.display.print("LVL");
            gb.display.setCursor(2, 23);
            gb.display.print("XP");
            gb.display.setCursor(2, 29);
            gb.display.print("HP");
            gb.display.setCursor(2, 35);
            gb.display.print("SPD");
            gb.display.setCursor(2, 41);
            gb.display.print("DEF");
            gb.display.setCursor(2, 47);
            gb.display.print("ATK");
            gb.display.setCursor(26, 23);
            gb.display.print("/");
            gb.display.setCursor(26, 29);
            gb.display.print("/");
            gb.display.setCursor(18, 17);
            gb.display.print(player_picomons[(cursor_pos*PLAYER_PICOMONS_LENGTH)+LVL]);
            tmp = player_picomons[(cursor_pos*PLAYER_PICOMONS_LENGTH)+XP];
            gb.display.setCursor(14+4*(tmp < 100)+4*(tmp < 10), 23);
            gb.display.print(tmp);
            tmp = pow(player_picomons[(cursor_pos*PLAYER_PICOMONS_LENGTH)+LVL]*10, 1.2);
            gb.display.setCursor(30+4*(tmp < 100)+4*(tmp < 10), 23);
            gb.display.print(tmp);
            tmp = player_picomons[(cursor_pos*PLAYER_PICOMONS_LENGTH)+CUR_HP];
            gb.display.setCursor(14+4*(tmp < 100)+4*(tmp < 10), 29);
            gb.display.print(tmp);
            tmp = calc_stat(player_picomons[cursor_pos*PLAYER_PICOMONS_LENGTH+ID], player_picomons[cursor_pos*PLAYER_PICOMONS_LENGTH+LVL], HP);
            gb.display.setCursor(30+4*(tmp < 100)+4*(tmp < 10), 29);
            gb.display.print(tmp);
            gb.display.setCursor(18, 35);
            gb.display.print(calc_stat(player_picomons[cursor_pos*PLAYER_PICOMONS_LENGTH+ID], player_picomons[cursor_pos*PLAYER_PICOMONS_LENGTH+LVL], SPEED));
            gb.display.setCursor(18, 41);
            gb.display.print(calc_stat(player_picomons[cursor_pos*PLAYER_PICOMONS_LENGTH+ID], player_picomons[cursor_pos*PLAYER_PICOMONS_LENGTH+LVL], DEFENSE));
            gb.display.setCursor(18, 47);
            gb.display.print(calc_stat(player_picomons[cursor_pos*PLAYER_PICOMONS_LENGTH+ID], player_picomons[cursor_pos*PLAYER_PICOMONS_LENGTH+LVL], ATTACK));
            for(byte i = 0; i < 4; i++){
              tmp = player_picomons[cursor_pos*PLAYER_PICOMONS_LENGTH+MOVE_1+i];
              if(tmp == NO_MOVE) break;
              gb.display.setCursor(39, 39+i*6);
              gb.display.print(move_names[tmp]);
            }
          }
          cursor_pos -= gb.buttons.repeat(BUTTON_UP, ARROW_DELAY) * (cursor_pos > 0);
          cursor_pos += gb.buttons.repeat(BUTTON_DOWN, ARROW_DELAY) * (cursor_pos < picomon_count-1);
          if(gb.buttons.repeat(BUTTON_B, BUTTON_DELAY)) return M_CANCEL;
        }
      }while(!gb.buttons.repeat(BUTTON_A, BUTTON_DELAY));
      if(option == STATS_MENU_ITEM) return cursor_pos;
      delay(50);
      switch(menu(first_release_menu, sizeof(first_release_menu), true, 0)){
        case M_FIRST:
          swap_picomons(0, cursor_pos);
          prev_pos = (cursor_pos == 0);
          if(option == STATS_MENU_BATTLE && cursor_pos != 0) return 1;
          break;
        case M_RELEASE:
          if(picomon_count > 1){
            gb.display.setColor(WHITE);
            gb.display.fillRect(12, 21, 55, 21);
            gb.display.setColor(BLACK);
            gb.display.drawRect(12, 21, 55, 21);
            gb.display.setCursor(14, 23);
            text("ARE YOU SURE?", NO_RESET);
            delay(50);
            if(menu(no_yes_menu, sizeof(no_yes_menu), false, 0) == M_YES){
              player_picomons[cursor_pos*PLAYER_PICOMONS_LENGTH+ID] = MISSING_NO;
              sort_picomons();
              cursor_pos = (cursor_pos-1)*(cursor_pos > 0);
            }
          }else{
            text("YOU CAN'T RELEASE\nYOUR LAST MONSTER.\p", WITH_RESET);
          }
          prev_pos = (cursor_pos == 0);
          if(option == STATS_MENU_BATTLE) return 1;
          break;
      }
    }
  }
}

byte item_menu(boolean during_battle){
  byte items_count, cursor_pos = 0, prev_pos = 1;
  while(true){
    if(gb.update()){
      if(player_items[0] == NO_ITEM) return M_CANCEL;
      update_map();
      do{
        if (gb.update()) {
          if (cursor_pos != prev_pos) {
            gb.display.setColor(WHITE);
            gb.display.fillRect(21, 0, 59, 64);
            gb.display.setColor(BLACK);
            gb.display.drawRect(21, 0, 59, 64);
            for(items_count = 0; items_count < sizeof(player_items); items_count ++){
              if(player_items[items_count] == NO_ITEM) break;
              gb.display.setCursor(27, 2+6*items_count);
              gb.display.print(item_names[player_items[items_count]]);
            }
            gb.display.setColor(WHITE);
            gb.display.drawChar(23, 2 + 6 * (prev_pos), 16, 1);
            gb.display.setColor(BLACK);
            gb.display.drawChar(23, 2 + 6 * (cursor_pos), 16, 1);
            prev_pos = cursor_pos;
          }
          cursor_pos -= gb.buttons.repeat(BUTTON_UP, 1) * (cursor_pos > 0);
          cursor_pos += gb.buttons.repeat(BUTTON_DOWN, 1) * (cursor_pos < items_count-1);
          if(gb.buttons.repeat(BUTTON_B, BUTTON_DELAY)) return M_CANCEL;
        }
      }while(!gb.buttons.repeat(BUTTON_A, BUTTON_DELAY));
        delay(50);
        switch(menu(use_drop_menu, sizeof(use_drop_menu), true, 0)){
          case M_USE:
            switch(player_items[cursor_pos]){
              case I_POTION:
                delay(50);
                prev_pos = stats_menu(STATS_MENU_ITEM);
                if(prev_pos != M_CANCEL){
                  text_frame();
                  if(player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+CUR_HP > 0]){
                    tmp = calc_stat(player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+ID], player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+LVL], HP);
                    if(player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+CUR_HP] == tmp){
                      text(picomon_names[player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+ID]], NO_RESET);
                      text(" IS\nALREADY FULLY\nHEALED!\p", NO_RESET);
                      break;
                    }
                    player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+CUR_HP] += 20;
                    if(player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+CUR_HP] > tmp){
                      player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+CUR_HP] = tmp;
                      text("HEALING ", NO_RESET);
                      text(picomon_names[player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+ID]], NO_RESET);
                      text("\nBACK TO FULL HP!\p", NO_RESET);
                    }else{
                      text("HEALING ", NO_RESET);
                      text(picomon_names[player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+ID]], NO_RESET);
                      text("\nFOR 20 HP!\p", NO_RESET);
                    }
                    player_items[cursor_pos] = NO_ITEM;
                    sort_items();
                    cursor_pos = (cursor_pos-1)*(cursor_pos > 0);
                  }else{
                    text(picomon_names[player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+ID]], NO_RESET);
                    text(" CAN'T\nBE HEALED. HE IS\nDEFEATED.\p", NO_RESET);
                    break;
                  }
                  if(during_battle) return ITEM_HANDLED;
                }
                break;
              case I_RESURRECT:
                delay(50);
                prev_pos = stats_menu(STATS_MENU_ITEM);
                if(prev_pos != M_CANCEL){
                  text_frame();
                  if(player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+CUR_HP > 0]){
                    text(picomon_names[player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+ID]], NO_RESET);
                    text(" IS\nNOT KO!\p", NO_RESET);
                    break;
                  }
                  player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+CUR_HP] = calc_stat(player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+ID], player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+LVL], HP)/2;
                  text(picomon_names[player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+ID]], NO_RESET);
                  text(" HAS\nBEEN HEALED.\p", NO_RESET);
                  player_items[cursor_pos] = NO_ITEM;
                  sort_items();
                  if(during_battle) return ITEM_HANDLED;
                  cursor_pos = (cursor_pos-1)*(cursor_pos > 0);
                }
                break;
              case I_CANDY:
                delay(50);
                prev_pos = stats_menu(STATS_MENU_ITEM);
                if(prev_pos != M_CANCEL){
                  text_frame();
                  if(rand()%2){
                    player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+DOUBLE_XP] = 1;
                    text(picomon_names[player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+ID]], NO_RESET);
                    text(" LOVES\nCANDY!\p", NO_RESET);
                  }else{
                    text(picomon_names[player_picomons[prev_pos*PLAYER_PICOMONS_LENGTH+ID]], NO_RESET);
                    text(" HATES\nCANDY!\p", NO_RESET);
                  }
                  player_items[cursor_pos] = NO_ITEM;
                  sort_items();
                  if(during_battle) return ITEM_HANDLED;
                  cursor_pos = (cursor_pos-1)*(cursor_pos > 0);
                }
                break;
              case I_PICOBALL:
                if(during_battle) return I_PICOBALL;
              case I_SMOKEBALL:
                if(during_battle) return I_SMOKEBALL;
              default:
                text_frame();
                text("CAN'T USE THAT\nHERE.\p", NO_RESET);
                break;
            }
          case M_DROP:
            gb.display.setColor(WHITE);
            gb.display.fillRect(12, 21, 55, 21);
            gb.display.setColor(BLACK);
            gb.display.drawRect(12, 21, 55, 21);
            gb.display.setCursor(14, 23);
            text("ARE YOU SURE?", NO_RESET);
            delay(50);
            if(menu(no_yes_menu, sizeof(no_yes_menu), false, 0) == M_YES){
              player_items[cursor_pos] = NO_ITEM;
              sort_items();
              cursor_pos = (cursor_pos-1)*(cursor_pos > 0);
            }
            if(during_battle) return ITEM_HANDLED;
            break;
        }
        prev_pos = (cursor_pos == 0);
    }
  }
}

byte search_item(byte item){
  for(byte i = 0; i < PLAYER_ITEMS_LENGTH; i++){
    if(player_items[i] == item) return i;
  }
  return NO_ITEM;
}

void sort_items(){
  for(byte i = 0; i < sizeof(player_items)-1; i++){
    if(player_items[i] == NO_ITEM && player_items[i+1] != NO_ITEM){
      player_items[i] = player_items[i+1];
      player_items[i+1] = NO_ITEM;
    }
  }
  return;
}

#define B_FIGHT   1
#define B_MONSTER 2
#define B_ITEM    3
#define B_RUN     4

unsigned short opponent_stats[] = {
  // picomon id, lvl, current hp, xp, move 1 id, move 2 id, move 3 id, move 4 id
  MISSING_NO, 0, 0, 0, NO_MOVE, NO_MOVE, NO_MOVE, NO_MOVE,
};

void battle(const byte zone[], byte size){
  for(byte i = 0; i < 3; i++){
    fade_in(BATTLE_TEXT_BLINK_DELAY);
    fade_out(BATTLE_TEXT_BLINK_DELAY);
  }

  byte opponent_move_count = 0, escape_tries = 0, move_accuracy = 95, effectiveness, i;
  boolean critical;
  short damages;

  opponent_stats[ID] = zone[rand()%size];
  int tmp = player_picomons[LVL] - 3 + rand() % 4;
  if(tmp < 1) tmp = 1;
  if(tmp > 100) tmp = 100;
  opponent_stats[LVL] = tmp;
  opponent_stats[CUR_HP] = calc_stat(opponent_stats[ID], opponent_stats[LVL], HP);

  opponent_stats[MOVE_1] = rand()%(MO_MAX+1);
  if(rand()%2){
    opponent_move_count++;
    do{ opponent_stats[MOVE_2] = rand()%(MO_MAX+1); }while(opponent_stats[MOVE_2] == opponent_stats[MOVE_1]);
    if(rand()%2 && opponent_stats[LVL] > 5){
      opponent_move_count++;
      do{ opponent_stats[MOVE_3] = rand()%(MO_MAX+1); }while(opponent_stats[MOVE_3] == opponent_stats[MOVE_2]);
      if(rand()%2 && opponent_stats[LVL] > 10){
        opponent_move_count++;
        do{ opponent_stats[MOVE_4] = rand()%(MO_MAX+1); }while(opponent_stats[MOVE_4] == opponent_stats[MOVE_3]);
      }
    }
  }

  draw_battle_screen();

  battle_text("A WILD ");
  battle_text(picomon_names[opponent_stats[ID]]);
  battle_text("\nAPPEARS!\p");

  while(true){
    if(gb.update()){
battle_loop:
      byte cursor_pos = 1;
battle_loop_skip_cursor:
      draw_battle_screen();
      gb.display.setCursor(23, 51);
      gb.display.print("FIGHT");
      gb.display.setCursor(51, 51);
      gb.display.print("MONSTER");
      gb.display.setCursor(23, 57);
      gb.display.print("ITEM");
      gb.display.setCursor(51, 57);
      gb.display.print("RUN");
      switch(handle_battle_cursor(19, 28, cursor_pos, 4, false)){
        case B_FIGHT:
          battle_text_frame();
          gb.display.setCursor(6, 51);
          gb.display.print(move_names[player_picomons[MOVE_1]]);
          i = 1;
          if(player_picomons[MOVE_2] != NO_MOVE){
            gb.display.setCursor(45, 51);
            gb.display.print(move_names[player_picomons[MOVE_2]]);
            i ++;
          }
          if(player_picomons[MOVE_3] != NO_MOVE){
            gb.display.setCursor(6, 57);
            gb.display.print(move_names[player_picomons[MOVE_3]]);
            i ++;
          }
          if(player_picomons[MOVE_4] != NO_MOVE){
            gb.display.setCursor(45, 57);
            gb.display.print(move_names[player_picomons[MOVE_4]]);
            i ++;
          }
          delay(50);
          i = handle_battle_cursor(2, 39, 1, i, true);
          switch(i){
            case M_CANCEL:
              goto battle_loop;
            default:
              battle_text_frame();
              battle_text(picomon_names[player_picomons[ID]]);
              battle_text(" USED\n");
              battle_text(move_names[player_picomons[MOVE_1+i-1]]);
              battle_text("!\p");
              if(rand()%100 > move_accuracy-1){
                battle_text(move_names[player_picomons[MOVE_1+i-1]]);
                battle_text(" MISSED!\p");
                break;
              }
              effectiveness = calc_effectiveness(picomon_stats[player_picomons[ID]*PICOMON_STATS_LENGTH+ELEMENT], picomon_stats[opponent_stats[ID]*PICOMON_STATS_LENGTH+ELEMENT]);
              critical = rand()%100 < (calc_stat(player_picomons[ID], player_picomons[LVL], SPEED)/512)*100;
              damages = calc_damage(player_picomons[MOVE_1+i-1], player_picomons[LVL], calc_stat(player_picomons[ID], player_picomons[LVL], ATTACK), calc_stat(opponent_stats[ID], opponent_stats[LVL], DEFENSE), effectiveness, critical);
              tmp = opponent_stats[CUR_HP] - damages;
              tmp *= (tmp > 0);
              for(i = opponent_stats[CUR_HP]; i > tmp; i--){
                draw_HP_box(15, 14, 38, 4, (float) i/calc_stat(opponent_stats[ID], opponent_stats[LVL], HP));
                gb.update();
                delay(BATTLE_DAMAGES_DELAY);
              }
              draw_HP_box(15, 14, 38, 4, (float) tmp/calc_stat(opponent_stats[ID], opponent_stats[LVL], HP));
              opponent_stats[CUR_HP] = tmp;
              if(effectiveness == 2) battle_text("IT'S VERY\nEFFECTIVE!\p");
              if(critical) battle_text("A CRITICAL HIT!\p");
              if(tmp == 0){
                battle_text("YOU DEFEATED THE\n");
                battle_text(picomon_names[opponent_stats[ID]]);
                battle_text("!\p");
                player_kills++;
                tmp = (5*opponent_stats[LVL]+rand()%(5*opponent_stats[LVL]))*(player_picomons[DOUBLE_XP]+1);
                player_picomons[XP] = tmp;
                battle_text("YOUR ");
                battle_text(picomon_names[player_picomons[ID]]);
                battle_text("\nEARNED ");
                gb.display.print(tmp);
                battle_text(" XP!\p");
                if(player_picomons[XP] > pow(player_picomons[LVL]*10, 1.2) && player_picomons[LVL] < 100){
                  player_picomons[LVL]++;
                  player_picomons[CUR_HP] = calc_stat(player_picomons[ID], player_picomons[LVL], HP);
                  player_picomons[XP] = 0;
                  battle_text("YOUR ");
                  battle_text(picomon_names[player_picomons[ID]]);
                  battle_text("\nIS NOW LEVEL ");
                  gb.display.print(player_picomons[LVL]);
                  battle_text("!\p");
                }
                return;
              }
              break;
          }
          break;
        case B_MONSTER:
          if(stats_menu(STATS_MENU_BATTLE) != M_CANCEL){
            break;
          }
          cursor_pos = 2;
          goto battle_loop_skip_cursor;
        case B_ITEM:
          i = item_menu(true);
          if(i != M_CANCEL){
            draw_battle_screen();
            switch(i){
              case I_PICOBALL:
                for(i = 0; i < sizeof(player_picomons)/PLAYER_PICOMONS_LENGTH/2+1; i++){
                  if(i == sizeof(player_picomons)/PLAYER_PICOMONS_LENGTH/2){
                    battle_text("YOU ALREADY HAVE 4\nMONSTERS.\pRELEASE SOME\nMONSTERS TO CATCH\pNEW ONES.");
                    goto battle_loop_skip_cursor;
                  }
                  if(player_picomons[i*PLAYER_PICOMONS_LENGTH+ID] == MISSING_NO) break;
                }
                player_items[cursor_pos] = NO_ITEM;
                sort_items();
                if(rand()%10 < 2+opponent_stats[CUR_HP]/calc_stat(opponent_stats[ID], opponent_stats[LVL], HP)*2){
                  battle_text("CLONK!\pOH NO, YOU MISSED\nTHE ");
                  battle_text(picomon_names[opponent_stats[ID]]);
                  battle_text("!\p");
                }else{
                  battle_text("YOU CAUGHT THE\n");
                  battle_text(picomon_names[opponent_stats[ID]]);
                  battle_text("!\p");
                  memcpy(&player_picomons[i*PLAYER_PICOMONS_LENGTH+ID], &opponent_stats[ID], PLAYER_PICOMONS_LENGTH);
                  return;
                }
                break;
              case I_SMOKEBALL:
                battle_text("YOU USE THE\nSMOKEBALL!\p");
                battle_text("YOU RAN AWAY!\p");
                player_items[cursor_pos] = NO_ITEM;
                sort_items();
                return;
              default:
                break; 
            }
            break;
          }
          cursor_pos = 3;
          goto battle_loop_skip_cursor;
        case B_RUN:
          battle_text_frame();
          if(rand() % 255 < calc_stat(player_picomons[ID], player_picomons[LVL], SPEED)*32/((calc_stat(opponent_stats[ID], opponent_stats[LVL], SPEED)/4)%256) + escape_tries *30){
            battle_text("YOU RAN AWAY!\p");
            return;
          }else{
            escape_tries ++;
            battle_text("YOU TRY TO RUN\nAWAY... BUT FAIL!\p");
          }
          break;
      }
      // opponent turn
      i = opponent_stats[MOVE_1+rand()%opponent_move_count];
      draw_battle_screen();
      battle_text(picomon_names[opponent_stats[ID]]);
      battle_text(" USED\n");
      battle_text(move_names[i]);
      battle_text("!\p");
      if(rand()%100 > move_accuracy-1){
        battle_text(move_names[i]);
        battle_text(" MISSED!\p");
      }else{
        effectiveness = calc_effectiveness(picomon_stats[opponent_stats[ID]*PICOMON_STATS_LENGTH+ELEMENT], picomon_stats[player_picomons[ID]*PICOMON_STATS_LENGTH+ELEMENT]);
        critical = rand()%100 < (calc_stat(opponent_stats[ID], opponent_stats[LVL], SPEED)/512)*100;
        damages = calc_damage(i, opponent_stats[LVL], calc_stat(opponent_stats[ID], opponent_stats[LVL], ATTACK), calc_stat(player_picomons[ID], player_picomons[LVL], DEFENSE), effectiveness, critical);
        tmp = player_picomons[CUR_HP] - damages;
        tmp *= (tmp > 0);
        for(i = player_picomons[CUR_HP]; i > tmp; i--){
          draw_HP_box(39, 42, 38, 4, (float) i/calc_stat(player_picomons[ID], player_picomons[LVL], HP));
          gb.update();
          delay(BATTLE_DAMAGES_DELAY);
        }
        draw_HP_box(39, 42, 38, 4, (float) i/calc_stat(player_picomons[ID], player_picomons[LVL], HP));
        player_picomons[CUR_HP] = tmp;
        if(effectiveness == 2) battle_text("IT'S VERY\nEFFECTIVE!\p");
        if(critical) battle_text("A CRITICAL HIT!\p");
        if(tmp == 0){
          battle_text(picomon_names[player_picomons[ID]]);
          battle_text(" IS KO!\p");
          for(i = 0; i < sizeof(player_picomons)/PLAYER_PICOMONS_LENGTH/2+1; i++){
            if(i == sizeof(player_picomons)/PLAYER_PICOMONS_LENGTH/2){
              battle_text("YOU WERE DEFEATED!\p");
              battle_text("...YOU AWAKE AT\nYOUR LAST SAVE\nPOINT.\p");
              //load_game();
              return;
            }
            if(player_picomons[i*PLAYER_PICOMONS_LENGTH+CUR_HP] > 0) break;
          }
          do{
            delay(50);
            i = stats_menu(STATS_MENU_ITEM);
            if(player_picomons[i*PLAYER_PICOMONS_LENGTH+CUR_HP] == 0) i = M_CANCEL;
          }while(i == M_CANCEL);
          swap_picomons(0, i);
        }
      }
    }
  }
}

void draw_battle_screen(){
    gb.display.fill(WHITE);
  // draw opponent stuff
  gb.display.drawFastVLine(1, 0, 20);
  gb.display.drawFastHLine(2, 19, 52);
  gb.display.setCursor(3, 1);
  gb.display.print(picomon_names[opponent_stats[ID]]);
  gb.display.setCursor(3, 7);
  gb.display.print(":L");
  gb.display.print(opponent_stats[LVL]);
  gb.display.setCursor(3, 13);
  gb.display.print("HP:");
  draw_HP_box(15, 14, 38, 4, (float) opponent_stats[CUR_HP]/calc_stat(opponent_stats[ID], opponent_stats[LVL], HP));
  tile_set.setFrame(picomon_stats[opponent_stats[ID]*PICOMON_STATS_LENGTH+SPRITE_ID]);
  gb.display.drawImage(55, 1, tile_set, -24, 24);

  gb.display.drawFastVLine(78, 28, 20);
  gb.display.drawFastHLine(26, 47, 52);
  gb.display.setCursor(78-strlen(picomon_names[player_picomons[ID]])*4, 30);
  gb.display.print(picomon_names[player_picomons[ID]]);
  gb.display.setCursor(66-4*(player_picomons[LVL] > 9)-4*(player_picomons[LVL] > 99), 36);
  gb.display.print(":L");
  gb.display.print(player_picomons[LVL]);
  gb.display.setCursor(27, 41);
  gb.display.print("HP:");
  draw_HP_box(39, 42, 38, 4, (float) player_picomons[CUR_HP]/calc_stat(player_picomons[ID], player_picomons[LVL], HP));
  tile_set.setFrame(picomon_stats[player_picomons[ID]*PICOMON_STATS_LENGTH+SPRITE_ID]);
  gb.display.drawImage(1, 24, tile_set, 24, 24);

  battle_text_frame();
}

void draw_HP_box(byte x, byte y, byte w, byte h, float hp_ratio){
  gb.display.setColor(WHITE);
  gb.display.fillRect(x, y, w, h);
  gb.display.setColor((Color)0xf809);
  gb.display.fillRect(x+1, y+1, (w-2)*hp_ratio, h-2);
  gb.display.setColor(BLACK);
  gb.display.drawRect(x, y, w, h);
}

void battle_text_frame(){
  gb.display.setColor(WHITE);
  gb.display.fillRect(1, 50, 78, 13);
  gb.display.setColor(BLACK);
  gb.display.drawRect(0, 49, 80, 15);
  gb.display.setCursor(2, 51);
}

byte battle_text(const char string[]) {
  for (byte i = 0; i < strlen(string); i++) {
    char c = (char) string[i];
    switch ((byte) c) {
      case 10:
        gb.display.setCursor(2, gb.display.getCursorY()+6);
        break;
      case 112:
        text_pause();
        battle_text_frame();
        break;
      default:
        gb.display.print(c);
        gb.update();
        delay(TEXT_DELAY);
    }
  }
}

byte handle_battle_cursor(byte x, byte w, byte cursor_pos, byte options_number, boolean allow_cancel){
  byte prev_pos = 4;
  do{
    if (gb.update()) {
      if (cursor_pos != prev_pos) {
        gb.display.setColor(WHITE);
        gb.display.drawChar(x + w * !(prev_pos % 2), 51 + 6 * (prev_pos > 2), 16, 1);
        gb.display.setColor(BLACK);
        gb.display.drawChar(x + w * !(cursor_pos % 2), 51 + 6 * (cursor_pos > 2), 16, 1);
        prev_pos = cursor_pos;
      }
      cursor_pos -= 1 * gb.buttons.repeat(BUTTON_LEFT, ARROW_DELAY) * (cursor_pos > 1 && cursor_pos != 3);
      cursor_pos += 1 * gb.buttons.repeat(BUTTON_RIGHT, ARROW_DELAY) * ((cursor_pos == 1 && options_number > 1) || (cursor_pos == 3 && options_number > 3));
      cursor_pos -= 2 * gb.buttons.repeat(BUTTON_UP, ARROW_DELAY) * (cursor_pos > 2);
      cursor_pos += 2 * gb.buttons.repeat(BUTTON_DOWN, ARROW_DELAY) * ((cursor_pos == 1 && options_number > 2) || (cursor_pos == 2 && options_number > 3));
      if(allow_cancel && gb.buttons.repeat(BUTTON_B, BUTTON_DELAY)){ return M_CANCEL; }
    }
  }while(!gb.buttons.repeat(BUTTON_A, BUTTON_DELAY));
  return cursor_pos;
}

byte calc_effectiveness(byte from_element, byte target_element){
  switch(from_element){
    case T_AIR:
      if(target_element == T_WATER) return 2;
      break;
    case T_FIRE:
      if(target_element == T_AIR || target_element == T_EARTH) return 2;
      break;
    //case T_EARTH:
    case T_WATER:
      if(target_element == T_FIRE) return 2;
      break;
  }
  return 1;
}

short calc_damage(byte from_move, byte from_level, byte from_attack, byte target_defense, byte effectiveness, boolean critical){
  short damage = (((2*from_level+10)/250) * (from_attack/target_defense)*moves_base[from_move] + 2)*effectiveness;
  if(critical) damage *= 2;
  return damage;
}

