/**********************************************************************
 * Copyright (c) 2008, j. montgomery                                  *
 * All rights reserved.                                               *
 *                                                                    *
 * Redistribution and use in source and binary forms, with or without *
 * modification, are permitted provided that the following conditions *
 * are met:                                                           *
 *                                                                    *
 * + Redistributions of source code must retain the above copyright   *
 *   notice, this list of conditions and the following disclaimer.    *
 *                                                                    *
 * + Redistributions in binary form must reproduce the above copyright*
 *   notice, this list of conditions and the following disclaimer     *
 *   in the documentation and/or other materials provided with the    *
 *   distribution.                                                    *
 *                                                                    *
 * + Neither the name of j. montgomery's employer nor the names of    *
 *   its contributors may be used to endorse or promote products      *
 *   derived from this software without specific prior written        *
 *   permission.                                                      *
 *                                                                    *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS*
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT  *
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS  *
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE     *
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,*
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES           *
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR *
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) *
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,*
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)      *
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED*
 * OF THE POSSIBILITY OF SUCH DAMAGE.                                 *
 **********************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace DnDns.Records
{
    /// <summary>
    /// RFC 1035:
    /// 
    /// 3.4.1. A RDATA format
    ///  +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
    ///  |                    ADDRESS                    |
    ///  +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
    ///  
    ///  where:
    ///  
    ///  ADDRESS         A 32 bit Internet address.
    ///  
    ///  Hosts that have multiple Internet addresses will have multiple A
    ///  records.
    ///  
    ///  A records cause no additional section processing.  The RDATA section of
    ///  an A line in a master file is an Internet address expressed as four
    ///  decimal numbers separated by dots without any imbedded spaces (e.g.,
    ///  "10.2.0.52" or "192.0.5.6").
    /// 
    /// </summary>
    public sealed class ARecord : DnsRecordBase, IDnsRecord
    {
        private string _hostAddress;

        /// <summary>
        /// The answer host address for the DNS A Record.
        /// </summary>
        public string HostAddress
        {
            get { return _hostAddress; }
        }

        internal ARecord(RecordHeader dnsHeader) : base(dnsHeader) { }

        public override void ParseRecord(ref MemoryStream ms)
        {
            _hostAddress = ms.ReadByte() + "." + ms.ReadByte() + "." + ms.ReadByte() + "." + ms.ReadByte();
            _answer = "Address: " + _hostAddress;
        }
    }
}
