package display;

import com.haxepunk.Entity;
import com.haxepunk.HXP;
import com.haxepunk.utils.Draw;
import com.haxepunk.Scene;

import environment.Level;

import openfl.geom.Matrix;
import openfl.Vector;

/* Used to apply a bending effect to Main.bendBuffer
   The direction of the effect is given by a curve calculated off an array of points by the DiscreteBezier class.
*/
class BendTransform
{
	// Those are calculated once and never change
	static private var indices:Vector<Int>;
	static private var uv:Vector<Float>;
	//
	static private var maxValue:Float;
	static private var level:Level;
	static private var iter_amount:Float; // how much of the curve is processed in a single iteration
	static private var steps:Int;
	static private var width:Float;
	
	static public function init(_steps:Int, points:Array<Position>, _level:Level)
	{
		level = _level;
		steps = Std.int(_steps * Main.bendBuffer.height / level.height);
		DiscreteBezier.init(points, _steps);
		DiscreteBezier.scale(level.height);
		iter_amount = Main.bendBuffer.height / level.height;
		// Build the triangle indices and the uv points once and for all
		uv = new Vector<Float>();
		width = level.width;
		for(k in 0 ... steps + 1)
		{
			uv.push((1 - width / Main.bendBuffer.width) / 2);
			uv.push(k / steps);
			uv.push((1 + width / Main.bendBuffer.width) / 2);
			uv.push(k / steps);
		}
		// We must draw steps * 2 triangles
		indices = new Vector<Int>();
		for(k in 0 ... steps * 2)
		{
			indices.push(k);
			indices.push(k + 1);
			indices.push(k + 2);
		}
	}
	
	// Use globals to apply the bending to Main.bendBuffer
	static public function apply(target:Scene)
	{
		var vertices = new Vector<Float>();
		var tbase = (HXP.camera.y - (Main.bendBuffer.height - HXP.height) / 2) / level.height;
		var tend = tbase + iter_amount;
		
		var origin = DiscreteBezier.curve((tend + tbase) / 2);
		origin.x -= HXP.halfWidth;
		origin.y -= HXP.halfHeight;
		var reached = new Array<Position>();
		reached.push(DiscreteBezier.curve(tbase));
		
		for(k in 0 ... steps + 1)
		{
			var t = tbase + k * (tend - tbase) / steps;
			var b = DiscreteBezier.curve(t), db = DiscreteBezier.dcurve(t);
			db = {x:-db.y, y:db.x}; // direct orthonormal vector to the tangent
			vertices.push(b.x + db.x * width / 2 - origin.x);
			vertices.push(b.y + db.y * width / 2 - origin.y);
			vertices.push(b.x - db.x * width / 2 - origin.x);
			vertices.push(b.y - db.y * width / 2 - origin.y);
		}
		target.sprite.graphics.beginBitmapFill(Main.bendBuffer, new Matrix());
		target.sprite.graphics.drawTriangles(vertices, indices, uv);
		target.sprite.graphics.endFill();
	}
	
	// Returns new x, y and angle values for the entity and its graphic
	static public function applyEntity(e:Entity) : {x:Float, y:Float, angle:Float}
	{
		var center = DiscreteBezier.curve((HXP.camera.y + HXP.halfHeight) / level.height);
		var local_e = {x:e.x - HXP.camera.x - HXP.halfWidth, y:e.y - HXP.camera.y - HXP.halfHeight};
		var t = e.y / level.height;
		var b = DiscreteBezier.curve(t), db = DiscreteBezier.dcurve(t);
		db = {x:db.y, y:-db.x}; // indirect orthonormal vector to the tangent
		return {x: b.x + local_e.x * db.x - center.x + HXP.camera.x + HXP.halfWidth, y:b.y + local_e.x * db.y - center.y + HXP.camera.y + HXP.halfHeight, angle:HXP.angle(0, 0, db.x, db.y)};
	}
}