package display;

import com.haxepunk.HXP.Position;

// Builds a B-spline made up of cubic Bézier curves and exposes a function that permits
// discretization of its values and its derivative's values

/* A Bézier curve is a polynomial which evaluates as follows :
	 		    n     /n\
	           \̅    |   |  i       n-i
	   Bn(t) = /_  Pi \i/  t (1 - t)
			   i=0
	 
	where the (Pi) are the n+1 control points. Here, we implement B-splines, which are several cubic Bézier curves glued
	together end-to-start. C0 continuity is guaranteed at the connection points.
	*/

class DiscreteBezier
{
    static private var interpolationPoints:Array<Position>;
	static private var curves:Array<Float>; // a sorted array 
	
	static private var curveLength:Float;
	
	static private function add(a:Position, b:Position) : Position
	{
		return {x:a.x + b.x, y:a.y + b.y};
	}
	
	static private function neg(a:Position) : Position
	{
		return {x:-a.x, y:-a.y};
	}
	
	static private function mult(a:Position, l:Float) : Position
	{
		return {x:a.x * l, y:a.y * l};
	}
	
	// Calculates the discrete, linear approximation of a Bézier curve given by its construction points.
	// The Bézier is calculated as a connection of several (at most) cubic Bézier curves.
    static public function init(points:Array<Position>, steps:Int)
    {
		var bcurves = new Array<Array<Position>>(); // hold the points making up the cubic Bézier curves
		bcurves.push(points.slice(0, 4));
		var k = Std.int(Math.max(0, (points.length - 4) / 3));
		for(i in 0 ... k)
		{
			bcurves.push(new Array<Position>());
			for(j in 0 ... 4)
				bcurves[i + 1].push(points[i * 3 + j + 3]);
		}
		var padding = points.length - 4 - k * 3;
		if(padding > 0)
		{
			bcurves.push(new Array<Position>());
			for(i in 0 ... padding)
				bcurves[k + 1].push(points[k * 3 + i + 3]);
		}
		var maxT = bcurves.length;
		
		curves = new Array<Float>();
		interpolationPoints = new Array<Position>();
		var l = 0.;
		curves.push(0);
		interpolationPoints.push(points[0]);
		
		for(k in 1 ... steps + 1)
		{
			interpolationPoints.push(bezier_curve(k / steps, bcurves));
			var dl = distance(interpolationPoints[k], interpolationPoints[k-1]);
			l += dl;
			curves.push(l);
		}
		for(k in 0 ... curves.length)
			curves[k] /= l;
		curveLength = l;
    }
	
	// Scale the curve so that it has a definite length, while keeping the same starting point
	static public function scale(targetLength:Float)
	{
		for(k in 1 ... interpolationPoints.length)
		{
			var temp = mult(add(interpolationPoints[k], neg(interpolationPoints[0])), targetLength / curveLength);
			interpolationPoints[k] = add(temp, interpolationPoints[0]);
		}
		curveLength = targetLength;
	}
	
	static inline private function length(a:Position) : Float
	{
		return Math.sqrt(a.x * a.x + a.y * a.y);
	}
	static inline private function distance(a:Position, b:Position) : Float
	{
		return Math.sqrt((a.x - b.x) * (a.x - b.x) + (a.y - b.y) * (a.y - b.y));
	}
	
	// Calculates a float to an integer power
	static private function ipow(v:Float, n:Int) : Float
	{
		if(n == 0) return 1;
		else
		{
			var t = ipow(v, Std.int(n / 2));
			if(n % 2 == 0)
				return t * t;
			else
				return v * t * t;
		}
	}
	
	// Returns the point of the Bézier curve calculated with parameter t
	// t ranges from 0 to n, where n is the number of curves of degree 3 or less making up the main curve
	// For any integer n, curve(n) returns the (n * 3 + 1)th point
	// curve(0) returns P1, curve(1) returns P4, curve(2) returns P(7) etc
	static private function bezier_curve(ct:Float, bcurves:Array<Array<Position>>) : Position
	{
		var ci = Std.int(ct); // curve index
		ci -= ((ct - ci) == 0 && ct > 0 ? 1 : 0);
		var t = ct - ci; // actual t step
		var c = 1; // binomial coefficient
		var r:Position = {x:0., y:0.};
		var n = bcurves[ci].length - 1;
		for(k in 0 ... n + 1)
		{
			var x = c * ipow(t, k) * ipow(1 - t, n - k);
			r.x += bcurves[ci][k].x * x;
			r.y += bcurves[ci][k].y * x;
			c = Std.int(c * (n - k) / (k + 1)); // poor Haxe doesn't know it's always integers
		}
		return r;
	}
	
	// Assumes 0 <= t < 1
	static private function findCurve(t:Float) : Int
	{
		var k = 0;
		while(t >= curves[k]) k++;
		return k -1;
	}
	
	// Returns the point of the discrete curve calculated with parameter t.
	// If t < 0, extends [P0, P1] so that length(0, -1) = curveLength.
	// If t > 1, extends [Pn-1, Pn] so that length(0, 2) = curveLength * 2.
	static public function curve(t:Float) : Position
	{
		if(t < 0)
			return add(interpolationPoints[0], mult(dcurve(0), t * curveLength));
		else if(t > 1)
			return add(interpolationPoints[interpolationPoints.length - 1], mult(dcurve(1), (t - 1) * curveLength));
		else
		{
			var k = findCurve(t);
			return add(interpolationPoints[k], mult(dcurve(t), (t - curves[k]) * curveLength));
		}
	}
	
	/* Calculates the unit tangent vector to the curve, ie its derivative, at the right of the point with parameter t, ie
		lim (f(x) - f(t)) / (x - t) when x -> t and x > t.
	   The same conditions on t as with curve(t) apply here.
	   If t < 0, returns dcurve(0).
	   If t > n, returns dcurve(n).
	*/
	static public function dcurve(t:Float) : Position
	{
		var b:Int;
		if(t < 0) b = 0;
		else if(t >= 1) b = curves.length - 2;
		else b = findCurve(t);
		var dt = add(interpolationPoints[b + 1], neg(interpolationPoints[b]));
		return mult(dt, 1 / length(dt));
	}
	
	// Populates an array with "steps + 1" discrete values of the curve (includes start and end points)
	// The array is assumed to be empty
	static public function populateSteps(steps:Int, r:Array<Position>) : Void
	{
		for(k in 0 ... steps + 1)
			r.push(curve(k / steps));
	}
	
	// Returns an array of "steps + 1" discrete values of the curve (includes start and end points)
	static public function generateSteps(steps:Int) : Array<Position>
	{
		var r = new Array<Position>();
		populateSteps(steps, r);
		return r;
	}
	
	// Cleans up the calculated arrays for new use via the init() function
	static public function cleanup()
	{
		
	}
}