/* This file is part of the MAYLIB libray.
   Copyright 2007-2009 Patrick Pelissier

This Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 3 of the License, or (at your
option) any later version.

This Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with th Library; see the file COPYING.LESSER.txt.
If not, write to the Free Software Foundation, Inc.,
51 Franklin St, Fifth Floor, Boston,
MA 02110-1301, USA. */

#include "may-impl.h"

static int
HalfExtendedEuclidian2 (may_t *s, may_t *g, may_t a, may_t b, may_t x)
{
  may_t a1, b1;
  may_mark_t mark;

  may_mark (mark);
  /* Naive & Slow implementation of the Euclidean algorithm */
  a1 = may_set_ui (1);
  b1 = may_set_ui (0);
  while (!may_zero_p (b)) {
    may_t q, r;
    int i = may_div_qr (&q, &r, a, b, x);
    if (i == 0)
      return 0;
    a = b;
    b = r;
    r = may_eval (may_sub_c (a1, may_mul_c (q, b1)));
    a1 = b1;
    b1 = r;
    may_compact_va (mark, &a, &b, &a1, &b1, NULL);
  }
  *s = a1;
  *g = a;
  return 1;
}

int
may_gcdex (may_t *s_p, may_t *t_p, may_t *g_p,
	   may_t a, may_t b, may_t c, may_t x)
{
  may_t s, t, g, q, r;
  int i;
  may_mark_t mark;

  may_mark (mark);
  s = t = g = 0;

  /* Compute (s, g) */
  i = HalfExtendedEuclidian2 (&s, &g, a, b, x);
  if (i == 0)
    return may_keep (mark, NULL), 0;

  /* Find the solution for c if requested */
  if (c) {
    i = may_div_qr (&q, &r, c, g, x);
    if (i == 0 || !may_zero_p (r))
      /*c is not in the ideal generated by A and B: there is no solution */
      return may_keep (mark, NULL), 0;
    s = may_mul (s, q);
    i = may_div_qr (NULL, &s, s, b, x);
    MAY_ASSERT (i != 0);
  } else
    c = g;

  /* Euclidean Division to compute t if needed*/
  if (t_p) {
    i = may_div_qr (&t, NULL,
		    may_eval (may_sub_c (c, may_mul_c (s, a))), b, x);
    MAY_ASSERT (i != 0);
  }

  /* Compact and return */
  if (s_p == NULL) s = NULL;
  if (g_p == NULL) g = NULL;
  may_compact_va (mark, &s, &t, &g, NULL);
  if (s_p) *s_p = s;
  if (t_p) *t_p = t;
  if (g_p) *g_p = g;
  return 1;
}
