/* This file is part of the MAYLIB libray.
   Copyright 2007-2009 Patrick Pelissier

This Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 3 of the License, or (at your
option) any later version.

This Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with th Library; see the file COPYING.LESSER.txt.
If not, write to the Free Software Foundation, Inc.,
51 Franklin St, Fifth Floor, Boston,
MA 02110-1301, USA. */

#include "may-impl.h"

/********************/
/* Define the tools */
/********************/

#define fail(s,x) (printf("Test '%s' failed:", s), may_dump (x), test_fail ())
#define fail2(s,s2,x) (printf("Test '%s' failed. Expected %s, got:", s, s2), may_dump (x), test_fail ())

static unsigned int fail_test_counter = 0;

void test_fail (void)
{
  printf ("\n");
  fail_test_counter ++;
  if (fail_test_counter > 7) {
    printf ("Too many errors. Abort\n");
    exit (1);
  }
}

void check (may_t x, const char *str, const char *func, unsigned int line)
{
  may_t y;
  may_mark ();
  y = may_parse_str (str);
  if (x == NULL || may_identical (x, y) != 0) {
    printf ("%s:\nError in test '%s' line %u.\n"
            "Expected: %s\n"
            "Got     : %s\n"
            "Dump got: ",
            may_get_version (), func, line,
	    str, x==NULL ? "@@NULL@@" : may_get_string (NULL, 0, x));
    may_dump (x);
    printf ("Dump exp: ");
    may_dump (y);
    test_fail ();
  }
  may_keep (NULL);
}

void check_bool (int b, const char *str_exp, const char *func, unsigned int line)
{
  if (!b) {
    printf ("%s:\nError in test '%s' line %u for condition '%s'.\n"
            "Expected TRUE, got FALSE instead.\n",
            may_get_version (), func, line, str_exp);
    test_fail ();
  }
}

void check_string (const char *get, const char *expected,
                   const char *func, unsigned int line)
{
  if (strcmp (get, expected) != 0) {
    printf ("%s:\nError in test '%s' line %u.\n"
            "Expected '%s', got '%s' instead.\n",
            may_get_version (), func, line,
            expected, get);
    test_fail ();
  }
}

void check_z (mpz_srcptr x, const char *str,
	      const char *func, unsigned int line)
{
  mpz_t z;
  may_mark ();
  mpz_init_set_str (z, str, 10);
  if (mpz_cmp (x, z) != 0) {
    printf ("%s:\nError in test '%s' line %u.\n"
            "Expected: %s\n"
            "Got     : %s\n",
            may_get_version (), func, line,
	    str, mpz_get_str (NULL, 10, x));
    test_fail ();
  }
  may_keep (NULL);
}

void check_si (long x, long s,
	       const char *func, unsigned int line)
{
  if (x != s) {
    printf ("%s:\nError in test '%s' line %u.\n"
            "Expected: %ld\n"
            "Got     : %ld\n",
            may_get_version (), func, line,
	    s, x);
    test_fail ();
  }
}

#define check(x,s) check(x,s,__func__, __LINE__)
#define check_bool(b) check_bool(b,#b, __func__,__LINE__)
#define check_z(x,s) check_z(x,s,__func__,__LINE__)
#define check_si(x,s) check_si(x,s,__func__,__LINE__)
#define check_string(g,e) check_string(g,e,__func__,__LINE__)


/******************************************************************************/
/************************* Now the test suite begins **************************/
/******************************************************************************/

void test_version ()
{
  char Buffer[100];

  sprintf (Buffer, "MAY V%u.%u.%u",
           MAY_MAJOR_VERSION, MAY_MINOR_VERSION, MAY_PATCHLEVEL_VERSION);
  if (strncmp (Buffer, may_get_version (), strlen (Buffer) ) != 0) {
    printf ("Error for get_version.\n Get: %s\nExpected: %s\n",
            may_get_version (), Buffer);
    exit (1);
  }
}

void test_restart (void)
{
  mpz_t z;
  void *top = may_g.Heap.top, *top2;

  may_kernel_stop ();
  mpz_init_set_ui (z, 17);
  top2 = may_g.Heap.top;
  mpz_add_ui (z, z, 23);
  mpz_clear (z);
  may_kernel_restart ();
  check_bool (top == top2);
}

void test_realloc (void)
{
  /* Bug of realloc if we try to realloc something just before the mark */
  may_mark();
  void * x = may_alloc (16);
  void * y = x;
  {
    may_mark ();
    x = may_realloc (x, 16, 32);
    check_bool (x != y);
    may_keep (NULL);
  }
  may_keep (NULL);
}

void test_op ()
{
  may_t x;

  may_mark ();

  x = may_parse_str ("{0,2,3,4,5,6,x+y}");
  if (may_nops (x) != 7)
    fail ("nop", x);
  if (may_identical (may_op (x, 3), may_set_ui (4)) != 0)
    fail ("op", x);

  /* Check MAP function */
  x = may_eval (may_map_c (x, may_sin_c));
  check (x, "{0,sin(2),sin(3),sin(4),sin(5),sin(6),sin(y+x)}");

  x = may_set_ui (1);
  x = may_eval (may_map_c (x, may_sin_c));
  check (x, "sin(1)");

  may_keep (NULL);
}

void test_expand ()
{
  may_t x, y;

  may_mark ();

  x = may_parse_str ("2*(4-(2+x)^2)*(-16*x^4+(2*x*(4-(2+x)^2)-4*x^2)^2)*(2*x*(4-(2+x)^2)-4*x^2)");
  x = may_expand (x);
  check (x, "12288*x^7+9728*x^8+2816*x^9+352*x^10+16*x^11");

  x = may_parse_str ("(1+2*x+2*y+2*x*y+x^2+y^2)*(2*x*y+x^2+y^2)");
  x = may_expand (x);
  check (x, "2*x*y+6*x*y^2+4*x*y^3+6*x^2*y+6*x^2*y^2+4*x^3*y+x^2+2*x^3+x^4+y^2+2*y^3+y^4");

  x = may_parse_str ("(1+x)^3");
  x = may_expand (x);
  check (x, "1+3*x+3*x^2+x^3");

  x = may_parse_str ("(1+x+y+z+t)^4");
  x = may_expand (x);
  check (x,
        "1+24*x*y*z*t+4*x+4*y+4*z+4*t+4*t^3+6*t^2+t^4+y^4+4*x^3+6*x^2+x^4"
        "+4*y^3+6*y^2+4*z^3+6*z^2+z^4+12*x*y^2*t+12*y*z^2*t+12*x*y^2*z"
        "+12*x*z^2*t+24*x*z*t+24*y*z*t+24*x*y*t+24*x*y*z+12*y*z*t^2"
        "+12*x*y*t^2+12*x*z*t^2+12*x*y*z^2+12*x^2*y*t+12*x^2*z*t"
        "+12*x^2*y*z+12*y^2*z*t+12*y*z^2+4*y*z^3+12*x*z^2+4*x*z^3"
        "+12*x*y^2+4*x*y^3+12*z*t^2+4*z*t^3+12*y*t^2+4*y*t^3+4*x*t^3"
        "+12*y*t+12*x*t+12*z*t+12*x*z+12*x*y+12*y*z+4*x^3*t+12*x^2*t"
        "+4*y^3*t+12*y^2*t+12*x*t^2+4*z^3*t+12*z^2*t+4*y^3*z+12*y^2*z"
        "+12*x^2*y+4*x^3*y+4*x^3*z+12*x^2*z+6*x^2*z^2+6*y^2*z^2+6*x^2*y^2"
        "+6*y^2*t^2+6*x^2*t^2+6*z^2*t^2");

  x = may_parse_str ("(1-x)^400*(1+x)^400");
  y = may_parse_str ("(1-x^2)^400");
  x = may_expand (x);
  y = may_expand (y);
  if (may_identical (x, y) != 0)
    fail ("expand 5.1", x);

  x = may_parse_str ("(y-x)^100*(y+x)^100");
  y = may_parse_str ("(y^2-x^2)^100");
  x = may_expand (x);
  y = may_expand (y);
  if (may_identical (x, y) != 0)
    fail ("expand 5.2", x);

  x = may_parse_str ("(x*y+(1+z)^100)*(x/y+(1+z)^100)");
  y = may_parse_str ("x^2+(x*y+x/y)*(1+z)^100+(1+z)^200");
  x = may_expand (x);
  y = may_expand (y);
  if (may_identical (x, y) != 0)
    fail ("expand 5.3", x);

  x = may_parse_str ("(1*y+(1+z)^100)*(1/y+(1+z)^100)");
  y = may_parse_str ("1+(y+1/y)*(1+z)^100+(1+z)^200");
  x = may_expand (x);
  y = may_expand (y);
  if (may_identical (x, y) != 0)
    fail ("expand 5.4", x);

  x = may_parse_str ("1+x");
  x = may_expand (x);
  check (x, "1+x");
  x = may_pow (x, may_set_ui (2));
  x = may_expand (x);
  check (x, "1+x^2+2*x");

  // Regression (Note string length is bigger that what ISO C requested, but it should be ok for most compilers).
  x = may_parse_str ("-53007993351613893450000*(933843470*b^40*(198*c^10*a^2*x^7+44*c^9*a^3*x^7+133*c^12*x^7+288*c^11*a*x^7)-681030*b^8*(16082*c^42*a^2*x^7-13244*c^41*a^3*x^7+1239*c^44*x^7-7392*c^43*a*x^7)+2160865067312*b^27*(80*c^24*a*x^7+120*c^23*a^2*x^7+460*c^22*a^3*x^7+3*c^25*x^7)+605360*b^45*(285*c^7*x^7+84*c^5*a^2*x^7+7*c^4*a^3*x^7+287*c^6*a*x^7)+20277743920*b^15*(1554*c^34*a^3*x^7+518*c^36*a*x^7-77*c^37*x^7-1332*c^35*a^2*x^7)+20*b^49*(2369*c^3*x^7+72*c*a^2*x^7+a^3*x^7+873*c^2*a*x^7)+868741239520*(19*c^21*x^7+266*c^18*a^3*x^7+126*c^20*a*x^7+252*c^19*a^2*x^7)*b^31+1224135382960*b^30*(308*c^19*a^3*x^7+14*c^22*x^7+110*c^21*a*x^7+231*c^20*a^2*x^7)-3329480*b^9*(14448*c^41*a^2*x^7-12341*c^40*a^3*x^7-6493*c^42*a*x^7+1071*c^43*x^7)+933843470*b^12*(848*c^39*a*x^7-133*c^40*x^7-2028*c^38*a^2*x^7+1976*c^37*a^3*x^7)+16*c^51*a*x^7+11760*b^47*(2*c^2*a^3*x^7+371*c^5*x^7+246*c^4*a*x^7+44*c^3*a^2*x^7)+6482595201936*b^25*(195*c^24*a^3*x^7+27*c^26*a*x^7-c^27*x^7)+868741239520*(899*c^28*a^3*x^7+155*c^30*a*x^7-372*c^29*a^2*x^7-19*c^31*x^7)*b^21+9988440*b^43*(504*c^8*a*x^7+216*c^7*a^2*x^7+28*c^6*a^3*x^7+357*c^9*x^7)+319339665120*(49*c^23*x^7+1012*c^21*a^2*x^7+1771*c^20*a^3*x^7+483*c^22*a*x^7)*b^29+1224135382960*b^22*(126*c^29*a*x^7-14*c^30*x^7-261*c^28*a^2*x^7+812*c^27*a^3*x^7)+245*b^4*(17296*c^45*a^3*x^7+11808*c^47*a*x^7-2101*c^48*x^7-23688*c^46*a^2*x^7)+4*b^51*(4*a*x^7+35*c*x^7)+6*b^50*(5*a^2*x^7+532*c^2*x^7+126*c*a*x^7)+273320040*b^11*(984*c^40*a*x^7-157*c^41*x^7-2296*c^39*a^2*x^7+2132*c^38*a^3*x^7)-30*c^50*a^2*x^7-6*b^2*(5635*c^48*a^2*x^7-3920*c^47*a^3*x^7+532*c^50*x^7-2910*c^49*a*x^7)+1969475878230*b^34*(27*c^16*a^2*x^7+4*c^18*x^7+16*c^15*a^3*x^7+18*c^17*a*x^7)+92120*(4*c^3*a^3*x^7+326*c^6*x^7+270*c^5*a*x^7+63*c^4*a^2*x^7)*b^46+140676848445*b^16*(476*c^33*a^3*x^7-21*c^36*x^7+144*c^35*a*x^7-378*c^34*a^2*x^7)-3*c^52*x^7-4*b*(35*c^51*x^7+360*c^49*a^2*x^7-245*c^48*a^3*x^7-189*c^50*a*x^7)-69104416780*(3168*c^31*a^2*x^7-5456*c^30*a^3*x^7-1188*c^32*a*x^7+161*c^33*x^7)*b^19+385868762020*(432*c^23*a*x^7+828*c^22*a^2*x^7+31*c^24*x^7+2024*c^21*a^3*x^7)*b^28-319339665120*(812*c^27*a^2*x^7-3654*c^26*a^3*x^7-522*c^28*a*x^7+49*c^29*x^7)*b^23+3*b^52*x^7+20277743920*b^37*(91*c^12*a^3*x^7+77*c^15*x^7+243*c^14*a*x^7+252*c^13*a^2*x^7)-1816080*b^7*(95*c^45*x^7+1188*c^43*a^2*x^7-946*c^42*a^3*x^7-558*c^44*a*x^7)+273320040*b^41*(33*c^8*a^3*x^7+297*c^10*a*x^7+157*c^11*x^7+176*c^9*a^2*x^7)-656491959410*b^18*(231*c^32*a^2*x^7+12*c^34*x^7-352*c^31*a^3*x^7-86*c^33*a*x^7)+1600874520*(1302*c^13*a*x^7+1183*c^12*a^2*x^7+466*c^14*x^7+364*c^11*a^3*x^7)*b^38+29171678408712*b^26*(40*c^23*a^3*x^7+6*c^25*a*x^7+5*c^24*a^2*x^7)+570111438435*b^32*(228*c^17*a^3*x^7+144*c^19*a*x^7+25*c^20*x^7+266*c^18*a^2*x^7)-385868762020*b^24*(378*c^26*a^2*x^7+31*c^28*x^7-3276*c^25*a^3*x^7-448*c^27*a*x^7)+7470747760*b^39*(43*c^13*x^7+84*c^11*a^2*x^7+22*c^10*a^3*x^7+106*c^12*a*x^7)-71583820*(184*c^42*x^7-2296*c^39*a^3*x^7-1134*c^41*a*x^7+2583*c^40*a^2*x^7)*b^10+245*(1072*c^3*a*x^7+138*c^2*a^2*x^7+2101*c^4*x^7+4*c*a^3*x^7)*b^48-570111438435*b^20*(25*c^32*x^7-992*c^29*a^3*x^7-192*c^31*a*x^7+496*c^30*a^2*x^7)+11760*b^5*(2115*c^46*a*x^7-371*c^47*x^7-4324*c^45*a^2*x^7+3243*c^44*a^3*x^7)-1688122181340*b^17*(56*c^33*a^2*x^7-77*c^32*a^3*x^7+3*c^35*x^7-21*c^34*a*x^7)+681030*b^44*(532*c^6*a^2*x^7+1239*c^8*x^7+56*c^5*a^3*x^7+1488*c^7*a*x^7)+1688122181340*b^35*(3*c^17*x^7+16*c^15*a^2*x^7+8*c^14*a^3*x^7+12*c^16*a*x^7)-1600874520*b^14*(466*c^38*x^7-8436*c^35*a^3*x^7-3078*c^37*a*x^7+7733*c^36*a^2*x^7)+46892282815*b^36*(112*c^13*a^3*x^7+224*c^15*a*x^7+63*c^16*x^7+264*c^14*a^2*x^7)+71583820*(24*c^7*a^3*x^7+302*c^9*a*x^7+184*c^10*x^7+153*c^8*a^2*x^7)*b^42+20*c^49*a^3*x^7+69104416780*b^33*(817*c^18*a*x^7+161*c^19*x^7+1368*c^17*a^2*x^7+969*c^16*a^3*x^7)-20*(25872*c^47*a^2*x^7+2369*c^49*x^7-18424*c^46*a^3*x^7-13132*c^48*a*x^7)*b^3+92120*(1886*c^45*a*x^7-3933*c^44*a^2*x^7-326*c^46*x^7+3036*c^43*a^3*x^7)*b^6-7470747760*b^13*(684*c^37*a^2*x^7-703*c^36*a^3*x^7+43*c^39*x^7-279*c^38*a*x^7))*(64*b*c^63*x^8+151473214816*b^54*c^10*x^8+41664*b^3*c^61*x^8+601557853127198688*b^38*c^26*x^8+621216192*b^57*c^7*x^8+41107996877935680*b^21*c^43*x^8+64*b^63*c*x^8+151473214816*b^10*c^54*x^8+146721427591999680*b^41*c^23*x^8+1118770292985239888*b^28*c^36*x^8+13136858812224*b^13*c^51*x^8+488526937079580*b^48*c^16*x^8+1777090076065542336*b^31*c^33*x^8+19619725782651120*b^20*c^44*x^8+13136858812224*b^51*c^13*x^8+743595781824*b^53*c^11*x^8+74974368*b^6*c^58*x^8+846636978475316672*b^37*c^27*x^8+635376*b^60*c^4*x^8+250649105469666120*b^24*c^40*x^8+27540584512*b^9*c^55*x^8+19619725782651120*b^44*c^20*x^8+2016*b^2*c^62*x^8+846636978475316672*b^27*c^37*x^8+488526937079580*b^16*c^48*x^8+1379370175283520*b^47*c^17*x^8+c^64*x^8+1620288010530347424*b^34*c^30*x^8+8719878125622720*b^19*c^45*x^8+b^64*x^8+4426165368*b^56*c^8*x^8+7624512*b^5*c^59*x^8+250649105469666120*b^40*c^24*x^8+7624512*b^59*c^5*x^8+146721427591999680*b^23*c^41*x^8+3284214703056*b^12*c^52*x^8+41107996877935680*b^43*c^21*x^8+1620288010530347424*b^30*c^34*x^8+159518999862720*b^15*c^49*x^8+47855699958816*b^50*c^14*x^8+1777090076065542336*b^33*c^31*x^8+1118770292985239888*b^36*c^28*x^8+27540584512*b^55*c^9*x^8+4426165368*b^8*c^56*x^8+401038568751465792*b^39*c^25*x^8+2016*b^62*c^2*x^8+601557853127198688*b^26*c^38*x^8+743595781824*b^11*c^53*x^8+3601688791018080*b^46*c^18*x^8+1388818294740297792*b^29*c^35*x^8+3601688791018080*b^18*c^46*x^8+159518999862720*b^49*c^15*x^8+635376*b^4*c^60*x^8+1388818294740297792*b^35*c^29*x^8+74974368*b^58*c^6*x^8+80347448443237920*b^22*c^42*x^8+621216192*b^7*c^57*x^8+80347448443237920*b^42*c^22*x^8+41664*b^61*c^3*x^8+401038568751465792*b^25*c^39*x^8+47855699958816*b^14*c^50*x^8+8719878125622720*b^45*c^19*x^8+1832624140942590534*b^32*c^32*x^8+1379370175283520*b^17*c^47*x^8+3284214703056*b^52*c^12*x^8)^2");
  // Can't be checked: too big result (It multiplies by 10 the size of this file).
  // But the test was to check it doesn't crash
  x = may_expand (x);

  x = may_parse_str ("(1+sqrt(5))^10");
  x = may_expand (x);
  check (x, "62976+28160*5^(1/2)");

  x = may_parse_str ("x^((y+1)^2)");
  x = may_expand (x);
  check (x, "x^((y+1)^2)");

  // Bug found
  x = may_parse_str ("(18 + -1 * a) * (16 + -1 * a) * (14 + -1 * a) * (12 + -1 * a) * (10 + -1 * a) * (8 + -1 * a) * (31 + -1 * a) * (6 + -1 * a) * (29 + -1 * a) * (4 + -1 * a) * (27 + -1 * a) * (2 + -1 * a) * (25 + -1 * a) * (23 + -1 * a) * (21 + -1 * a) * (19 + -1 * a) * (17 + -1 * a) * (15 + -1 * a) * (13 + -1 * a) * (11 + -1 * a) * (9 + -1 * a) * (32 + -1 * a) * a * (7 + -1 * a) * x ^ 33 * (30 + -1 * a) * (5 + -1 * a) * (28 + -1 * a) * (3 + -1 * a) * (26 + -1 * a) * (1 + -1 * a) * (24 + -1 * a) * (22 + -1 * a) * (20 + -1 * a)");
  x = may_expand (x);
  check (x, "a^33*x^33-528*a^32*x^33+133672*a^31*x^33-21605760*a^30*x^33+2504646364*a^29*x^33-221783846592*a^28*x^33+15600262127208*a^27*x^33-894988986503040*a^26*x^33+42669229615802790*a^25*x^33-1713839619772257120*a^24*x^33+58588605091754279880*a^23*x^33-1717750737160208150400*a^22*x^33+43438311479828650125180*a^21*x^33-951323872658115219215040*a^20*x^33+18093853291291024209999960*a^19*x^33-299355217555964945750812800*a^18*x^33+4310704065427058593776844065*a^17*x^33-54005406709047719373010498320*a^16*x^33+587826383619146473939963593680*a^15*x^33-5545451763959981516982710342400*a^14*x^33+45183216816109661755524738933344*a^13*x^33-316453995650145899860617467653632*a^12*x^33+1893448925578239663637174767335168*a^11*x^33-9602115221951625415659706174525440*a^10*x^33+40856661618141858281716601067008256*a^9*x^33-143984983283859317750146955086368768*a^8*x^33+413260829804662494127982011660050432*a^7*x^33-944666565040451665724861862989660160*a^6*x^33+1667775417965472372495519818096640000*a^5*x^33-2175167854639965937281416679260160000*a^4*x^33+1954695838786353608553674747412480000*a^3*x^33-1067915237466585625763819972198400000*a^2*x^33+263130836933693530167218012160000000*a*x^33");

  x = may_parse_str ("(1+3^(1/5)-3^(2/5))^3-10+5*3^(3/5)");
  x = may_expand (x);
  check (x, "0");

  // Another bug
  x = may_parse_str ("y*(1+x)^1000");
  x = may_expand (x);
  check (x, "x^1000*y+1000*x^999*y+499500*x^998*y+166167000*x^997*y+41417124750*x^996*y+8250291250200*x^995*y+1368173298991500*x^994*y+194280608456793000*x^993*y+24115080524699431125*x^992*y+2658017764500203964000*x^991*y+263409560461970212832400*x^990*y+23706860441577319154916000*x^989*y+1953840414726664053684327000*x^988*y+148491871519226468080008852000*x^987*y+10468676942105465999640624066000*x^986*y+688141030994399298376377021938400*x^985*y+42363682220592706806295710413082750*x^984*y+2452109606180189617493822296851378000*x^983*y+133912430159729244110912628766939143000*x^982*y+6921158232466006195627168497322854654000*x^981*y+339482811302457603895512614793686020778700*x^980*y+15842531194114688181790588690372014303006000*x^979*y+704992638138103624089681196721554636483767000*x^978*y+29977513047785449754769922191029584107874962000*x^977*y+1220334593653599350433758915859829319724743244750*x^976*y+47641862536236518640933948075167736642053976275040*x^975*y+1786569845108869449035023052818790124077024110314000*x^974*y+64448852930964401605930090868351910401889684572068000*x^973*y+2239597639351012955806070657675228886465666538879363000*x^972*y+75065134670661537691155195836562844056711306061749684000*x^971*y+2429608192173745103270389838576750719302222606198631438800*x^970*y+76023224077694604844267036884498328958811481548795886956000*x^969*y+2302078254102689752940461210658715023784010175649475451886375*x^968*y+67527628787012232752920195512655640697664298485717946588667000*x^967*y+1920565206971789090355112619433470722195334606932036892683558500*x^966*y+53007599712421378893801108296363791932591235151324218238066214600*x^965*y+1420898158957961962014390819610862755970848386695218627770386030250*x^964*y+37020157438796630577888452705537072885294536345248398842450057653000*x^963*y+938168726672661980171225788300847399698385223696689686454721197890500*x^962*y+23141495257925662177556902778087569192560168851185012265883122881299000*x^961*y+555974423571664033815804589243553849851258056649719919687842027223208475*x^960*y+13017937722653596889345668431068577947736774009359295680495813320348296000*x^959*y+297242911333923795640059429176065863139989673213703918037987737481286092000*x^958*y+6622295559486023167980858910480723183444421091598333801869587267606327352000*x^957*y+144034928418821003903583681302955729239916158742263760190663523070437619906000*x^956*y+3059942034853175105151688873902792825630218839057870105383873956785296991780800*x^955*y+63527057462712657074344845099503633662540412854353607622643470189781709285884000*x^954*y+1289464102541018613806914515423967372639650082192624290893656820447909588483688000*x^953*y+25601235202533140394958115274980018877616386006866061442117811455976204954686555500*x^952*y+497395426792072441959186239628183223907975499561969193732574622573251981976767364000*x^951*y+9460461017585217846063722277728044918729694001668654064793569321343252697198115263280*x^950*y+176224273856979548112951689487091032799866849050690614932429232456393922790945284316000*x^949*y+3216092997889876753061368333139411348597569995175103722516833492329189090934751438767000*x^948*y+57525587962256663432116550562569093556047101045773553376338833032605118079361214414172000*x^947*y+1008828366671427042041006914495424659214381568339769537914682868182908274465834630559646000*x^946*y+17351847906748545123105318929321304138487362975444036052132545332746022320812355645625911200*x^945*y+292812433426381698952402256932297007336974250210618108379736702490089126663708501519937251500*x^944*y+4849384862359724979141539132352427630282520915768833233517042932467440974921768867277557288000*x^943*y+78844309055262425091904679341523090609593400406379478262182266988220635161228069686943733148000*x^942*y+1258836256441647532823291659995165277190457342081516415643655855981420988506387146527135536024000*x^941*y+19742748621859838806445290867590842097270339314978449118678002674641952503075171748033908989976400*x^940*y+304232519746692598000960219926809697892362605837372822484546270723990743490010843330358597550456000*x^939*y+4607650581324908863272607201794746876144007852924081940532079809835924324792260998180753598385132000*x^938*y+68602797544170865297614373893388453489255228032425220003477632724223762169129219306246775798178632000*x^937*y+1004387832795126574747885442782890326866127322912225486613414716603088518007407476405519201920209034125*x^936*y+14463184792249822676369550376073620706872233449936047007233171919084474659306667660239476507651010091400*x^935*y+204895117890205821248568630327709626680689973874093999269136602187030057673511125186725917191722642961500*x^934*y+2856299106111227418599449264568370019697976650722444706229456514069941401000886431707492636672670873523000*x^933*y+39190103911790811493430678880033665123209003163588836925177690112165519516673927070339568088464734191131750*x^932*y+529350389069406323360541923423063418765663636934272406003849379486061799848407246805166339977523656030939000*x^931*y+7040360174623104100695207581526743469583326371225822999851196747164621937983816382508712321701064625211488700*x^930*y+92218802287316715685162578180561569390316810214648104082557929223423921159506327263846513509605494386573021000*x^929*y+1189878712846072623215500487913079138383393287352890120731893281230011427183074694834908486811437559515643562625*x^928*y+15126129390700758826629924010730649868764232474842219616975300890156857594875250915161576380287863770281057892000*x^927*y+189485431691616262598458642674963681464114101407820778174812215205073067438504832410199747358470942095277576566000*x^926*y+2339513463285822122215636041560218253810262105381893874531681483731968805974072997491266214052587898403027145334880*x^925*y+28474341493939282408545570242673709010190690098398050446602702269106199283237072666834490105245313237142106703089000*x^924*y+341692097927271388902546842912084508122288281180776605359232427229274391398844872002013881262943758845705280437068000*x^923*y+4043356492139378102013470974459666679447077993972523163417583722213080298219664318690497594944834479674179151838638000*x^922*y+47189552984208944431093927069010287068989948233451472869253318884562785252639626605476440285305536585564470607534484000*x^921*y+543269728730705472762968835381980929881746779037610081407278833658529065221013701295547518784579989941310967869240747050*x^920*y+6170470992990728826443596648782993277669222675488904628329586752664774567942377841875354534343377663530939388144462806000*x^919*y+69154424909249753554898357563799644172902629741150040895547441776816193023646893130285985573921513082743088996399528277000*x^918*y+764864603213147876667430027030940642779814627739466717374850018688159821635034311971114876588674084457327177092708035642000*x^917*y+8349771918410197653619444461754435350346309686155844998008779370679078052849124572351337402759692088659155016595396055758500*x^916*y+89981071497220482949593072081965444481379054970808870802071081218141594075409389509103824246210328861315129355310385730291600*x^915*y+957356749069264440684623964593004438377463201142908334696453945518599518360460365125930223084679661722131899536151197014149000*x^914*y+10057747915509283894089037972850644329620705354535841585201826508091953560706445675001151998843646101310673059494737862884278000*x^913*y+104349134623408820401173768968325434919814818053309356446468950021454018192329373878136951988002828301098232992257905327424384250*x^912*y+1069285514343245440515398621338346029740124877130540821114378454152427692038251561537762923742231229332602117853249546725966724000*x^911*y+10823545595185517736772534933769258145480597367399140978168875241476240304964968584010022483657473888022228104047892634081729839600*x^910*y+108235455951855177367725349337692581454805973673991409781688752414762403049649685840100224836574738880222281040478926340817298396000*x^909*y+1069413363698221263339808071173506049374115544235415124908207347228467656218821352485338091048330843936109276802123304823944828717000*x^908*y+10441154131591235560349954071242403148727923808233945520609164207348910019856879441469752544858972110688034659530408180431633381452000*x^907*y+100746029759077134608908599389541060169108796745406261565877786553887887106491379291628357001990294727596249321213619357994590180606000*x^906*y+960798978544461936375486221546570531718027051066716557670371311766551849668223048823318857303191652875812651421258306719401038985568800*x^905*y+9057532037320188046039739901037982616716817512660192548871729553632598166143144366511495477701962977631358849335820495636020211270205875*x^904*y+84412463523066494779586854335446765830020649808709423342062304293648131362818582549756617647861593111121117523707028124277961556580063000*x^903*y+777800556748255559040478872090902342290904558951679686509002660991472067557399796351328834041010393666758868611300473430846931485630580500*x^902*y+7086627294817439537924363056828221340872685981559748254859802022366745504411864811200996043484761364519358580680737646814383153535745289000*x^901*y+63850511926305130236698511142022274281262900693853331776286816221524376994750901948920974351797699894319420811933446197797592213357065053890*x^900*y+568964957759154625871570891364554919337986243806613847511466679201702369260156552020087890263543860444430482482575263148691415762587708401000*x^899*y+5014700951230196163319041483693479142008329737079861263851064162767945391812556277118225621048293436662186311292501584026211595789866175024500*x^898*y+43720402468006952957868924780162565723528933047550635096487918622967135551919179969438510754382208797307216578064722548112019543876697331767000*x^897*y+377088471286559969261619476228902129365437047535124227707208298123091544135302927236407155256546550876774742985808231977466168565936514486490375*x^896*y+3217821621645311737699152863819964837251729472299726743101510810650381176621251645750674391522530567481811140145563579541044638429324923618051200*x^895*y+27169342937476924577742847293574231408870734695360900330904265806906520312037926631574090381251555263171895947455466072539952371643828364510904000*x^894*y+227003669029012809088804724116405260556359222594884531736714146087611487466933704753525577577933555189492289504908286624773060002332547269838768000*x^893*y+1876984041138041097373172394777313867378044312752147100378571596817009799147887021712021673861987636890894578961880555147429097982249673258944628000*x^892*y+15360273070597547328962108037994164859644179146558855170070512517071309548990047920799296633806357542263100591137591332032172067891437693091546864000*x^891*y+124418211871840133364593075107752735363117851087126726877571151388277607346819388158474302733831496092331114788214489789460593749920645314041529598400*x^890*y+997587464557997465716106638251350761019593580788673756045390312933036671519542841991370535433423707406979208662260323537116472409273642608080732816000*x^889*y+7918350499929104884121596441120096665593024047510097938610285608905978580186371308306503625002800677542897468756691318075861999748609538201640816727000*x^888*y+62225621627761461390265288847032264062359339417601477606070209032818663532792015236957302823030858421753034975716299915498809343157214778080150842952000*x^887*y+484159003366880844326011501818575598450111702310636058215651538702720653978829101010360768456389222983288965118073316009188104275267100948746436821916000*x^886*y+3730129365070055896285619048793547654146077984758465631122324028614004342828196378218953398716181317940817592127069199862092699025101316874689939341022400*x^885*y+28458314552474133346661834984330083395855853590614155892614282459684429684508222368308394464343279882565720422693588292051310677907022977880177554455214000*x^884*y+215018376618693451952556086548271741213133116017973622299752356362060135394062124560552313730593670223829887638129333762165458455297506943983563744772728000*x^883*y+1608993445375477271814466308662067351620309673253141597378655344641517792821668271075997398509442464471540599868374590779594066237522869758792260903680668000*x^882*y+11925480830430008014624867934789440370832883460582108309982974907343014229148835420916216012481750030789065622553835202248756020348698917035754404344927304000*x^881*y+87552905096740308840704238754579141389198086073106978509125007444743296132334366715226552558303514809376390112249406776509617116060031215904163585232341290200*x^880*y+636748400703565882477849009124211937375986080531687116430000054143587608235159030656193109514934653159101018998177503829160851753163863388393916983507936656000*x^879*y+4587720034577331235229748188690018794700752170388139142147295472067323833104137606121260190685471804318440948355721523490429415500254392773756172364782592792000*x^878*y+32748115368771518898631861054226313022335450452038911925246548166464311589149860310361515832697920684484480915905069086378837616335149242726487149075439971312000*x^877*y+231613686922682436081452759230294165488614435858371981922913086628945171481326028162798785365129648712039433574586657973824520883273595853799429272090006893876000*x^876*y+1623148717954158512058820936685901511744209966495470849315774911095647761741132805364893887838828578173972350490703299080562242349981359743426400338806768312283008*x^875*y+11271866096903878555964034282540982720445902545107436453581770215941998345424533370589540887769642903985919100629884021392793349652648331551572224575047002168632000*x^874*y+77571739910976298093799731991659991320234006491526767404964308415222886251189308392876053038666676362863726724019831769270089666113501116347040348650323463743184000*x^873*y+529063507361580533092868484599368534551283497399241155816670634738199841385064579898287455490281316131094011172416508863849908425914738082585673627904159248810934625*x^872*y+3576305259064327324472723399772475675416427982419676650171603050323335361920746617607028381298645795862899052266257331234706357731764741147400832585522688875683217000*x^871*y+23961245235730993073967246778475587025290067482211833556149740437166346924869002337967090154700926832281423650183924119272532596802823765687585578323002015467077553900*x^870*y+159131934008289801330927516773082142839712661904765612166795986109425357439969710183445560569387834687670523478320717433336666864263028062199995825503906514934026503000*x^869*y+1047618565554574525428606152089457440361441690873040280098073575220383603146467258707683273748469911693830946232278056436133056856398268076149972517900717889982341144750*x^868*y+6837089585724591639639324361004880137095724719381947091166374911964608778429575793671196102358435213159738806989604157793710476325967644286452452222088895703042647471000*x^867*y+44236990080770305608711150902919634916880547251523493492845127228905341872376434426215873289140024849324578698955125409008559574437417519375778179675754272944313248935500*x^866*y+283772099332941367830695234680951139540878177183847002702250964298015007862799942319281083469594529774185815950334360031121574751576322753921658545179282965702039063541800*x^865*y+1804874014139663846864348367639873056638673700470791598069463853807227807362661397839545126479406384225520079389994275197942368824364111633398784129265292392148998455615125*x^864*y+11382563125669120902852532771101097233108131950414335333810341384594487777819995968856693352395672379349265318196752217306731435505478777016471164143687683407421420917164000*x^863*y+71182260706177183617114027401885847189654477342083850674480613151485818494627945805241495384909168575205912823215921474896443687255276699747931990260887469424671639503714000*x^862*y+441432436897300232215484112377162591924332082509901289794261068608494788074599203482864525336630959077895660817353412311947729916647831044479981119459604306791848584548212000*x^861*y+2714809486918396428125227291119549940334642307435892932234705571942242946658785101419616830820280398329058314026723485718478538987384160923551883884676566486769868794971503800*x^860*y+16558412473402985306295712555764630841757392797126722849091111999080347050542944590218939535499582571368724468531788636297103145596811194285493759863984731763277213926776548000*x^859*y+100166734610233551958507162573252238683588735300928555826544121177535338848002742274634289161930573442293903651188777736473321141321555041487599575515231581582078357486627146000*x^858*y+601000407661401311751042975439513432101532411805571334959264727065212033088016453647805734971583440653763421907132666418839926847929330248925597453091389489492470144919762876000*x^857*y+3576787148373756417851693263553215356326481089703990514306179660381157724697431255390066075490604226668578142877865938339901509088023861273119701509023061058993381348584977671750*x^856*y+21115377924192658577110685748976223069072191812321488829283377857146696636834490721475148693930739434677950967610022367027280632961023622412348030977405105286195409892336144048400*x^855*y+123655124145100843037189289831333361123676191777636116089296493615482367291051298403159261187060152168833206008949103587728253021792295870976421688258091541230801886698269884667000*x^854*y+718377387890585850025575874258222383670880733184362198233055820051849943309917066913591898324825645933221482528180506557278422317079052202815402188927960382388468103675663139494000*x^853*y+4140377782909930608593352842853133062643657198690952399275652800704243254347022014035769522101866729601607598625256568198368204300462375195956338291591555447144346570509058499921500*x^852*y+23675180342545375023634473973898452143438898881105311705925209303355807065125253395694467334434835259198454188112205342986642349422778145415804028351919498261523377705192737194182000*x^851*y+134317189810040760967419582345250551827110019652137468411615687447705278749477270931573278010693632037185896760556578312544217595725228011658994854183223286803709296180793462348325880*x^850*y+756090141314798985578189701943463371212208719896138067217704200864566138655997882727399247080063491600053061234921136196440959975936713972914871695733376117769224514924996311232298000*x^849*y+4223161381422791702341335900986844751047139494683034335972571490355372708677251331812907636651144107687138480187158188360384046181383356335557408353142344236750471139285012291027769750*x^848*y+23406802950630897801212110091744080711686106480334726254279350482492523248093523721420559973073007864828061641821634926337291968377863308317337792702383711848133330236037192305827116000*x^847*y+128737416228469937906666605504592443914273585641840994398536427653708877864514380467813079851901543256554339030018992094855105826078248195745357859863110415164733316298204557682049138000*x^846*y+702657123414745596574451279076678758396615828729016008136527856742178778537930102424321713256185197387386908512232692337080125992659341765164985480285105878899125068311490682574281101600*x^845*y+3806059418496538648111611094998676607981669072282170044072859224020135050413788054798409280137669819181679087774593750159184015793571434561310338018210990177370260786687241197277355967000*x^844*y+20460599676503685471377068561648936669659418452268481001257918376261108169103421135349410397682760047065841720265968949900326811017670641845515447690255259297455414674930137391733047364000*x^843*y+109166364096788650964372587325759832990651200982672971418103956906253887256672050741136411172446624808079142849267163447885920896758837665036515964575222680935157687158013327982474423594000*x^842*y+578101123078591472402526531624464021246090007719563785748701457327457692264892243547401623944654453386180114962785859264905316950131706377111612843851179228600017437654385045039267073372000*x^841*y+3038644028181846426815780081851089011674760603075957148841612035077449494967339855146029785859089970611109229273143172761158572219129781644692915010492760820328841656670861392987647554411575*x^840*y+15853794929644416139908417818353507886998750972570211211347541052577997365046990548587981491438730281449265544033790466579957768099807556407093469619962230366933086904369711615587726370843000*x^839*y+82106999666491760131994830553077735291308346086335846952596215698228023390582870804106891798253671025530455502742902478151756589109497159416984080315730316529980616745470296577025323611958500*x^838*y+422120648592147822028292441739135841558996282333432145682672569049785788965082489164672241269549547971745531971156762433688171912108948586450507112298049111976219367071804346819308105440621000*x^837*y+2154359651656266628278541303266199386493170050689528694734127684723601861974231972139211377698859583246042745486940305835347560307531646139384600323130896992220095184384757550535127343010974250*x^836*y+10915422235058417583277942603215410224898728256826945386652913602599582767336108658838670980340888555113283243800497549565760972224827007106215308303863211427248482267549438256044645204588936200*x^835*y+54905888953456498084560735383643780348135169243677707215995077458859347052564161024881266678220734599515611497430213577635002480769461150202950496588709527359954715020504704480706498468866034500*x^834*y+274200666989118080254632654550652172517034318258845555797244877848435302046937187393718421614587381173628862208723342058368814784201979636342878527874154166576061271419766009203049219898408819000*x^833*y+1359578307154377147929220245480317022063628494700109214161339185998491705982730220827187173838995764985909775118253237706078706638334815696866772700709347742606303804123006462298452381996277060875*x^832*y+6693308589067702882113084285441560724005555666215922285101977531069497629453441087149229163515056073776786585197554401014541324988725246507651804065030635040523341804913262583623150188289363992000*x^831*y+32718467279501535853152782595305511539109510344855478934822019578345603118092997314241231969888303513579468542936280630841669653327239293222697936341414457168675864940487771805828457685108596925600*x^830*y+158808934748457747123490114351482892265853178866842383133931440058636553146299343689007149327528022902169350237643935225722724048313500663010756065282888885672520280120496202332383741980351669288000*x^829*y+765422133177159723054495958124298358653443519073327532662960254701219200920245092547598411584422854569176693877946641291419408349138907265325097547206481896642554140813321812404337919195997289766000*x^828*y+3663407666304556362364870828479300814826885744466561832629659484928378603248340674158447888970532506261724292086357335198238555566976966564677345486051832430173611725973586477865848538117258704776000*x^827*y+17411713448470506388941081466392998700355370750999118595314531000205569568312515732925496575739255072864632123881709863269788996861436502005679107568763594366399867226322735731006073224269959476148000*x^826*y+82183287476780790155801904521374953865677349944715839769884586320970288362435074259408343837489283943921063624721670554633404065185980289466805387724564165409407373308243312650348665618554208727418560*x^825*y+385234160047409953855321427443945096245362577865855498921333998379548226698914410590976611738231018487129985740882830724844081555559282606875650254958894525356597062382390528048509370086972853409774500*x^824*y+1793406485192462158060931391038478866136603187352909215317396693021173665536189120491326147301143272505057108759816115916788266676727959706584948079582650219739186324311241780293625542099805826043244000*x^823*y+8291986164682002000472733341711618577699013613435080248349536395260819813125189023395288872072139962200348317468138558424251367836781521564715799266834388375535676094989617894278954051394046038390954000*x^822*y+38078282834461484046863613446295812686416699386835955106945915736896055231222935068329203647169268429769197301445865335333712985261644752660315011158312107512236456704365731335739107431541373427694772000*x^821*y+173679278928293768902639036885604790086378389981068439682236648999953674137966831617212645524033163226891727691594752446160990893887835232967325689783190223708589616412690363481343373340530375467430043400*x^820*y+786834302327076743094828785890585236855415910411470279223392553480453109354324872519968891324349137271001197276838105004707251563469750779188989312829922560447754063306111038976251746625607225874544948000*x^819*y+3540754360471845343926729536507633565849371596851616256505266490662038992094461926339860010959571117719505387745771472521182632035613878506350451907734651522014893284877499675393132859815232516435452266000*x^818*y+15826978507464314160284506889963083370845824952047115288641027264270753527504206862000029994343875269369155230470169751488127830629137446001063768636759261994580233371747512210227227755895410920460109036000*x^817*y+70275225220643177548654576788586082141201298835991810819672387363636987130276831555728394050972533125405433822250699385683698030565246159689505972696914766573761144916944116716063288459600819141390810230500*x^816*y+309970723135377475025416944105331043390379782973888203399203611290420440531383213780942538084289659623409913507873355128204851853736437115171010128219905132563184293255277833731392666935320369834458925124800*x^815*y+1358205050297487323364058115300240862167524317869456375109413673127379887274609243179936389992989637597199352198477335642402979896748366929378350830640982166876318274209953948876801201894011297930559268692000*x^814*y+5912186689530238936996488266601048458846870560137633632829212459495653626960063764430311344675366657776044238981607225737518853668198773692588115380437216491108679546560976012757840525891578590991846228424000*x^813*y+25567062652064277956266728514609853175758009390382426295160370901967906376162403406818314485218473897722999820702375928322355468256625547936564562788805622379102959954011029246660235891222624438704100977174000*x^812*y+109843676579239120108405203988694184014367744047568942601429741652899153319808844266330536306864554523550665896350948432792342011769206057801536640129683414665775679802417755281947680125252756847765767161192000*x^811*y+468859061609278560041666423341215701240274949592517960261892213055269543907184066842074026025616598518945210747055890415760996692341190067773927448132490785757600401682951576492945097797789398965989669303824800*x^810*y+1988355182740919547820679596368506376987553451151515957131584777878368222852455990272669953302353114137935186937776289197729881260713947407837074518258206997191917933838695167326102247205284885667286032126168000*x^809*y+8378017410611478719723592674281883640536097614487377131872146277622916105664775500680156209487519111133278990795109468546684760103737413817396840027452549274626362544143252033160503739526434752629345833281614125*x^808*y+35074808641316449769620014926527264153125216955988604780065773017198529602990355463987389726766401252827406344883152593708400446444662333494594024570889429087554927127812163952298896484649529948831665457469141000*x^807*y+145903972028568943113831711575811866863773454038571154935634426932367079327903179687823832523198380469235654228457237851147830723097126304794522566127359635431220753567754723244872213727382323034572958887513385500*x^806*y+603069751051418298203837741180022383036930276692760773733955631320450594555333142709671841095886639272840704144289916451411033655468122059817359939993086493115712448080052856078805150073180268542901563401721993400*x^805*y+2476893620389753724765762151275091930330249350702410320693032057208993513352261121843295061643820125584881463449762156854009602513529787031392728324971605239582390411757359944609378294943418960086917135399929615750*x^804*y+10108743506565289313257222180838446253733606487130649227599988700487465912361512395746239743967671984620531454891415097008242235638974359255024129813589698541239806553568108606426092127586339309187215110972301579000*x^803*y+40996570887737006659320956622289254251252959642252077423044398618643611755688355827193083406091114159849933122615183448977871288980284901423153415355113777417250326578359551570505818072989042753925927950054334181500*x^802*y+165222361065151152466208076437567748288969214236613899966239234633930535819407343584969110008467706312561037006720488070755039064131600456991804216657292710998164632742936484218822442686116644666575850331374753837000*x^801*y+661715556065930365627163346132458831897321703017638669364788134708891795956726411057801285583913163781806953211915554723373931451847059830252175887712457307547649354135460619296383882957897161889636280577155889117185*x^800*y+2633693755486290012446421278139139629442076429920949927820052277448325556046672282817119544612589706594256530196678824769647488365560437135332043334178934557403579518947106942473169683414516067222432957521018464148000*x^799*y+10417432230859137227448963372441448336258510235182371249149612721194119402382629475103358990819104829548569146669041490054199718832093016193714369425786973818640891265538309143742884044793061077775861054748978974526000*x^798*y+40951285321308332549282131877873279666671385062441045600105374145383779719711026212475273274254412088570237335181749305730302342995124270554601314294472931562933158767978180771954785555393412512636143456599434589516000*x^797*y+159991050985699710989107152483651979874201440660615261486686192126818002140243568094817611762650815855835682137940461748367896898858402174666751213199484933606165331069012794486509627880630145943975521249557594940413000*x^796*y+621233544315204731450386796960912077950557789101706088504401019185107949773823805870608872990585606932903429179515158788784614299957503078218214466862390278782963919663093582493959335575519981323924462998282173524725600*x^795*y+2397478969565960007296395648465655834809191467649788060004848593457091359564028765374437155473376492775039932998614326393610526060515606539725633500755341124429399592874560184867464426128827112390873534386574407534742000*x^794*y+9196127062006629206731102149187104989557961474946530046588646295675992944414680385059435272685318527842423704352172826843124433294924597065421028983573627308197793607451211530361192049982071146079002832381353041461764000*x^793*y+35060234423900273850662326943775837772689728123233645802619214002264723100580968968039096977112776887399240372842658902339411901936900026311917672999874454112504088128407743959502044690556646244426198298453908470572975250*x^792*y+132859835711622090381457238944834753664929496045938026199399126745424213854833145563095525386953680836460279307614286366759876681024042204971477497683734773478962860276071450793902485143162027873615067236246389993750222000*x^791*y+500438714513776540436822266692210905471234435106366565351070044074431205519871514954326478957525531150667052058680478648128868831857225638725898574608734313437426773706535797990366027372576971657283419923194735643125836200*x^790*y+1873680495098973776990945927425813342759598121962225529039551349852135793178665861677336106049503173502497493489846341857923252972356437225561421203511374917609322991602669575414166642769364017105468728622387872787058818000*x^789*y+6973273163363633537952152531787578903006240180321678973642481202987429909518713985204802772042726433459766614922116810027837012241458627221546987403634315141479980379125029693404610760118057591963277485297471847306553808500*x^788*y+25797836867279545670921578380509916317224963671800389817982512619502792341318059250429035607369335350076507476801070639915190449043518301645910920535510987471766312388500109851656494267479011185291373983166233876420490146000*x^787*y+94873353339014030107548047595613570755402086026667788723141296409106063423445386121904911322428350095842109272160946699127359268211443473809962123651622182898505083410044796510531126114513933658057529554915075050200587593000*x^786*y+346839328951000128672245420512336123784865300544004102029716553383987748143386388334033768834552014768985571571713972583786532022391602653091303391582209468642906956094396325847802163376781171419689387117038367392826334177200*x^785*y+1260504042715440282443114143991591931347774356143718611543182844474214732835918124269520872847793201822470711499053094806816794618413926308688301677740900152243897965435653313845021751160987127613223004105903325941521631153250*x^784*y+4554079122068687472052541423453493429385507351228918854607628341326195163794284836070527024482349632390861925415933761882692935395559991824938380255063897324236018455767521650020723746130018009441321821285844274369368473844000*x^783*y+16357082351283405002830917131027914473435102091799281941090701794763352354362041406620287431971008083312132511929707043826369579884052631187737393301445099104939460783788850697092782996421119731158509110398238838675300527614000*x^782*y+58407481272619281790930489481569995973635843999027572958597848417830783292744823652863309460280037996119121572278679946448497769266343185337034892975936381278824923894625028516559617822837057670164174083705126812073447546092000*x^781*y+207346558517798450357803237659573485706407246196547884003022361883299280689244123967664748583994134886222881581589313809892167080895518307946473870064574153539828479825918851233786643271071554729082817997153200182860738788626600*x^780*y+731811383003994530674599662327906420140260868928992531775373041941056284785567496356463818531744005480786640876197578152560589697278299910399319541404379365434688752326772416119246976250840781396762887048776000645390842783388000*x^779*y+2567932735856359186466275391682158113915600076106690010148718917441814620936743602079663579442471082295192762353864474688489636820629710045950765417810862728259560982263764469175195470718040399585938238788272542805222822199366000*x^778*y+8958976091911423529465301590711744451239178740856523891908983487756644731339849876313803878054899112222690444445320902724865190342824728321747513430748211670788961633189276937301803032370562470304304707521417212118669756372676000*x^777*y+31076448318817750367832764892781363565235901257346067250059286473155861411835104258463507202002931295522457479169706881326876129001673276366061687212907859233049210665125304376265629268535388568868056954214915954536635717417719875*x^776*y+107179217312900330157503224696881502784991375003113547493537805791861993135929070686967473727796776379224120017047522399598470560467993166489172752343184438954871944338387716426587236943926495686407165317647887914313019185405113880*x^775*y+367539351404857326867544243982668870169771308085898227024299997737579843718340839745131823624081865902206606253149689644640772939657941168270393287902513009690379455142701239958429684210367407774183863368040323599967211808358244500*x^774*y+1253195850164579607909600197544430420755079261931652985536599992285844929682800924945956085837177815895629573744219646629744309494692715701503455527914295460353981049693615681620372579642398121661754670691027358882707585637309609000*x^773*y+4248773649900087881202284880271248751068755567864770867630665763319991801073706644663263396281308998628603774141586784406983996664023987882728820715253291187954505927250723341633982473963042754581299826509491879019004226744036525250*x^772*y+14323376671278898883354427631307441204476328813937131483890279341847308604493019780262180532441792781402978662171637543939701508404482614172343448000766553699130474130295014933368709475543532779636521685874793583417778441250638417000*x^771*y+48014449624156656691592450885817552907179345719763166843823501619844673626365731524270179089185314062876941515366663245119608099912417806638594775689526143052302589367206332667944673937582885969998948781780286316587422513931487910900*x^770*y+160048165413855522305308169619391843023931152399210556146078338732815578754552438414233930297284380209589805051222210817065360333041392688795315918965087143507675297890687775559815579791942953233329829272600954388624741713104959703000*x^769*y+530504479324374554537853372574622100368116621530141886535923458989375776130391487674766777580222794746442069329266724647945095241848409386567232507259275919643975448611805600885768020948293668260476890994095404848501837833524629360375*x^768*y+1748615622837423424399448026340385292200487404871883986521842130917770798575710998000948005071292301996856262853548688968334047835792182012376114015343879426122631521604578117941072275057036640446550438984829488942701336721660580896000*x^767*y+5731573430411554557753746308560151791101597604857841955821593651341582061998163826780885127733680323211917750464409591618428267906207707707232818161404938118957514431926117164362403568242508988130359772228052213756632159254331904048000*x^766*y+18682490415724471452082424137689686263760952192855774204933364837990007912725929750273012799336166500341825518535054243317940651983638740866980164730366734464346621510023003182560004822441539935778108874581651045691830782931141440428800*x^765*y+60559767661140765512046840954799194880411561133621471469381458055348966327268373978639215218187149884582612379997103797195866943930015410013728076350553185869598158708337277265501710547321093435890903767182046821840044698908149160712000*x^764*y+195222204612284999372167875482981370838119969223994954441381577866188229004358808943798989142172921990806396026657330384209461371993804950424001056252416177233641321743331982408621547924697533270129326911928623510066641982978168602464000*x^763*y+625859420668796027399008777283675571216325783688689706885605646688662263572797358084532053426377896970526387261930853290553861457274257046947532797985687156425497178530093708309992609523295033130708724511771175370507764004253540519664000*x^762*y+1995417901881265995305626310837492825384268816614148772580884948856739099759295342512190061551882667328623879052683306307121516445368133346334811682280726415046982636150340609758219114881802574249372586100291362478355297787620074794912000*x^761*y+6327137597215180926781590093947216833822619039347363399725222691999910228820098981882402653504094624321178216496216650415497808395521456152336632042565136674378140775460038350108353110104382329182385575093007195191784923401578653828866800*x^760*y+19952799061757417030514557972613629849399130580514506986685349568132496987150519610915460650054406284166371139158193586372524209048117455086206806441284248433723597466181033801171569973773155892857315506517367088571603907822405713319248000*x^759*y+62579233420966444322977477277742748164024545911613681003694960009142831459699356961507581129716092436703618572814334429986553201105459290952194074747664233723951282962113242376401742190470352573052489543168105868701848619988454282683096000*x^758*y+195206003839887097929287768627691370816175332514416338274900327929754182084165072332603895046604107271692769046062820979135009573818675483710959294891890901904341862079349126425154405680561840536517642278688988676855972238482503482608176000*x^757*y+605618626667190709559306724799845769294445601284480197024998148536163589498823605556480117009341431166686172819137522463955746915494825168726213878004759888285191760631423314360007725820431611828458422971178542739262176166111701378419628000*x^756*y+1868766048001617046640146465096666945251432141106396036534280001197304790453512840002852932485967844742917333270481497888777733339241174806355174252128973369565734575662677655739452411103046116499243133739636646166866143598287535681980566400*x^755*y+5735440513175694594363051142877981884816387262338735803184477239447012669887813797569731561084982612930498319590298906122061742565557264141455920977062499569195648799289925325541815326759348853483449455176527105105625765921573534308517592000*x^754*y+17508186829694225603845103488785418385228971642928772451826298941469828150183852645212864765417315344735205396644070345004188477305385332642339127193138156579649875282042929941127646786949591236949477284223082741901383917023750788941790544000*x^753*y+53159938236934483385868398899417016306763772770666797000908077027930566923743713878408416001448542155587135740616874878178040013753851433385811946679165451227727242287815831635762572703923557263802243528306376228434443909350340097069226934000*x^752*y+160547283350099323317963999889002394629262478407796913030854915361460989263675794524349914992326520887556329626280682363011590724268659750627030457440692447081328860242720905181098211539560301455338502543318855115593179999323115473879753632000*x^751*y+482284039183698367247163855666563193466304485137021926744688165745828811748082086751147144636948868746219214197347169818486818535703053890883599494151840111032311896169133599164019027464839145571836861640129840767241912717966638883534779910528*x^750*y+1441087766485154483806266500995706753385371967540902171547873005216619955422556036109005412261799408604240679872551304238506429887558925968775695699656892762048740725604980873996072791229599040553297395339033388746738783021812665986657708896000*x^749*y+4283233083719764715757514322403906183673188903524348120989511432171620423061485996212877197555903797795937576287860820931116333276911252184972206662869097931644868267770359819932771907265752703866745036146571460997251382870387646127010412552000*x^748*y+12663471725780173942239607561889809586512036758245898792490729451637834294268741206194593453643541663048858921198892861883300463601302832546874350133699941710950045313408020337192543030177007994040811411215950406426656262399406953766813393632000*x^747*y+37242572358888936751389712002880660476868076607912151173191239765249851251254920004044729566424116623218494543840838456011123804370760692568957242322338017551494818303605477133396967100559940832868055607001239974805953653591956671117360649776000*x^746*y+108952780312671163986418530016270481238210137841186136373335940646574074640926158129479875515891729415376458547863786228173719051217990104535067069695937886640843664527410533103977009635363591613017919540482058906687221276782743830013925665619200*x^745*y+317069614581796942069850800242662142665884971451889342180215921022256584404257764868994168981794290681466646945931721640583674582646104015150878777044819240419642695597347059228370594446663577155071679912730991740163983793762281849063963362837125*x^744*y+917898028205668968482369631830897409118359606070839185144282666305676649014660611138255493083482304540899553804564984049004878947426853646973750233935196555923012317215666194808979464079057203904176380759034466360630365535249563018301901719653000*x^743*y+2643404011460511796831010218799832461143182896552843079698457445988828489216638891766371439383826946798016932080588306776785368441620745192641459007030430391669760277873023188926634658181160862406213375596754296534683572064691571017822918518225500*x^742*y+7572995276076060823353704951156276780572361811745982876973959169589616752890911419655010069586098820556480940555198932928087812292210783524864720398519611392351205120392985352060088480194677065271854535493404200883147530779927203456465658457619000*x^741*y+21583036536816773346558059110795388824631231163476051199375783633330407745739097546016778698320381638585970680582316958845050265032800733045864453135780892468200934593120008253371252168554829636024785426156201972516970462722792529850927126604214150*x^740*y+61193283667603112170317868743251293985544486823648574281755095358867822727382881931235311251942844492542598864486262641936157839556599779517010326898382606997964335628003088534462554041113310079150732625883484519779916254463089931378107562019611000*x^739*y+172602429886865266770476736646040863569913647949146169443576394924440156471511258577033950439640313282400689163570030886987865051268424568943017677778262391494258183317153749721251249757186015833940425231022500229455565313161158241558860642490429500*x^738*y+484336856488618125006128637432616567736107498807870239731404484616870096866826269315023024427583844876090146778382824314057203071620141946311585726997557585257652240639009381347085255972636044431361345325074544370107251715258307156921821878927517000*x^737*y+1352107057697392265642109112832721251596633434171971085916837519555429020419890001837772609860338233612418326422985384543409691908272896266786510154534848258844279171783901189593946339590275624037550422365833103033216077705096107479740086078672651625*x^736*y+3755286016850115877406008705829746570472159273775738563150160054312436826524675627745662795687580905429207125461574501977168049979203213782471213108443955918903356492199816134117526437503557959591083437212276089933762389399814094736183786241143666400*x^735*y+10376448204454267555990287213476931313146755888064540766599126465863312283818182655613015619663052501843861794038561123884280138100429932819986246747016193986443485044236334054798428314154568046238520023876026037974869760183696840718402567245265394000*x^734*y+28525516786776900322460190317198754995691830793405891096193853280687907177238000259250762040571837214806721186607879643936560379646874796591273052855093207438387707949323854667498301058387464217000276020692895550088218741478777082723998068756647188000*x^733*y+78019417181744283344639251875024953029261611834203426020560053935612820749684530559816449909474465218109427723073043951513055068213280693661952043816355675568426081816620841310732293566410489817392545981969747903786060960835610453868248449248591003000*x^732*y+212305625936939834231509042277019574785946096143631627684200592865682471333714038549389001240651704608386992911856759005604298549933537054871928981686142581844192906653406899031435088812685793852532876055025485002124151016102850751790177936245236484000*x^731*y+574797824295937106752715221868523367290839245485165629026483827351162542759055415480012444099690355810114414142841817900358304592597835507819926243009519360474463017643112752562885370081753019652598268134161590876121312565819199627994889153315806925200*x^730*y+1548348382789793682396612959276834162812961805181442469333332819064017181601883591514424664917985091296618163558208586964064805729138080888223417554970292004230103331658569407272717048559703706075264707520066278005788037538922567263602468936976158876000*x^729*y+4149801364168233803187981056297103326068563073445851324058822151094369578631518890492704340901511513070715592771816396679423688884344341792334086020490231143690240179334915801109598266176558829885544013904883517155218674139244674761640440643586837575750*x^728*y+11066136971115290141834616150125608869516168195855603530823525736251652209684050374647211575737364034855241580724843724478463170358251578112890896054640616383173973811559775469625595376470823546361450703746356045747249797704652466031041175049564900202000*x^727*y+29361611598543123843480897595406268788825745541558480901126654052025369184088703001344973779419940340656060690463362728816944251279010574044057231502641343469224375770087433454079590652168936927754652049721170968095805120187161835053163993653407600171000*x^726*y+77514654620153846946789569651872549602499968229714389578974366697346974645994175923550730777668642499332000222823277604076732823376587915476311091166973146758752352033030824318770119321725993489272281411263891355772925517294107244540352943244996064451440*x^725*y+203616393476853402305878398542056516165987235386025117553465274839045494994006440378892318165977412362375725222995928488969678612130529850435962105420491055797447301536041114605464987348736758259863782692631598670055692029124013595259985086422544009881500*x^724*y+532195916524338856568433070557577320231677828229177563569346061312162232403107086044469452534901251084332220438444231862866596805713009428576305286369803337174555401848713960196233396536048422310979706387961290386715960393811501238152451994837263043878000*x^723*y+1384092257723370479492723417313411519883104567660774742664162598304652136789375623058098612168106491129396386248184099413138667232124121643383700439012114434450372501930288464826894768689075573132511970210417312768329638002610487033036772634055184103323000*x^722*y+3581772795972306402128122965233989667941224006634693061661381347584081873698670967196943361954741529015857314950497920345111533123991454575351368161171134844706698732593793088190028756249148974199547105705811110461412181497794880422410572909633845600714000*x^721*y+9223064949628688985479916635477523394948651817084334633778056970029010824774077740532129157033459437215832585997532144888662197794277995531529773015015672225119749236429017202089324047341558608563833797192463609438136367356821817087707225242307152421838550*x^720*y+23632052539973865016176298852469099090259890776870893011815661987262945885542120901007590722648009945891101287965206919287675382248683831966909026942388911039452738257042321656598979765430328107352172007041187896069246208174063018872417089588829714390476000*x^719*y+60253353816458187754009783244415894488995962654504156296083194924971837204626896907178928119091911883318091581726892819034888651903559131858892164438218535593498293641182373301754136352285127337539757705895794671183645474032451455919389671682158030662242000*x^718*y+152868933004300278471303973037069301212364315144643053782995526346748336088063999927047598549498207534354734119010279307657420678681114687896411922496964341187744787400596975373355017317811736495948925911071309448444725972986926308657674149356146523022932000*x^717*y+385940228746772181915228692491474256934032443516581230853548564755699144278668619534130732957712023951170226631444965716867502206388588842330025874754659974054975396359957856840477279636869771364772464359993411530052353952928261138406874524959003721857191000*x^716*y+969590188711189060530890329206651115665849928273235653653125517070458201065006075741886332623585295259782043046016124397463619578155191617923854478331005408503025908048174826308006077964907916832200296427211518089535036597532052544208147929370690052104381600*x^715*y+2423975471777972651327225823016627789164624820683089134132813792676145502662515189354715831558963238149455107615040310993659048945387979044809636195827513521257564770120437065770015194912269792080500741068028795223837591493830131360520369823426725130260954000*x^714*y+6030378002959834400862854486529171573043700773406709553208463581779679055404306080833683288268640251005961487237417359057395682742184728355380070535961131199226136745177672700208330484903695580297831111925339929581254495911479838994465310292427462519185788000*x^713*y+14929373319827645582691719614219789345764439761940916359158453242392052661469688318175056196303960065858508820834300614610844172899922608740923577403264883836973039928165557761279651512973385238723449940287386700664702970780851129177270021661461044361734259875*x^712*y+36781016621859112992652264239877128076762218375439212621871345012398413477392450112597370282935707843914388513612533002086232010743061928801168121491780613466867835393958052339207999575214706885713136185067886958038991402062166103717011264439308870538251879000*x^711*y+90176906269454583923364689222595303664061852637714759221208711392466455111813903551919759555749269920769414597167279187873486067718334590957346670278124193706700106776221293838540992061991919295662206302011267679881803058159310688768258651780512437767920986100*x^710*y+220019255846435582768346836247569297599601083755249068890234313019419873296865537875817970050109902555829155889995767090687886969347139379999024521984426726913254556051948861255546750391801590034089919156109965816893746293103472814520493617746267459846130241000*x^709*y+534224836969598726653280503080570657527798521857779417271151123050577706053005706691626509470986030520831751801393831737320931031736718563079823240023830648566772192605588159692406322012970299089622440690691663575950911376062884333887088955418163113119542263250*x^708*y+1290891414929951871913046403348273124674680387287740025351450495289450566162211741766797162817263172726105393431354378395983683175664152705325989262583181225888309598514527020690183194488679084489599617778190094920727799502568334840928528943467779809176231817000*x^707*y+3104286497807741406267087779480371085527207598001470060964202381529393028151985379010631272489132867746110588965876005190341714303382843410426783702878602471779030224999219740231154824841823512701180033228504752071273994041890519498423367221196327636352366988500*x^706*y+7429241584583950619744284651908955886041384963352670722171955530033055857204412466377985350431619676707640934948842236150444916265045042195801048455024723203647441826608302157976933241824838643956044418506184254109557423029066802596226770366659685800897529131800*x^705*y+17694646341661098604458515809445317228578298645823083983551448137409812092328076989177296189372607675942185334928830325966431303942083630905539659327001452224903535431617746693830195727994970418881794983266418578200128321741527350778175246988159048951462020398375*x^704*y+41942865402455937432790555992759270467741152345654717590640469659045480515148034344716553930364699676307402275386857068957466794529383421405723636923262701570141713615686510681671575058951040992904995515890769963141044910794731498140859844712673301218280344648000*x^703*y+98945752946062161125005908935938815902087349325487471363155201913788499336070698470925293332370415679342630199989800400929862941456901158551086297842462010751039008965864486608104420357189871872524200831111447262040787155331195446956457955815467552873996920428000*x^702*y+232307419960319856554361699240899828639683341894622758852625256667155607136861639888259384345565323768891392643454313984791852123420550546163420003630128199154613325398116620732071247795141438309404645429566006615226195929908024092854292591914575993704166682744000*x^701*y+542825004640614064815358503892902599588060075560435179852301016412253602009800031872232761420804306539976220810204913677796961128392686442868524741815732892024613137013599170443939815681313827516308854820419235457578544489551749630302863689773725905288736148678480*x^700*y+1262383731722358290268275590448610696716418780373105069423955852121520004673953562493564561443730945441805164674895148087899909600913224285740755213524960214010728225613021326613813524840264715154206639117254035947857080208259882861169450441334246291369153834136000*x^699*y+2921874928721617367210346482528406877499260687022518024924983909380604249228786556897356385593271294251065596383283803024642505996815707866664860576999825131104301422859277838751839913454784887062220002460134341481960592932363106357474986286399464098235226920732000*x^698*y+6730919802797653208953207408596792080839881054593127331345342471114395267200306985855956294204961595337438238533109222809242472560321333633439183771438541061091757073121372711052093265978349343793496903357009143083856415401945373721180001412233748978772898979112000*x^697*y+15432404942598566732369689354578829211662490444248058387985867441995833885653335424807899792963349447204587014005187922033032905837315689284562865423331128682832087763044726248695095415746412804684432044867879515557394478734065544354152832185285931046726021672503500*x^696*y+35216242098520008018784602592743820102678994587529995534551356523374099621031873625135404117713085951653746104090527192573740663812366294236248374867667100207380764206816817931448480030686896105115949846649325058452283794094785635640952036724455763962364954373975200*x^695*y+79984602151867338474036924189401813631901638033769107504945074456683004041232523429637600855590178877122070399813452283786763925979067236909126210892250440013495526548162380595936907259239845728939820730134904953020709924496326852191051194521231228607332167614094000*x^694*y+180812097372625188602546011034022340913810217574709317942774858869504901643698277720418550468337407624503963705115752068234573826154634079527470978368800668955589235910178150272248252892222973732521940021868482206502842630620361027428630387614770269229604313759548000*x^693*y+406827219088406674355728524826550267056072989543095965371243432456386028698321124870941738553759167155133918336510442153527791108847926678936809701329801505150075780797900838112558569007501690898174365049204084964631395918895812311714418372133233105766609705958983000*x^692*y+911082315887305561987586210938423251789004882730816854488350987895854795660965108125215802845311791816675312261699760421492658405575292109463664444402015021242240907159053009624241196612269158904649387100482934613349275002834634691606399720117143395438491639235004000*x^691*y+2030831871864929494623942166962743441890975399893530472424033976245276334844280289401690708922936929501040776686563014358875570833072667250449652035747717353801253118860985902097905376964767705816492666085270025218788222667608814748064587763228858342735476524875444400*x^690*y+4505704153012222994503280048888401848568402012625517768400589850833571289525895175842979386356355245516778572069866494879820398311318779430258070433009405061488310778180322419445513537317330279785787587134521920903420815564791261016606320117774637480667134412103076000*x^689*y+9950096671235325779528076774628554082255221111214685071884635920590803264369685179986579478203617833849552679987621842859603379604162304575153238872895769510786686301814878676275509061575771034526947588255402575328387634372247368078338956926752324436473255160060959500*x^688*y+21871139008977329508994622431132412807001891771615665589318305154525471712096943782206922303527441117215630171985571335103537141110746535296183477139144694643518339219324717345934665285508404063113546136484718759827254608460403160504463905321423639655890094409335272000*x^687*y+47851823245756131760125177102509450950351272761464847961342916054646493841435033052153361855169910979385789580108558940178758012557588757160758117180230589872920697591325098142220111627848005068022312725366247732488292726153812010403078671833815415425466544137622076000*x^686*y+104210637290757798055383719023242804291876105124967891115813461630119031032458516424689543595703361688440163974458639469722628560680971071150095455192502173501027296976663547065279354211757877703693036601908717284085615270290523933766704663104753571371016029455265854400*x^685*y+225899640962560416670689390920637091582073202565199384222570320305796000814031910604152966338787350495511114944633443154303799253374889821955111983566025281165201577307008005505431511503335905781739652127555289049362805253636104096930989538692266444269449304357142754000*x^684*y+487430140121108280765777739399734292246492336134373434726303151700834273049835415940822173425017500753721143918388880496983592079837301697846361503972117641378542204662439986642634554789532364525898807745261254604934254869044464360570337048787098573754900076278503608000*x^683*y+1046901841832443257116434578647856986177214671634519043767500165445502542430935814740822466821657084952174658164338381696351551542543638552292656940921246380696680269762410411562639625538523915003738634245325273255251874451438267793300440894093045049920115572635905548000*x^682*y+2238203937710740756593756685385073556654734815218626921158103801986936470024759328066585963549749629897752717454792402247372282608196744491108438977141975020799799197423084328168401958047878714835579149076212653166400559172040434592573356394267889417070591913911246344000*x^681*y+4763177754940670172626088446085109662755857528637140416589589653603449175271440945041703253679310931126155001833480081032689138925568696870140146698230265591139572667016001335883380416970641890009466876627815052519746189987998549867320174076551352165703353416792371125825*x^680*y+10090220789282416565064611038435746326087174827019487486856451602649051212413021316599246767918789511419892215722013878823142101150737426391574142538307104679049562036046357970095634528162107430549649458276991388515350184398252379781238998043784795865041371724046144441000*x^679*y+21277204707834661017636245015831899861531651265671527961414691422977347121827457993698411662785273534950642280979029266214017039383076746956145474482951938127561032989054276589114707591994009147028608640279742710564977562752836539973482235005372286932804631678966869799500*x^678*y+44662367776817028389960910590507826953927119375000916278139816671141304484826676531664158227146797079555837357596847809576171989788625493610732605880623572911722539834609286462599912530563276166208658384240450643229271788069421591647123700723351116224277214484023336607000*x^677*y+93322293163287432777788692807943823604347715484184013334261283599884762766134753123261219505488832169318832997200820886058853200885492157945882636361673329818630121814908910293765866614788080137417474463366620634154990742354933387484885016634903412604431093227419132354750*x^676*y+194110369779637860177800481040523153097043248207102747735263469887760306553560286496383336571416770912183172634177707443002414657841823688527435883632280526022750653375010533411033002558759206685828346883802570919042380744098261445968560834600599098217216673913031795297880*x^675*y+401915642948636673681028603381451313927926970980964278286205037344288978293414703635149546581921228115716691803895559889652238938782917146490856507520826242531768990883840828381740112660007559855626178363701642240348487737013271398861283936673019605204359677580663993331500*x^674*y+828413282407893327403710332351982218921782197067797931391138211529207251895295138379482551670381980886828899926072193778671587292781914852400114024675953784300955045430301890915268611415428426124440502193073109694173947201060993647805826829717477718372288754401735570353000*x^673*y+1699762619086927467508222724612451321141339690934841487275109805973038050382724476004243162421241076636694663567824958576359689780616550901418526642094258831812630321873759672518218827690802837749233103585177447634692275811933075381016223952438605196538263206440146459901125*x^672*y+3471855562390319933633816628995645251692949155952016654859798752625779847590245738221432842392322199087716759627897787730436813168918912479493160800873379741574734274465551671526574626772703668594178254131426276019796988892459047586756542541151193592929218464218171492564000*x^671*y+7059439643526983865055427145624478678442329950435767198214924130339085690100166334383580112864388471478357411243392168385221520110135122041636093628442538807868626358079955065437368407771164126141495783400566761240253877414666730093071636500340760305622744210576948701546800*x^670*y+14289500184782716584855396337064654726756377845292942667081568481351019372710306477453168204287432857675224971399011337819028454603596772712677289217693356499311116797321963425507664148660664545361940105372748429096586398392225707439148025544496403035550569852225243595276000*x^669*y+28794203685601317455627289606916427747590411983436682663486654560312746868503599498241474483940640306580498511644393328316054325692187472725244296646496552704937160052434920276098275046548146327852825091850508129715711748567466862279488039425506306116817262744393638449517000*x^668*y+57761345531476516697774863235496017223394580195002114171799054793660405129610824218694609475292335509897216233568933163108481350037180876217607177657236327948642711456536116349650593787069554795812874358426845137087373717847050642650744775784499136594696491030795647099932000*x^667*y+115349752902679151609029442449328872718575404161875479498772363914285898866618023215177558443173616123058213256857719819740590001421555821668095770950229433358517031561405956901847143880165847451518524542127861396518797214981984367209720854635511748828331016519582923998966000*x^666*y+229322195322938253646607787078367251434540952751668863719947445871386294463186876003905235591503368173005283668857437014767859525214197543972990398366724783930663710507153335213821486042359565381227872671812405045019459537844780861378131609514181566327368528364305156368093600*x^665*y+453866844909981960342244578592601851797528968987677959445729319953785374458390692091062445441517082842406290594613677425061388643653099305779876830100809468196105260378740975944021691125503306483680164662962051651601013668651128788144218810496817683356250212387687288645185250*x^664*y+894265830920557927795995252775927684253884971536552418611169936051375337211784627740253601700793302692456311438645346617925109968503436020883792923403375331994699978906480735984659949279923428798705131561444517200780632272950592033613534985667320301924481130639241423324638000*x^663*y+1754136822190325166061375303522011996036466674937083590352679489946928546069269846721266680259248401435202764745035102981314638784372124502502824580522005458912680727855019905200679131279849802643613911908987322201531240227710776681318857087270512899928789910100050484213713000*x^662*y+3425482525929189557323393660565108971611035217723744356381928679483382588489252621030910154370567674779068525844286838270295843289835830149430294608570995910915028442005968074462683141319352711947116252754423620346353041388626944433725909710245072388651501240372369972122354000*x^661*y+6659541028350571462914009440098638324220277290927632410495455462172105561739399948533622388349838920673424398773745882637251624748769069790510072753721847932690687647546896756528922224741447478226599538443158861908645177523183559619684783289623508379113653882018048681096694100*x^660*y+12889434248420460895962598916319945143652149595343804665475075088075043022721419255226365912935172104529208513755637192201132176933101425400987237587848737934240040608155284044894688176918930603019224913115791345629635827464226244425196354754110016217639330094228481318251666000*x^659*y+24836658390962233129939627736417672075049025097460723024994369833454541964834547629222734317614849172177626931476505583802766387716122337249270729737989234791415750762498047326273682773653728852016576660068147651374064357599196184433346192347831873355041867052913944996280257000*x^658*y+47645834464294896208455612392311452552134864472679754374478995190708713157029540349937490323587669840504019011403908670968572253986030606151662216232060981028430215748465641809586248586192867593664453184620528147533919379884172272178255960830534614191304806183141037339802942000*x^657*y+90998003613493450026032957388804140484745947553926158790792732093882629488861651191595730065689241526776571193291767432634744101362854965818726965303674606208368173682389321711913271282350912816969609715975834281772630908674131345410215599609480353266532725762568783523984107250*x^656*y+173028088030294791933558318976972510602879251001088580193507339865469579549835487483150141806064180990044726674780867929879397479692848862542275041273074033833882672277238826211637988293397677704150910068638108083602451814754290326345221545924113367370566574203609049251401664800*x^655*y+327553172427292163920464447774326573540132686143679248632217652057464088454168336131396944748474099851096231132894417612921980778031260129957196971196137260581483093472807604533592145468715256925487994494098152586010421788046416658254682406301428484473182387581976668380543614000*x^654*y+617348054084867651884679391482448354741345177919211033445159494079485630688835999509895106240639946117051686342688614175363041581649694884703189680582921522825043063778720960705963294341613193167922617864957325046832322332513995661379142056833239852580580061898019426861312748000*x^653*y+1158414595739708553680159892638042458753156325233462082872669970212368151838534217471153748204419209236881468913148462806069155611543824022158571440863930328749290576573289618795959859784693721662797326051198658780406627825090917146208562537678464436020456265573007717644934553000*x^652*y+2164144173129770707734854584527231183687845054590880452816563955812217865325857621178201271717138465393830136766111168336839797876007373244834924296399090470901253455374741637406778878451634116115025377035477150500931579776387615986613131159215927828897815143706593214626066844000*x^651*y+4025308162021373516386829527220650001659391801539037642238808957810725229506095175391454365393877545632524054384966773106522024049373714235392959191302308275876331426997019445576608713920039455973947201285987499931732738384080965735100423956141625761749936167294263379204484329840*x^650*y+7454274374113654659975610235593796299369244076924143781923720292242083758344620695169359935914588047467637137749938468715781526017358730065542517020930200510882095235179665639956682803555628622173976298677754629503208774785335121731667451770632640299536918828322709961489785796000*x^649*y+13743818377272050779330031371876061926962043766828890097921859288821341929447894406718507381842521712518455972726449051694722188594505158558344015757340057191938863089862508523670133919055690272133268800687110098146541178510461630692761864202103930552271194089719996491496792561375*x^648*y+25229445633065974235144080252055773735613043515311956893635594388544559689184803330180149528141512945359658556166399392346118918439771509194920459520556252295683805332098882502374636769258037666692232812592768678750591171883227016115891467430491067982639472436653138035382214107000*x^647*y+46111444419756173248978022381582162731473556933352644379045846241209972087295389137363154646066550496179940920451018098440505480877209509743258579971186144732506841948779595985978502795790820255225634547309382302688227367820474235669440054880021810691434290018402769234159018438500*x^646*y+83909839704683064560112119601414301759244838813931854278489061047384906953219215162638303947490117240935892491862979412936807156751203783927169134257426054921688506757497518329414402270650337703875380049470030894469281351019792552795657113950687576638497327751797715282441481440200*x^645*y+152027659015507237756382913322787147850317193918500129240520911167312542092208971291858724848682937135965310834976465509393934314900355170317483403359662374787890693422994099220427779394858055671347247561539803165541254133167882574587637186792678334078176338201993051565097627890250*x^644*y+274245973126013056344847608346988580435866310598078664512312231909661840636925987428451033060369219931545266604271271114985136411192797562141342609982136048636979290096381512319203053026018453367920525012973762573133242750028337193373776885978949151670435747344771779293901603253000*x^643*y+492570281340855852597030760243334238045424686353532349948091522675733417680288854515346408541389408983194431359068232756802912604460806794572299715694171729814462803161936626874993193004832026579812563081961255124370600805218493897595917703029229900905279847884604061692678019250500*x^642*y+880863845740472026092740245337661785028308213479018854224720773141562267829374497489839538394350976510336559700617842423029164044746066746839600048678713789807479441866193076472829052671593763410138344007295615013498400325766777387901334722408817817217798502345169380520053728019000*x^641*y+1568427014221229357570684714615114456119848791222364126550127821065948371329525146919408733641052710953127041022488991647671372646339524402011621197786265386851650895100638227830787285451365562071940773635212470065701318357823623071235432102955700613435024555564593258092651221278275*x^640*y+2780590828536251492646089244747017318328817801613055515213523006875919550279490565175683073491062977867039629513553891009722101090463422762569079131809445561177442030095314309727711530994110691761889460184310196238362448058191464724627912869505951226034392563881827382768135129136000*x^639*y+4908280495675869347516163059097635542574902141521388050335472932026830366377332793224479237460743764798448406793262255124896195018801456202435473937089048932575650434339518905845325050566952298441567306789431534243960232898299298229384630728216306169712643227404662148035465048392000*x^638*y+8626674810581830974422347194777662468768009824492136573316891819925944280299554606273327144627973889645757805879066993855878160942135892719432045101550449639072355308839154440576631907057067676048815266478394817762111924487919978706191169158683204783131312339074860745032029478992000*x^637*y+15096680918518204205239107590860909320344017192861239003304560684870402490524220560978322503098954306880076160288367239247786781648737812259006078927713286868376621790468520271009105837349868433085426716337190931083695867853859962735834546027695608370479796593381006303806051588236000*x^636*y+26305449490897473628854992952842570760928205300437665770141645467335824613625765141869077019098451888152680651899730312771486008571499311223911962186371645063801456051336928472224085787820592666965291483809461457997891977959054619999974715818121662804452467488740602764988078931830400*x^635*y+45639236138578950148423280123101181511446476409229283508305860305350406091946341161439518871933106417969814792230406416966922446565306182041486601061054630097032580854095490655361460314934634818368743421363409906635686901650272359836021706405757529729036384850683832666031229840744000*x^634*y+78842713111332573280927410348899588769092823006679470692822657857199339134316022605865544863230489016329325826359884654923784280987477164616628079217189742456454104254758967508172114004546480857890417790584201310100886909117909199280756844308583852447436152575840735450310080978288000*x^633*y+135618036411612823062030029214275651333792817834858980838469408759802124108755549754111113854415487900370824043711432028714009374633350666310667321044785616779715891286039202262698228708907397780012593645216846275255058188781620986806301854476449941845725773316595612880560546900153000*x^632*y+232278046103358547900279074426618459737011005072170395365616981940907703080578610960970796628700781444537563131776761631835376489886931222515831292412749349064445645779882861327981790092220800533788507273108528037835221613306190958432473636935274697144982896303762675719550855395384000*x^631*y+396128235381673631689394853954584454308253903244701403988390042174899353091473252746952899115432954301360006313921990782940871797617982706506728501384986052053149195911097528372855431211327905775190670511706705923983850913503260796678083418665292794320227588020741211835234026363479200*x^630*y+672670588383974091548028997281369828070619835698549553942549128221527203362879108438221904158282375228724539023641116423861857769539970633690671040087712163863838257207524104784094128472066255089946421623652896852048048721043273050962783163771251914883405338148428472927755893824776000*x^629*y+1137391935735268020386317847553714037248440528640827068359847853901453255148524083891509617514945198975450900660941565136045990690969466474708150764019276750189124365009496402981707545185294823794559944089456107849296297434237147174880619919387412511993714940041294380299888325848882000*x^628*y+1914965511103882886870261684353170014455819442322893830911486467158478938963198725694016192491650361813895886367483385805460810064152345700044822197866235386377399735190251316548290451411166620222476259753829586405785723294104365752882116110925724014831241239533332093373538521804552000*x^627*y+3210383356850627192694262235533255612470050241541322010645727312589214691791244922487027146236002077158590162439604499732684299225196579555957496037599276971279758379583656618919192815601073451549445494293184894856758418463645554350420018185963713789570022078041174391832108698319396000*x^626*y+5359199950369313660337621758516848035750003869879646876437934127148929058830184857271677316116632800803406511165846444887094323506594823472078380052099059690723009988318317449182439206843391948453207678473423317814215386555312312062301150358435426219388890188943400451431733453727845056*x^625*y+8908244598353247440720780848598484101978064943283987494079012844330001760023578552645740219608764628995024120953867095889452000509632352845874966841919979539100747985901458525901660915630638212189507444270982908600756959034761157018452710037292929221058660553429854473789450554733785000*x^624*y+14744680714515719901882671749404387479136107492332117231579055742339313257970061052655018294524851799715901993302952434575644690498701825400068910634902035099201238045630000318733783584492090833968839907758868262511597725298915018513301037303105538021062610571194241887651504366455920000*x^623*y+24301418214664797616065884920314638623020621607732563585380295575337016295543248771968456078013181669902134766740051234763562545451564119640854315676042243033868707149279074599394569241107334893022717625750727321546892547251915493475625783703266534886566154459931250518536738678047720000*x^622*y+39882538600320591338240053879777586341738328865460829947510669783270776083978629910723956940697094983322237005045677752039408715754282011653328190898412335533156558962669088128821694110735520589604565602155547213726034734540082947076092974837550883111989836607063952038337338938642960000*x^621*y+65176464396839703213281772261426002942682900593292566835274015619503031442501918880419940158349726275376603105614120747411770559166866129570307385652405422016026902936361851915784926428333574437222197997206828472957546237235240816142772993089787101085646548771017668988967072318150732000*x^620*y+106061438126090855622663251449039689827987922225305489338241180273207032793572676393334285821986431209274262271603031137520466526728233596676090758804439269422405983780956294456132951143219989897841897003328697252581833771878869569576166025500440951897902520309792532213017283037410640000*x^619*y+171863953403272878613687310594124523569435926328440046859610708348468987694297085569303463151334033818169550644299152550065886858755959676289267486125518083174003413508931796513995541250400978394670508494922679579445432211500053045988604109384222380169637853590998893821617010995175880000*x^618*y+277315726379171381157333571663626515837888779297587334097230855768547870483226106741069295633223062401119536026571478527260360518828154255735684873173812468411316212920417363565663823740855886809155024151076281932368869730305568622509026996343210002466935231120724063659945986409970480000*x^617*y+445582820770699849411653160719941563208274418819300482130186036482276135646225280883436863035673514326798837834360943362811568854471279103616972830073547638046307560864316440937537966791948130628251692451078296750707272457287853750229348064436876488338799577087205071036944462538936943125*x^616*y+712932513233119759058645057151906501133239070110880771408297658371641817033960449413498980857077622922878140534977509380498510167154046565787156528117676220874092097382906305500060746867117009005202707921725274801131635931660566000366956903099002381342079323339528113659111140062299109000*x^615*y+1135889885073493916634887850125446886520575202378734907813738497146527765481569109816844231158297248957435379349769865981882341328496732222691972188581271699061053471218879217312272951614707151653367008735391305706466207507697533912501757760118876851102017574750802564508687438182160497500*x^614*y+1802161213010659599002121808726161210138587013593134970019729812010253354019853833146104284059934136588799284032968211144381802521180861976053930035630234685332007316094035760800350367678114188928080990603437368743592380903685493080816742286080078518285888348571040761261845186159810195000*x^613*y+2847228926741067871619331620487466035605551132300494166551789625675993056737552576594231768373040272497252477093323488225531043673927495853920255442890035727083815682385680209718079318006917520136375379484296667628407550242162905305517172735482185906467138035242391718179152317309184663750*x^612*y+4479444995284147911133755659995704919770172989634710616785848974071228150959851354436169260268125055959687701750935667850964007013993900932131610105523655179885077628843280946908649209820651728337947897800487302284281287270446524542356066103123644665187373978324791083613473568620105435000*x^611*y+7017797159278498394109550533993271040973271017094379966297830059378257436503767121949998507753395921003510732743132546299843610988590444793672855831987059781819954951854473483490217095385687707729451706554096773578707350057032888449691170228227043308793552566042172697661108590838165181500*x^610*y+10948481501687682916641498275539374258295895960172817850234466333045363264110736430663680536392766014864812140596702949470344252437442893412123892730209990963964635602637413874498804164156699492877149721222503917859364408017365887351180598054267254266915772545487788607604287059875398365000*x^609*y+17009248047264793102639470535212956436995409795268484874471403053124046499600608383352503690467332915950690289855592082212856249322455923693835333348719093104730773168383125126453499326457729569291286174042104300960083991026979146420584143405736627164672718061739957301099517396592136745625*x^608*y+26314561864470723171513481133357449144257529657819946065339982331550687714394834343710743622911293671496233323746055943983248345007768960829139650575117579154392646530221221569678696158998217756053694640757250419805931467034105142553982593360528929557559828451750366511624698669536944380000*x^607*y+40540454446024692804844373218142059976051574878925652948379109835663115336643818392468074566261815377152826465771208015222923211725166901581948649490092311032274965593513404804048143574903345629250235144516880722898985788044928481041287903984368173201621360076681402214609624600022652890000*x^606*y+62196241504534085670216937139731869229081656649693533384095545722561640237990263154014311866214329414062310982930005208164788522292281373059900966053154279710275010505491451420894113940231461902090234171081594222979203512798042175977267012188676235342234289130301087954565651918009437092000*x^605*y+95022035631927075329498098407923689099985864325920676003479305965024728141374013151966309795605225493706308446143063512473982464613207653285959809247874594001809043827834161893032674075353622350415635539152435618440449811219231102187491268621588692883969052837959995486141968208069973335000*x^604*y+144567530281319782113392572892659718429197637412735738806301009579030064980830992301732118681474952640298766502444358593285353674121857487618941372256212228657664137209097818094185730835046820905922024850498919681455999209008603490481724751252996399249161984670347197162795337021849531220000*x^603*y+219030705426220674910491762447924146263332098894169976131154544663706354732264041100362983831480895583166221610487307114952432827878090615663873486106773803720028830997703478167824109782746816598670806494600122029944642017668813831056482474888333740570966524513113969570767809608480571170000*x^602*y+330467380116754000742145466149499589099062464998221367496127909492609587841661535695284501921181702107935001026349270383963319705219575314861282803599693809121447008171973668814611814759933793464661216816414219203074372167009087534576447242813977222615844229967154410229579502216304019660000*x^601*y+496527238625422886115073562889623132621341353659827604662932184012645905732096457382164964136575507417172339042089778751904887857092411910579077412408539948204974129778390437393954251676800524680653478266662364352619244180931154020701111982328000776980305955525649501369943202079996789539150*x^600*y+742933524127814792192130019286219151054376090264081203984437183061315569673959786606730619655723951247639410038039569204845218738791638768946250492381855284097218149294349781636839279316908515731651089675804036437834280569971801527233584013458355277277265768866308480852782846004982727490000*x^599*y+1107007912817316070952949954110560376819829049920857316384770827496835886155974905914009057646215539296855737842750502372393746329692018961688567275962018196950830028426158007961360020673701992346415429641309994592693368311972908245803275681745161221614632327241091492613972449644240432255000*x^598*y+1642656902890210943994699931905992817216520525689014082377401873059820992360478892646594085539545638956624643250533003520326204231155899104441099828846865711604457461535589302136211643580331988643068056887105153266577256204863025138933892947105723103041067324293232537427184925278550318830000*x^597*y+2427391512439247360308999651851182455144214737218666849453734946080973100097044304232714527393833530834418099060812383914937484965346712290473605440152422846108567090437492112315144433706579696088890173172281625000362925629463430712731520023322071021078012852977870853574330198988352822627500*x^596*y+3572161336824176362331268623464949983372720946623025783393644513245086339895897297093081131670925393524230091457393039045191953183571951913882145289705787694520261693582087157876113783923756787330811217804147774074608157222617789394538236873827047724845668297221755626494569873079156252558000*x^595*y+5235064028104396393071686775767599113563470352809606751525168683204005842950883969877791313655666524992406168515144971014505448631096826080689350855603309552314176619904782903783959855750333222812395750230216565454167126964181243078202588521987914769170375952824986694000662745029797956335000*x^594*y+7640363716692902843942461780850009517092632406803209853577273213324765284306695523605425160470432225664592786481562930669818762867006178604249322870339965292566636147969142616333346816500486325185658662498153906338514185299075327735755129194252632365816224363582413012865832114367813233570000*x^593*y+11104744323526694574651666264813861871656693669691920203851281900739180915671251091906904706271976249556626280351879455605888545049349666451764334465959802496303958911141425420308026132805853899105626438385797221712595372260665856243389195127921105374826031979422477736836858930931650116438750*x^592*y+16073370756791694836659624520219575129635116509676324598239508276864535701900686177038844953821540194957268356890739945522459703347713942639228572136548173784381280379940645107145113620100404665698119441379931430938524353003213170895076781212052064503415674649922021565299316594404735620860000*x^591*y+23169175895765589386502044125487241223449643554191970335511096077138879511764159830804774067581781110292062436396164165375057767508534000243375819835853587089193504157426637215421371096291071115677045341111071891913824128353412156095098482186153097857362594434399792061199746603154143297386000*x^590*y+33259887538933571138774223927098472802518953033998205591122984636282089810075071289963057663925184562219748996286464373652759325620523260690004218255848215042881186016743834445495398897352145883818629565098619017589187921480567328700992954963090821741712726803639604175444891717423222738340000*x^589*y+47548722719494838351305868672478156506513745963652774497988927064976094412947128615990876126339644920260757667021183291459891365996330583850515739205569414223924802339471161379603859103253431858177603917094870391650562344058383875254574879789467218460846592445009045775089905877578345128355000*x^588*y+67696486583687566466265982516748561805883977304183611149679150397593083570975572944800569400212375818676332949657277906485269063452402865143107154123183572793384464347721653489605494316496411459100317441287612083027919269506851619006513388174834683910357860430182370256060204978247135436980000*x^587*y+95985115035325124434053458302732864203028731105207197451356669766635604000392901735743802507064407259814027636349811910886118213155943192847835505966929365289170726019595677773909239526046844266888614343081710852022677804832178503277351108354173815109613681334582249614268937976403547105090000*x^586*y+135535608218555476911699582085304719091505629946147994473481948152405937215012627511194863299131910010243422156387927180191000657612970387973088208425591826649286856499959198013279070752442049976859585554327427853699492032847365308242235540953122543745141246414614935599907464226921635189356000*x^585*y+190596949057343639407077537307459761222429792111770617228333989589320849208611507437617776514404248451904812407420522597143594674768239608087155293098488506225559641953067622206173693245621632779958792185772945419264910671191607464715643729465328577141604877770552253187369871569108549485031875*x^584*y+266927142085104761184012666157209833462587526602575636597954556163461333184242494828702113871491801189238394354756799032930118201593889523076495662277019874426203431416286550044137738262453317850110155003576499100361409669006951461376345175078541700361384289251804594391904088720286313907095000*x^583*y+372293119223961903756649244903476872987293129208855493149778723070090806809601374366347685136554354290253550021108167072244638544328319597975112371070580351173388996449031240851034213892369101211995742504988275061030387169930748090867007744188492371556667561324885355336076755320399332554632500*x^582*y+517123139351660687318305156405306778230559907397503334160313166651056920198539379191442369330488387104839059933854303665981812966107594286447530787501378912608382806523475375120052297101095028413798382190699704261383497214557745558197132475221247160491600288045544813378512342712344657629585000*x^581*y+715353676103130617456988799694007709885607871899879612255099880533962072941312807881495277573842268828360699575165120071274841269782172096252417589376907495774929549024140935582739010989848122639087762030467924228247171146804881355506033257389391905346713731796336991840275407418743443054259250*x^580*y+985522879191961420724592645659203020745017970788432719971396509999282665809884628435314159127858707649523054046545771119571040229153586260870314018619017452611541896517818866123488423691477223588291928688055572570981850986096986190483371233458069608316137682759799181157624076728910206583065000*x^579*y+1352174756047738536965732563594024997657263993095977594463124595472949439582756397782101654348412776609179735291350714403392493584549588732331544589527040533322470990719945790249999519709396475018059305000910370897152823983294206171303014085716166595296312128715459066090673792478765425619892500*x^578*y+1847652503535680554057194850490180729659334723426654963592638336130885995458234510444573891757405638014434721036408304787614329295200147251270999462757989192104936720179973207481086814165558303925386000686823154559230099910978844366461328939820199272059736194793227754610897049770038808530255000*x^577*y+2514376166368131320025946765879326134465651262776367721681491320630946272121229510675752678169865691354549136882093377034088367932383219254677751627385282461897520017792086180935346914560205522087140854708247547595933414265648097168509874524236450424477518359423802864175678296503095265381974375*x^576*y+3407719227830690918435165499168216125769917946727500723973032942784529535863125172115843629707865031106400712574319494521493882186006433625163258676173935760124638894701745035808846641851007954640454428969295499800606227334148950515439265237553401045880119000066142234741625173613606759670628800*x^575*y+4599621023480392671596760943712967775393668590066462244799281554227944796059382567996737293619770875319672323310407768426899019382520420972931628494835711413313773155994139426267809434423308858963054686989072564284855823279661142127646895567120200942209080809948431420132475293023060767161060000*x^574*y+6183097113531019656900563891548579632496406957138523017599034220437565135686383124192335378308544455347756237892679295262388845727322533111153992402893907145766055717893761195966563502011661089097876792345966397891117664080855961548640089123014040610838436498619202892637097934883458736183720000*x^573*y+8277837958068397811691642780040504975281404641215826376365062168950291641935274603182729373296252273164168982038563635947076655611578998767970181417893011202158761510170853189927198333300658420684774303771585855120585097005445013942455072587586554369183233910534587050189385786654723962227270000*x^572*y+11037117277424530415588857040054006633708539521621101835153416225267055522580366137576972497728336364218891976051418181262768874148771998357293575223857348269545015346894470919902931111067544560913032405028781140160780129340593351923273430116782072492244311880712782733585847715539631949636360000*x^571*y+14656265035835829923956365976443808808947851318245695692726978289831369077659044336177793712099721078997644926338046003490793086369648397818638677797261734562581869216457541616894357359115274289026375589003334955887919660124369311507416578131819914867608144381132555676459346617611929868005492000*x^570*y+19382995523031143982958535050053297032715255803712404976460272912305986947252100398193369874470628805170899322535234853804529139746402753495647439314244057310143075297867282416774440126904191055092886509818795649318130408981184472295191298225376685555769471687344679200885910839997215834717240000*x^569*y+25529917714362779921998626026574828730590232759982311184272905757180802252283437792990804302254138403107041005839232944015687686378942515599591187430103862521924559825200193738760778778260381273953362092793737788106518987755310103555471871968146606669520438403007227928944637194348184745264142500*x^568*y+33489591828540551953106742686130491267841229117020676103157068060227934594219382601429045828361086865969513374865321737184551052801938450024405991825170886633840993027052448137681575856932786521028890689854141024583147309572785539998863795099093008287038357997478303611179108375034108395635180000*x^567*y+43752531259867495293574938025428545043469992878688302650898750207717140356964032108318592130600774776508557796195017108257236052854145394386723957061916803505501942503084649986325929748573479164570002352873958435342498904441864979675934958113331188245969467706382945040411415780286496452362090000*x^566*y+56928580903643683531410149246879440217480496481235814483698143948431957338026763616800742864183996605755962557807769386835851967621715616604335079763321633986469194153438877913242474109638136108383037544199219481388170988308265697693285485729070005855675215452443096305454853635958981590889524000*x^565*y+73772128923299727512033794322217623217606606678665677025893236997394623614644773952964265408862289179477336800828875466885909086482269090324425046023570466060447464900671940415096325394370520415679853698331557355468616074298555319258500686781937048872606643877592544524270624551185377520304085000*x^564*y+95211626344945186079604256287713362688169625095577670120374795575584823154827580113207884875511055142391803102213926231861905548686498322523971912945752271986481396347777973441909216298455316966689788297160179287149426695433375743848499742208266580238329856171538203916907625278875407142909620000*x^563*y+122383894137452373887710493812745715053514837737009653602216917600580491863397094985698719600257360833713664718142558147347609187010270674842457047918855089334221520876253422483549974374498501032525915094294933649920381802577603981248185741696927133959314404165698650240226011488600123793283370000*x^562*y+156673686800109872721852613947068546378303733048290262698054459434000538558608581735678087506479810452271251871517352343529285565147544690800594216242361184979117300073928071607642564005622226834349804744860484535888962580976340404240274229689460248941081310116452486184526237942125898796868460000*x^561*y+199758950670140087720362082782512396632337259636570084940019435778350686662225941712989561570761758326645846136184624237999839095563119480770757625709010510848374557594258291299744269107168339213796001049697117783258427290744834015406349642854061817399878670398476919885270953376210520966007286500*x^560*y+253662159581130270121094708295253836993444139221041377701611981940762776713937703762526427391443502637010598268170951413333129010238881880343819207249537156632856581072074020698087960771007414874661588634536022581915463226342646368769967800449602307809369740188542120489232956668203836147310840000*x^559*y+320808025352605929859031542843997499727002881956022918857921035983905864667627095934959893465649135687983991927392673846274251395302115319258359585639120521623906852532328908529934773916274083517954362096619087383010732903903935113444371041745085271641261730238450328854029915786257792774540180000*x^558*y+404087761053621013231014900467157121552297083818195911337968257514716642177281985398888533981562568202923402924345625296210005143518240063535360381008192440329887186711150182753281272788444556665956058803416367403431126321395927298650020409240987768794185204228115764109590729139349544849195080000*x^557*y+506929916456907442274043467477942605190606927222376402286595314044362994803482130331488543756149438038352106821757912815290479425539774133759449847345863038882313430175924891426976731853972112754363794489871433882232291353643088976009147225106374295536849456655541622993337919213102920002255990000*x^556*y+633377603483237163830040826781429412328039216934025347576060662041945674406148459470354225457121545054660160433477302304048329349663178468247762056459100785659699476804076942996402388563614594812193864576109027502294728073315859484631653611593582265884243366068496949178192995690978030384841192000*x^555*y+788171681464566425842315378618146466013591402238529300234784007697936881828278912569611199840140039249633159283811441207952517464266959753088582828104934834172944416202382742966375169625125785024142589326772444537608910494821304964059568956131027259116042753739945755143267068628907638707593860000*x^554*y+976839175685391051267657091173273248705882856465649289329016421173729379268157757412896207408137766765764586673896059125739809116787238709644462833937659727364230887194899417457207704636061934906879182297610591216633862224006718009147653695070669130984983636626241495188747105191084634997778520000*x^553*y+1205785857486654578908514221917009166371324150949785841515504644886322202534132231806543756019420055851490661675590447983335076878534247782217383810641798725965222501381203968423740760410138950900678990648613073533032423682758292542541635029852857208559589176460516845623609707970245096325382860625*x^552*y+1482391521898960640439866036744296347075659089809090834112602592376948453894968801686441321431447373786242417026561085271271631262696892596400881655844705783369271315729230713964153451550994879503730073135934112673126721320451174796175907653627566100500875780414711133149738438306403770983544185000*x^551*y+1815106063480727361960813747213571749419307018855131221324542285332663551324728466064953706908283339902710159514744795521045930723879972934704190649712073081414374433259569163098330115121329285792345045106443769073095163216819105139317611371441753158613294566685568520812235287792952172915406324300*x^550*y+2213543979854545563366846033187282621243057340067233196737246689430077501615522519591406959644247975491109950627737555513470647224243869432566086158185454977334602967389718491583329408684547909502859811105419230576945320996120859925997087038343601412943042154494595757088091814381648991360251615000*x^549*y+2688574435708286536036279805796057874031943539152458019930859363931664930059561644370978807178522430408449917908468845081626958686083814863891109072663307041054639444904768698847893463203134518400597425435564507935272082360332637387992037132855391981649845448711356350976465500211339151010571098750*x^548*y+3252403511629450379134395879859248818917229711822399547289428104712036162632758898709263545990795346277771644622165401997199941191995431667576882498497775405072720564697159485581999156369354781641340814875693930129203313760402395780617298783233454317757428931332943223697799324758970981796452455000*x^547*y+3918644759606408276181750101944953973453138000808045269531535623959215376564139025537372598363359150691500197375164041613366448969210354894635583098410315300825502530593273653333377838180698382285932655808380131675493860411762357911889124304909029761703333976738149654983912402297702923001452627500*x^546*y+4702373711527689931418100122333944768143765600969654323437842748751058451876966830644847118036030980829800236850196849936039738763052425873562699718092378360990603036711928384000053405816838058743119186970056158010592632494114829494266949165890835714044000772085779585980694882757243507601743153000*x^545*y+5620161563119717132944878435684210304031474238001012294459702408046769421651199391889126489757975624018072651498590533366538722863735903730464191549035846944604997050456142476491291899496001627225877098462018873060905668222132855426262033542566897947706097414006030426226926998032231823778399163125*x^544*y+6690082911022157812520818094118622331275978086373196254236494770191340405641690304568238097217371421150616022790444748690146751067554336169305295848305253255722359727457640059543244624345349858229490462939471043643616375301619854161677344085681383990267214427175668603648683341202481645810610820000*x^543*y+7931692184901815921831450273158104641665624674455558004476892271209383930706196147119111979888717645599966158024479254451418528012406123449634881322335704187461225615741263214698650286068831818817933016105093398904986226613055853296486458162718322067063531515188620200395709725486785008024370470000*x^542*y+9365963320733734705082017533881683476650911924956236249295153836591472963927578021216903470805413864738957859802326265604943011291337949694340099513520591872775564888304498175090780947819840622656469923156776954698262603102998415003694249072316624314484605841464558057983604948178295151087600860000*x^541*y+11015187296775979294455155403978240784495963807394182197540604838252145377140912411909445168925497610486470004680561977591900367627421371270952160514814435224286044792549420679835027158196812558385109192234383331503826235388526396776083888582876725552469938609200708498628544080357516688561721881000*x^540*y+12902822430062969238624260126135032589214361075906417324667953606629411070837511285100000848632903925515604777716927262255154443641664946824976500386116692019771071991272640275728665217844422519583425084179104119332030731257709879086952928058033474616776066917501914510324108033390583539746919340000*x^539*y+15053292835073464111728303480490871354083421255224153545445945874400979582643763165950000990071721246434872240669748472631013517581942437962472583783802807356399583989818080321683442754151826272847329264875621472554035853133994858934778416067705720386238744737085566928711459372289014129704739230000*x^538*y+17491731199286228276695091301304727404960865303046640620842157409131159860609815514646005470104937431062551329331154812690032985872753847999590172949645594725146816817542391388910782293161301371040741132836035318000153971892201369561362392752539260405607871854324049692541609378599329593479804980000*x^537*y+20243663047449794363330310406897928052724104887362168132310858898067743200748859765872639951392998707932306172092306324169283865115665552534008454469740699067680691015129879689321314852214695767777754285200325357254488540745931326410456045060589618184938420658991410958825095336870344809695377746250*x^536*y+23334630953619547911279669630316751475828215526077681976169076063149054528175029751629537664401392059035948619874142343558572369251605884211244153969421536989842688998085194652637042496316294476406185584661020196749259909332944496679579441188120506122853749404772895212753228173252698533326284886000*x^535*y+26789758712846476679258848180728459312377886923715793685086814793529493932561461195540349035310611054901786505649498184128403900321049673933509919256739318217952443377629998152705617458217205031925556411574347221589815561144045720436856225398378692651774154359556864675585787709635608831179318485000*x^534*y+30633257286209889821679282502160593731926748645105425755538242183607601199117388176485110031811276880765640244147391927889866558397088920514977081120125901345581594782985907951916059363357574918732863220087155067085570684477345641784328103560458719220658240745189219993068116995600460633511254970000*x^533*y+34887876353739041185801405071905120639138797068036734888251886931330879143439247645441375314007287558649756944723418584541236913730017937253168342386810054310245705169511728500793289830490571435223538667321482159736344390654754758698818117943855763556860774182021056103216466578322746832610040382500*x^532*y+39574307505733837762998608738280435351858933987623758977718558310166370371662430164978276475590356036677336235805668842166179185723602436287176030170112897426547367057953602478511791449511692971298342368901979763283017517757632263598659357667657284034648042355725377072305245670933265063856165210000*x^531*y+44710547416052484791813321787291300365610838186017480887592669069570941845431383867241414486252083096756735194069808840830300314083474241847852068128361592624460961505900772161893109063171721207998765527419045221921877238147452621214655572173459612388081086150830160054029969045245880316824731333000*x^530*y+50311231699591968024758090333894669201218140633947483801325084091024626705050177175452122457990666754312249793751589566114775300348707745603740119125332577687822313371820401795760823361955440000508165030853702691334596467554458363574877820067799563833721816687770668425978521430956086131458827190000*x^529*y+56386952476873201451476758022521779676788975413894531633264765856254295607990558741131721992112421002184703688335997628124398588738276265729615514867162994908597465622230916419401431267954296102264447672291543906177969346051501004938793150033614341669573815736929414401149656434270698227842626236250*x^528*y+62943574857904969062113590350721986615950484182952035311551366537214097422873181850565643153985958328020134349770415956976072843242726994302826621247065668735178566275978697398401597694460609602527755541162653662710291363034233679931676074456127637212547515241223532354771709508023104998522001380000*x^527*y+69981569515012486699860468596688791026594736633788444323180527774497530257076301340185852198629957887904242199006348542882680144280415877632045631639670057855356760395444669892315700390254728397747103734583794262127264869871394829797454200924850769643486372430643041246760951288456068215656317990000*x^526*y+77495380136624353692898118909175377010502803093416256239985173914496212453099230536711070013640753366395013466689135439065873170297892108704117899457824106172458223090534518659701175590050499236242055925033843751324086992741797222049391388813624220699944909260038399359571074479427140803021522658400*x^525*y+85472845738923919514225866443943430526289856353032635558807177111576704911506504268431327220927301507053323676495369969557948349593263355188365330284364822984328922526324836757023355430202756510561091093787327666901566536112276347848593443544438478713174532272101175764232802734662287650391385285000*x^524*y+93894698463723550996759652026470351353827850584882811389549184080641914829411757309555587974352004171270317833298896989619213700601404608215311180438170161936663218875878856311698612673849568996926649335732829554415976656022710285686924453705001599257239947401637350315425552689649976370660557420000*x^523*y+102734157524115935504822799183774045518937167062539143005720132372752555346824998060455172616288908329653506750659671810817675241453001276352735873157244758771704735297248204709243461147329130931783760674870857441337982826568781337686739517338317649396519858767900280784450970829888990882542827470000*x^522*y+111956639306030309673314198692964617454875159095293178807903776823751218979212210829974112955538225778870836166689663225985023958326652747925110909787227064882734596712241258576670327179344063353634912468230871783671037652335916196811018847704805455083472580953745191168023813722760027642353561460000*x^521*y+121519602246753731957909786497988678529145745601349471164412224427446635600353253838367735103823782564149386755927738626537911421433721003477047466664886043341468176848078532746760917625913035431757894574892258748526272118472942371955293374279590921038519197243544259580292514478245780003471261501375*x^520*y+131372542969463494008551120538366138950427833082539968826391593975617984432814328473911064977106791961242580276678636353013958293441860544299510774772849776585371001997922738104606397433419497764062588729613252701109483371322099861573290134356314509230831564587615415762478394030535978382131093515000*x^519*y+141457157263799903299663965890896319741228309895929966433396757828518120167283478170041167475349429518433400754348988106253618992316028262430386083209769780182173340325564110116785726696980745517735443051181075003891746617668402133104849750479102137532783365188739420706901009339933968423912940942500*x^518*y+151707675906104244118480195303280111026824564236214746609729856221888998440275034559174585408345764990783647185823552461779243556976610020577515509529318314977983002667996292009306431530095292294382938924455065946202742749383503736953027268629761712716318391651691552642183691176161067585066052605000*x^517*y+162051381081520442581103844983049209505926239070502115696756891873381430152111968733663761686187521694700714039402431038718737435861378885616891566997226381908300025577177857373577324588965425859909048396577002260716566118659651719018006400581790920401521918355215976685968942847262958556775101646250*x^516*y+172409304408380512106906358786089468257851421361606374638199084962195500945339744054784538206335590091681584421302380239131687663720559804079002162001172810442644975665616029700548246366816824213841379325018006528927315705625526364975858356082895082324093422415033905092700978369459147660404025669000*x^515*y+182697102408057538549499536573736782207393995887298936087803557110145438244547259646530940691898824891390979376482974944758887133366436829425280068787251023411444778740313282501609767240556922778041790848527311445262484749788366415560837558400598698347547556674367203956257209589035928076354060122500*x^514*y+192826099872159291200087806568584612021766968965239534187127368284629887592807579996543949724098554402823333469224330845186997918994555503746599497652252620191956090908667406993485462754920448270869569807275232203008043452548707058723348059585026141582421856531056966803934714022103628400915784195000*x^513*y+202704486136101877839436567150991610588455850572065329995894139200850681014570263398006242230456062312804037028098528122092069533697145437340175291589355725734576792287185204482905824576381536809336248588385643688817881744175177707223519579030980349655291828689410295021349402240449101167356141991875*x^512*y+212238643970724256551720904665250929695888334341303576601017994418886602616482566175417578777082830069847989689951833125789651536304577635824478014915644440850927029961224590378829820415352447538609732673319937768251033646252946801837304753504830141152370994455987875359776879237443639668070234560000*x^511*y+221334585855183867546794657722333112397140691527359444169633051322553171300046104725792617867529237072841474962378340259752065173574773820216955644126314916887395331245277072823636812718867552433121578359319363672604649373949501664773189242940751432916044037075530212875195888347619795653844673184000*x^510*y+229899467996219495822536202522178996583588090995831601886991560436867856136504100631678686583380673945313955663570170127237379304527769141162214620171936064384056250376968038981781618099027396620961313570779787114110735602269339814733862553767379288772265700424685149829633203782660072878738866240000*x^509*y+237843148800966917426160420902010384676923451863573750732680293216190525962358917116919616810855209427164234619425236981227288752041940026121071629405519221080253315938773845206761877260985660325344123186030308213582041507225800743291739918430073288587567564057245409071714025864581254258695290480000*x^508*y+245079755762456783067930007744870741208675686707293033209333851833315998354722778692485122393335591052737182934417891250432987192773439215556804031922928528009672788026160473357069033769940599280476297319479506232250866299534902185785403404792043064102402276959595675067810801499406241710785410880000*x^507*y+251529223019363540517086060580262076503640836357484955135895268986824314100899693921234730877370738185703950906376256809654907908372740247545140980131426647167822071921585748971728745184939036103646726196307914290994310149522662769621861389128675776315623389511163982306437401538864300703174500640000*x^506*y+257118761308682730306354639704267900425943966054317954138915163853198187747586353786151058230201199034275149815406840294313905861892134475268366335245458350438218117964287654504433828411271014683727764556225867941905294819512055275613458308887090793567081687055856515246580454906394618496578378432000*x^505*y+261784222703396731461107042440837277651414723502884207339016447068276380670425622302432025012604043371590626324154141831912343669870015947601864917941444489861492237040252551461167506749378754869521211896963837319883415088414491762469347673362864618450355346700015202015167600257518714396717905460000*x^504*y+265471324431613586833798690925919492829603663270530463780411044914308442370009081771480363393063255250063733737170397350671672453952692228554003860447662017887710437561946249369071274450074230290218693754667553338473322343180893054898493415241214824344022323414099923170310805894948555444559002720000*x^503*y+268136699174902879874298677782605431512631812500154263617563766248789450827539293435852656198214492752574413794772509773871187237626916046109766951416012038147627209023411573157917371583107104088313258953007589014562411924939737362678598770815925816556311704171269601113787822018391814033359795920000*x^502*y+269748743458519530454705283059855564367417174098351583839713448210205018668185822254104275373754860444473658767486572958884440868314051813922050119460597280861941601061628476403355752574588709924515543074969558487595853279197892096321957080059308136094726403795545771060363700707881143576646528160000*x^501*y+270288240945436569515614693625975275496152008446548287007392875106625428705522193898612483924502370165362606085021546104802209750050679917549894219699518475423665484263751733356162464079737887344364574161119497604571044985756287880514600994219426752366915856603136862602484428109296905863799821216320*x^500*y+269748743458519530454705283059855564367417174098351583839713448210205018668185822254104275373754860444473658767486572958884440868314051813922050119460597280861941601061628476403355752574588709924515543074969558487595853279197892096321957080059308136094726403795545771060363700707881143576646528160000*x^499*y+268136699174902879874298677782605431512631812500154263617563766248789450827539293435852656198214492752574413794772509773871187237626916046109766951416012038147627209023411573157917371583107104088313258953007589014562411924939737362678598770815925816556311704171269601113787822018391814033359795920000*x^498*y+265471324431613586833798690925919492829603663270530463780411044914308442370009081771480363393063255250063733737170397350671672453952692228554003860447662017887710437561946249369071274450074230290218693754667553338473322343180893054898493415241214824344022323414099923170310805894948555444559002720000*x^497*y+261784222703396731461107042440837277651414723502884207339016447068276380670425622302432025012604043371590626324154141831912343669870015947601864917941444489861492237040252551461167506749378754869521211896963837319883415088414491762469347673362864618450355346700015202015167600257518714396717905460000*x^496*y+257118761308682730306354639704267900425943966054317954138915163853198187747586353786151058230201199034275149815406840294313905861892134475268366335245458350438218117964287654504433828411271014683727764556225867941905294819512055275613458308887090793567081687055856515246580454906394618496578378432000*x^495*y+251529223019363540517086060580262076503640836357484955135895268986824314100899693921234730877370738185703950906376256809654907908372740247545140980131426647167822071921585748971728745184939036103646726196307914290994310149522662769621861389128675776315623389511163982306437401538864300703174500640000*x^494*y+245079755762456783067930007744870741208675686707293033209333851833315998354722778692485122393335591052737182934417891250432987192773439215556804031922928528009672788026160473357069033769940599280476297319479506232250866299534902185785403404792043064102402276959595675067810801499406241710785410880000*x^493*y+237843148800966917426160420902010384676923451863573750732680293216190525962358917116919616810855209427164234619425236981227288752041940026121071629405519221080253315938773845206761877260985660325344123186030308213582041507225800743291739918430073288587567564057245409071714025864581254258695290480000*x^492*y+229899467996219495822536202522178996583588090995831601886991560436867856136504100631678686583380673945313955663570170127237379304527769141162214620171936064384056250376968038981781618099027396620961313570779787114110735602269339814733862553767379288772265700424685149829633203782660072878738866240000*x^491*y+221334585855183867546794657722333112397140691527359444169633051322553171300046104725792617867529237072841474962378340259752065173574773820216955644126314916887395331245277072823636812718867552433121578359319363672604649373949501664773189242940751432916044037075530212875195888347619795653844673184000*x^490*y+212238643970724256551720904665250929695888334341303576601017994418886602616482566175417578777082830069847989689951833125789651536304577635824478014915644440850927029961224590378829820415352447538609732673319937768251033646252946801837304753504830141152370994455987875359776879237443639668070234560000*x^489*y+202704486136101877839436567150991610588455850572065329995894139200850681014570263398006242230456062312804037028098528122092069533697145437340175291589355725734576792287185204482905824576381536809336248588385643688817881744175177707223519579030980349655291828689410295021349402240449101167356141991875*x^488*y+192826099872159291200087806568584612021766968965239534187127368284629887592807579996543949724098554402823333469224330845186997918994555503746599497652252620191956090908667406993485462754920448270869569807275232203008043452548707058723348059585026141582421856531056966803934714022103628400915784195000*x^487*y+182697102408057538549499536573736782207393995887298936087803557110145438244547259646530940691898824891390979376482974944758887133366436829425280068787251023411444778740313282501609767240556922778041790848527311445262484749788366415560837558400598698347547556674367203956257209589035928076354060122500*x^486*y+172409304408380512106906358786089468257851421361606374638199084962195500945339744054784538206335590091681584421302380239131687663720559804079002162001172810442644975665616029700548246366816824213841379325018006528927315705625526364975858356082895082324093422415033905092700978369459147660404025669000*x^485*y+162051381081520442581103844983049209505926239070502115696756891873381430152111968733663761686187521694700714039402431038718737435861378885616891566997226381908300025577177857373577324588965425859909048396577002260716566118659651719018006400581790920401521918355215976685968942847262958556775101646250*x^484*y+151707675906104244118480195303280111026824564236214746609729856221888998440275034559174585408345764990783647185823552461779243556976610020577515509529318314977983002667996292009306431530095292294382938924455065946202742749383503736953027268629761712716318391651691552642183691176161067585066052605000*x^483*y+141457157263799903299663965890896319741228309895929966433396757828518120167283478170041167475349429518433400754348988106253618992316028262430386083209769780182173340325564110116785726696980745517735443051181075003891746617668402133104849750479102137532783365188739420706901009339933968423912940942500*x^482*y+131372542969463494008551120538366138950427833082539968826391593975617984432814328473911064977106791961242580276678636353013958293441860544299510774772849776585371001997922738104606397433419497764062588729613252701109483371322099861573290134356314509230831564587615415762478394030535978382131093515000*x^481*y+121519602246753731957909786497988678529145745601349471164412224427446635600353253838367735103823782564149386755927738626537911421433721003477047466664886043341468176848078532746760917625913035431757894574892258748526272118472942371955293374279590921038519197243544259580292514478245780003471261501375*x^480*y+111956639306030309673314198692964617454875159095293178807903776823751218979212210829974112955538225778870836166689663225985023958326652747925110909787227064882734596712241258576670327179344063353634912468230871783671037652335916196811018847704805455083472580953745191168023813722760027642353561460000*x^479*y+102734157524115935504822799183774045518937167062539143005720132372752555346824998060455172616288908329653506750659671810817675241453001276352735873157244758771704735297248204709243461147329130931783760674870857441337982826568781337686739517338317649396519858767900280784450970829888990882542827470000*x^478*y+93894698463723550996759652026470351353827850584882811389549184080641914829411757309555587974352004171270317833298896989619213700601404608215311180438170161936663218875878856311698612673849568996926649335732829554415976656022710285686924453705001599257239947401637350315425552689649976370660557420000*x^477*y+85472845738923919514225866443943430526289856353032635558807177111576704911506504268431327220927301507053323676495369969557948349593263355188365330284364822984328922526324836757023355430202756510561091093787327666901566536112276347848593443544438478713174532272101175764232802734662287650391385285000*x^476*y+77495380136624353692898118909175377010502803093416256239985173914496212453099230536711070013640753366395013466689135439065873170297892108704117899457824106172458223090534518659701175590050499236242055925033843751324086992741797222049391388813624220699944909260038399359571074479427140803021522658400*x^475*y+69981569515012486699860468596688791026594736633788444323180527774497530257076301340185852198629957887904242199006348542882680144280415877632045631639670057855356760395444669892315700390254728397747103734583794262127264869871394829797454200924850769643486372430643041246760951288456068215656317990000*x^474*y+62943574857904969062113590350721986615950484182952035311551366537214097422873181850565643153985958328020134349770415956976072843242726994302826621247065668735178566275978697398401597694460609602527755541162653662710291363034233679931676074456127637212547515241223532354771709508023104998522001380000*x^473*y+56386952476873201451476758022521779676788975413894531633264765856254295607990558741131721992112421002184703688335997628124398588738276265729615514867162994908597465622230916419401431267954296102264447672291543906177969346051501004938793150033614341669573815736929414401149656434270698227842626236250*x^472*y+50311231699591968024758090333894669201218140633947483801325084091024626705050177175452122457990666754312249793751589566114775300348707745603740119125332577687822313371820401795760823361955440000508165030853702691334596467554458363574877820067799563833721816687770668425978521430956086131458827190000*x^471*y+44710547416052484791813321787291300365610838186017480887592669069570941845431383867241414486252083096756735194069808840830300314083474241847852068128361592624460961505900772161893109063171721207998765527419045221921877238147452621214655572173459612388081086150830160054029969045245880316824731333000*x^470*y+39574307505733837762998608738280435351858933987623758977718558310166370371662430164978276475590356036677336235805668842166179185723602436287176030170112897426547367057953602478511791449511692971298342368901979763283017517757632263598659357667657284034648042355725377072305245670933265063856165210000*x^469*y+34887876353739041185801405071905120639138797068036734888251886931330879143439247645441375314007287558649756944723418584541236913730017937253168342386810054310245705169511728500793289830490571435223538667321482159736344390654754758698818117943855763556860774182021056103216466578322746832610040382500*x^468*y+30633257286209889821679282502160593731926748645105425755538242183607601199117388176485110031811276880765640244147391927889866558397088920514977081120125901345581594782985907951916059363357574918732863220087155067085570684477345641784328103560458719220658240745189219993068116995600460633511254970000*x^467*y+26789758712846476679258848180728459312377886923715793685086814793529493932561461195540349035310611054901786505649498184128403900321049673933509919256739318217952443377629998152705617458217205031925556411574347221589815561144045720436856225398378692651774154359556864675585787709635608831179318485000*x^466*y+23334630953619547911279669630316751475828215526077681976169076063149054528175029751629537664401392059035948619874142343558572369251605884211244153969421536989842688998085194652637042496316294476406185584661020196749259909332944496679579441188120506122853749404772895212753228173252698533326284886000*x^465*y+20243663047449794363330310406897928052724104887362168132310858898067743200748859765872639951392998707932306172092306324169283865115665552534008454469740699067680691015129879689321314852214695767777754285200325357254488540745931326410456045060589618184938420658991410958825095336870344809695377746250*x^464*y+17491731199286228276695091301304727404960865303046640620842157409131159860609815514646005470104937431062551329331154812690032985872753847999590172949645594725146816817542391388910782293161301371040741132836035318000153971892201369561362392752539260405607871854324049692541609378599329593479804980000*x^463*y+15053292835073464111728303480490871354083421255224153545445945874400979582643763165950000990071721246434872240669748472631013517581942437962472583783802807356399583989818080321683442754151826272847329264875621472554035853133994858934778416067705720386238744737085566928711459372289014129704739230000*x^462*y+12902822430062969238624260126135032589214361075906417324667953606629411070837511285100000848632903925515604777716927262255154443641664946824976500386116692019771071991272640275728665217844422519583425084179104119332030731257709879086952928058033474616776066917501914510324108033390583539746919340000*x^461*y+11015187296775979294455155403978240784495963807394182197540604838252145377140912411909445168925497610486470004680561977591900367627421371270952160514814435224286044792549420679835027158196812558385109192234383331503826235388526396776083888582876725552469938609200708498628544080357516688561721881000*x^460*y+9365963320733734705082017533881683476650911924956236249295153836591472963927578021216903470805413864738957859802326265604943011291337949694340099513520591872775564888304498175090780947819840622656469923156776954698262603102998415003694249072316624314484605841464558057983604948178295151087600860000*x^459*y+7931692184901815921831450273158104641665624674455558004476892271209383930706196147119111979888717645599966158024479254451418528012406123449634881322335704187461225615741263214698650286068831818817933016105093398904986226613055853296486458162718322067063531515188620200395709725486785008024370470000*x^458*y+6690082911022157812520818094118622331275978086373196254236494770191340405641690304568238097217371421150616022790444748690146751067554336169305295848305253255722359727457640059543244624345349858229490462939471043643616375301619854161677344085681383990267214427175668603648683341202481645810610820000*x^457*y+5620161563119717132944878435684210304031474238001012294459702408046769421651199391889126489757975624018072651498590533366538722863735903730464191549035846944604997050456142476491291899496001627225877098462018873060905668222132855426262033542566897947706097414006030426226926998032231823778399163125*x^456*y+4702373711527689931418100122333944768143765600969654323437842748751058451876966830644847118036030980829800236850196849936039738763052425873562699718092378360990603036711928384000053405816838058743119186970056158010592632494114829494266949165890835714044000772085779585980694882757243507601743153000*x^455*y+3918644759606408276181750101944953973453138000808045269531535623959215376564139025537372598363359150691500197375164041613366448969210354894635583098410315300825502530593273653333377838180698382285932655808380131675493860411762357911889124304909029761703333976738149654983912402297702923001452627500*x^454*y+3252403511629450379134395879859248818917229711822399547289428104712036162632758898709263545990795346277771644622165401997199941191995431667576882498497775405072720564697159485581999156369354781641340814875693930129203313760402395780617298783233454317757428931332943223697799324758970981796452455000*x^453*y+2688574435708286536036279805796057874031943539152458019930859363931664930059561644370978807178522430408449917908468845081626958686083814863891109072663307041054639444904768698847893463203134518400597425435564507935272082360332637387992037132855391981649845448711356350976465500211339151010571098750*x^452*y+2213543979854545563366846033187282621243057340067233196737246689430077501615522519591406959644247975491109950627737555513470647224243869432566086158185454977334602967389718491583329408684547909502859811105419230576945320996120859925997087038343601412943042154494595757088091814381648991360251615000*x^451*y+1815106063480727361960813747213571749419307018855131221324542285332663551324728466064953706908283339902710159514744795521045930723879972934704190649712073081414374433259569163098330115121329285792345045106443769073095163216819105139317611371441753158613294566685568520812235287792952172915406324300*x^450*y+1482391521898960640439866036744296347075659089809090834112602592376948453894968801686441321431447373786242417026561085271271631262696892596400881655844705783369271315729230713964153451550994879503730073135934112673126721320451174796175907653627566100500875780414711133149738438306403770983544185000*x^449*y+1205785857486654578908514221917009166371324150949785841515504644886322202534132231806543756019420055851490661675590447983335076878534247782217383810641798725965222501381203968423740760410138950900678990648613073533032423682758292542541635029852857208559589176460516845623609707970245096325382860625*x^448*y+976839175685391051267657091173273248705882856465649289329016421173729379268157757412896207408137766765764586673896059125739809116787238709644462833937659727364230887194899417457207704636061934906879182297610591216633862224006718009147653695070669130984983636626241495188747105191084634997778520000*x^447*y+788171681464566425842315378618146466013591402238529300234784007697936881828278912569611199840140039249633159283811441207952517464266959753088582828104934834172944416202382742966375169625125785024142589326772444537608910494821304964059568956131027259116042753739945755143267068628907638707593860000*x^446*y+633377603483237163830040826781429412328039216934025347576060662041945674406148459470354225457121545054660160433477302304048329349663178468247762056459100785659699476804076942996402388563614594812193864576109027502294728073315859484631653611593582265884243366068496949178192995690978030384841192000*x^445*y+506929916456907442274043467477942605190606927222376402286595314044362994803482130331488543756149438038352106821757912815290479425539774133759449847345863038882313430175924891426976731853972112754363794489871433882232291353643088976009147225106374295536849456655541622993337919213102920002255990000*x^444*y+404087761053621013231014900467157121552297083818195911337968257514716642177281985398888533981562568202923402924345625296210005143518240063535360381008192440329887186711150182753281272788444556665956058803416367403431126321395927298650020409240987768794185204228115764109590729139349544849195080000*x^443*y+320808025352605929859031542843997499727002881956022918857921035983905864667627095934959893465649135687983991927392673846274251395302115319258359585639120521623906852532328908529934773916274083517954362096619087383010732903903935113444371041745085271641261730238450328854029915786257792774540180000*x^442*y+253662159581130270121094708295253836993444139221041377701611981940762776713937703762526427391443502637010598268170951413333129010238881880343819207249537156632856581072074020698087960771007414874661588634536022581915463226342646368769967800449602307809369740188542120489232956668203836147310840000*x^441*y+199758950670140087720362082782512396632337259636570084940019435778350686662225941712989561570761758326645846136184624237999839095563119480770757625709010510848374557594258291299744269107168339213796001049697117783258427290744834015406349642854061817399878670398476919885270953376210520966007286500*x^440*y+156673686800109872721852613947068546378303733048290262698054459434000538558608581735678087506479810452271251871517352343529285565147544690800594216242361184979117300073928071607642564005622226834349804744860484535888962580976340404240274229689460248941081310116452486184526237942125898796868460000*x^439*y+122383894137452373887710493812745715053514837737009653602216917600580491863397094985698719600257360833713664718142558147347609187010270674842457047918855089334221520876253422483549974374498501032525915094294933649920381802577603981248185741696927133959314404165698650240226011488600123793283370000*x^438*y+95211626344945186079604256287713362688169625095577670120374795575584823154827580113207884875511055142391803102213926231861905548686498322523971912945752271986481396347777973441909216298455316966689788297160179287149426695433375743848499742208266580238329856171538203916907625278875407142909620000*x^437*y+73772128923299727512033794322217623217606606678665677025893236997394623614644773952964265408862289179477336800828875466885909086482269090324425046023570466060447464900671940415096325394370520415679853698331557355468616074298555319258500686781937048872606643877592544524270624551185377520304085000*x^436*y+56928580903643683531410149246879440217480496481235814483698143948431957338026763616800742864183996605755962557807769386835851967621715616604335079763321633986469194153438877913242474109638136108383037544199219481388170988308265697693285485729070005855675215452443096305454853635958981590889524000*x^435*y+43752531259867495293574938025428545043469992878688302650898750207717140356964032108318592130600774776508557796195017108257236052854145394386723957061916803505501942503084649986325929748573479164570002352873958435342498904441864979675934958113331188245969467706382945040411415780286496452362090000*x^434*y+33489591828540551953106742686130491267841229117020676103157068060227934594219382601429045828361086865969513374865321737184551052801938450024405991825170886633840993027052448137681575856932786521028890689854141024583147309572785539998863795099093008287038357997478303611179108375034108395635180000*x^433*y+25529917714362779921998626026574828730590232759982311184272905757180802252283437792990804302254138403107041005839232944015687686378942515599591187430103862521924559825200193738760778778260381273953362092793737788106518987755310103555471871968146606669520438403007227928944637194348184745264142500*x^432*y+19382995523031143982958535050053297032715255803712404976460272912305986947252100398193369874470628805170899322535234853804529139746402753495647439314244057310143075297867282416774440126904191055092886509818795649318130408981184472295191298225376685555769471687344679200885910839997215834717240000*x^431*y+14656265035835829923956365976443808808947851318245695692726978289831369077659044336177793712099721078997644926338046003490793086369648397818638677797261734562581869216457541616894357359115274289026375589003334955887919660124369311507416578131819914867608144381132555676459346617611929868005492000*x^430*y+11037117277424530415588857040054006633708539521621101835153416225267055522580366137576972497728336364218891976051418181262768874148771998357293575223857348269545015346894470919902931111067544560913032405028781140160780129340593351923273430116782072492244311880712782733585847715539631949636360000*x^429*y+8277837958068397811691642780040504975281404641215826376365062168950291641935274603182729373296252273164168982038563635947076655611578998767970181417893011202158761510170853189927198333300658420684774303771585855120585097005445013942455072587586554369183233910534587050189385786654723962227270000*x^428*y+6183097113531019656900563891548579632496406957138523017599034220437565135686383124192335378308544455347756237892679295262388845727322533111153992402893907145766055717893761195966563502011661089097876792345966397891117664080855961548640089123014040610838436498619202892637097934883458736183720000*x^427*y+4599621023480392671596760943712967775393668590066462244799281554227944796059382567996737293619770875319672323310407768426899019382520420972931628494835711413313773155994139426267809434423308858963054686989072564284855823279661142127646895567120200942209080809948431420132475293023060767161060000*x^426*y+3407719227830690918435165499168216125769917946727500723973032942784529535863125172115843629707865031106400712574319494521493882186006433625163258676173935760124638894701745035808846641851007954640454428969295499800606227334148950515439265237553401045880119000066142234741625173613606759670628800*x^425*y+2514376166368131320025946765879326134465651262776367721681491320630946272121229510675752678169865691354549136882093377034088367932383219254677751627385282461897520017792086180935346914560205522087140854708247547595933414265648097168509874524236450424477518359423802864175678296503095265381974375*x^424*y+1847652503535680554057194850490180729659334723426654963592638336130885995458234510444573891757405638014434721036408304787614329295200147251270999462757989192104936720179973207481086814165558303925386000686823154559230099910978844366461328939820199272059736194793227754610897049770038808530255000*x^423*y+1352174756047738536965732563594024997657263993095977594463124595472949439582756397782101654348412776609179735291350714403392493584549588732331544589527040533322470990719945790249999519709396475018059305000910370897152823983294206171303014085716166595296312128715459066090673792478765425619892500*x^422*y+985522879191961420724592645659203020745017970788432719971396509999282665809884628435314159127858707649523054046545771119571040229153586260870314018619017452611541896517818866123488423691477223588291928688055572570981850986096986190483371233458069608316137682759799181157624076728910206583065000*x^421*y+715353676103130617456988799694007709885607871899879612255099880533962072941312807881495277573842268828360699575165120071274841269782172096252417589376907495774929549024140935582739010989848122639087762030467924228247171146804881355506033257389391905346713731796336991840275407418743443054259250*x^420*y+517123139351660687318305156405306778230559907397503334160313166651056920198539379191442369330488387104839059933854303665981812966107594286447530787501378912608382806523475375120052297101095028413798382190699704261383497214557745558197132475221247160491600288045544813378512342712344657629585000*x^419*y+372293119223961903756649244903476872987293129208855493149778723070090806809601374366347685136554354290253550021108167072244638544328319597975112371070580351173388996449031240851034213892369101211995742504988275061030387169930748090867007744188492371556667561324885355336076755320399332554632500*x^418*y+266927142085104761184012666157209833462587526602575636597954556163461333184242494828702113871491801189238394354756799032930118201593889523076495662277019874426203431416286550044137738262453317850110155003576499100361409669006951461376345175078541700361384289251804594391904088720286313907095000*x^417*y+190596949057343639407077537307459761222429792111770617228333989589320849208611507437617776514404248451904812407420522597143594674768239608087155293098488506225559641953067622206173693245621632779958792185772945419264910671191607464715643729465328577141604877770552253187369871569108549485031875*x^416*y+135535608218555476911699582085304719091505629946147994473481948152405937215012627511194863299131910010243422156387927180191000657612970387973088208425591826649286856499959198013279070752442049976859585554327427853699492032847365308242235540953122543745141246414614935599907464226921635189356000*x^415*y+95985115035325124434053458302732864203028731105207197451356669766635604000392901735743802507064407259814027636349811910886118213155943192847835505966929365289170726019595677773909239526046844266888614343081710852022677804832178503277351108354173815109613681334582249614268937976403547105090000*x^414*y+67696486583687566466265982516748561805883977304183611149679150397593083570975572944800569400212375818676332949657277906485269063452402865143107154123183572793384464347721653489605494316496411459100317441287612083027919269506851619006513388174834683910357860430182370256060204978247135436980000*x^413*y+47548722719494838351305868672478156506513745963652774497988927064976094412947128615990876126339644920260757667021183291459891365996330583850515739205569414223924802339471161379603859103253431858177603917094870391650562344058383875254574879789467218460846592445009045775089905877578345128355000*x^412*y+33259887538933571138774223927098472802518953033998205591122984636282089810075071289963057663925184562219748996286464373652759325620523260690004218255848215042881186016743834445495398897352145883818629565098619017589187921480567328700992954963090821741712726803639604175444891717423222738340000*x^411*y+23169175895765589386502044125487241223449643554191970335511096077138879511764159830804774067581781110292062436396164165375057767508534000243375819835853587089193504157426637215421371096291071115677045341111071891913824128353412156095098482186153097857362594434399792061199746603154143297386000*x^410*y+16073370756791694836659624520219575129635116509676324598239508276864535701900686177038844953821540194957268356890739945522459703347713942639228572136548173784381280379940645107145113620100404665698119441379931430938524353003213170895076781212052064503415674649922021565299316594404735620860000*x^409*y+11104744323526694574651666264813861871656693669691920203851281900739180915671251091906904706271976249556626280351879455605888545049349666451764334465959802496303958911141425420308026132805853899105626438385797221712595372260665856243389195127921105374826031979422477736836858930931650116438750*x^408*y+7640363716692902843942461780850009517092632406803209853577273213324765284306695523605425160470432225664592786481562930669818762867006178604249322870339965292566636147969142616333346816500486325185658662498153906338514185299075327735755129194252632365816224363582413012865832114367813233570000*x^407*y+5235064028104396393071686775767599113563470352809606751525168683204005842950883969877791313655666524992406168515144971014505448631096826080689350855603309552314176619904782903783959855750333222812395750230216565454167126964181243078202588521987914769170375952824986694000662745029797956335000*x^406*y+3572161336824176362331268623464949983372720946623025783393644513245086339895897297093081131670925393524230091457393039045191953183571951913882145289705787694520261693582087157876113783923756787330811217804147774074608157222617789394538236873827047724845668297221755626494569873079156252558000*x^405*y+2427391512439247360308999651851182455144214737218666849453734946080973100097044304232714527393833530834418099060812383914937484965346712290473605440152422846108567090437492112315144433706579696088890173172281625000362925629463430712731520023322071021078012852977870853574330198988352822627500*x^404*y+1642656902890210943994699931905992817216520525689014082377401873059820992360478892646594085539545638956624643250533003520326204231155899104441099828846865711604457461535589302136211643580331988643068056887105153266577256204863025138933892947105723103041067324293232537427184925278550318830000*x^403*y+1107007912817316070952949954110560376819829049920857316384770827496835886155974905914009057646215539296855737842750502372393746329692018961688567275962018196950830028426158007961360020673701992346415429641309994592693368311972908245803275681745161221614632327241091492613972449644240432255000*x^402*y+742933524127814792192130019286219151054376090264081203984437183061315569673959786606730619655723951247639410038039569204845218738791638768946250492381855284097218149294349781636839279316908515731651089675804036437834280569971801527233584013458355277277265768866308480852782846004982727490000*x^401*y+496527238625422886115073562889623132621341353659827604662932184012645905732096457382164964136575507417172339042089778751904887857092411910579077412408539948204974129778390437393954251676800524680653478266662364352619244180931154020701111982328000776980305955525649501369943202079996789539150*x^400*y+330467380116754000742145466149499589099062464998221367496127909492609587841661535695284501921181702107935001026349270383963319705219575314861282803599693809121447008171973668814611814759933793464661216816414219203074372167009087534576447242813977222615844229967154410229579502216304019660000*x^399*y+219030705426220674910491762447924146263332098894169976131154544663706354732264041100362983831480895583166221610487307114952432827878090615663873486106773803720028830997703478167824109782746816598670806494600122029944642017668813831056482474888333740570966524513113969570767809608480571170000*x^398*y+144567530281319782113392572892659718429197637412735738806301009579030064980830992301732118681474952640298766502444358593285353674121857487618941372256212228657664137209097818094185730835046820905922024850498919681455999209008603490481724751252996399249161984670347197162795337021849531220000*x^397*y+95022035631927075329498098407923689099985864325920676003479305965024728141374013151966309795605225493706308446143063512473982464613207653285959809247874594001809043827834161893032674075353622350415635539152435618440449811219231102187491268621588692883969052837959995486141968208069973335000*x^396*y+62196241504534085670216937139731869229081656649693533384095545722561640237990263154014311866214329414062310982930005208164788522292281373059900966053154279710275010505491451420894113940231461902090234171081594222979203512798042175977267012188676235342234289130301087954565651918009437092000*x^395*y+40540454446024692804844373218142059976051574878925652948379109835663115336643818392468074566261815377152826465771208015222923211725166901581948649490092311032274965593513404804048143574903345629250235144516880722898985788044928481041287903984368173201621360076681402214609624600022652890000*x^394*y+26314561864470723171513481133357449144257529657819946065339982331550687714394834343710743622911293671496233323746055943983248345007768960829139650575117579154392646530221221569678696158998217756053694640757250419805931467034105142553982593360528929557559828451750366511624698669536944380000*x^393*y+17009248047264793102639470535212956436995409795268484874471403053124046499600608383352503690467332915950690289855592082212856249322455923693835333348719093104730773168383125126453499326457729569291286174042104300960083991026979146420584143405736627164672718061739957301099517396592136745625*x^392*y+10948481501687682916641498275539374258295895960172817850234466333045363264110736430663680536392766014864812140596702949470344252437442893412123892730209990963964635602637413874498804164156699492877149721222503917859364408017365887351180598054267254266915772545487788607604287059875398365000*x^391*y+7017797159278498394109550533993271040973271017094379966297830059378257436503767121949998507753395921003510732743132546299843610988590444793672855831987059781819954951854473483490217095385687707729451706554096773578707350057032888449691170228227043308793552566042172697661108590838165181500*x^390*y+4479444995284147911133755659995704919770172989634710616785848974071228150959851354436169260268125055959687701750935667850964007013993900932131610105523655179885077628843280946908649209820651728337947897800487302284281287270446524542356066103123644665187373978324791083613473568620105435000*x^389*y+2847228926741067871619331620487466035605551132300494166551789625675993056737552576594231768373040272497252477093323488225531043673927495853920255442890035727083815682385680209718079318006917520136375379484296667628407550242162905305517172735482185906467138035242391718179152317309184663750*x^388*y+1802161213010659599002121808726161210138587013593134970019729812010253354019853833146104284059934136588799284032968211144381802521180861976053930035630234685332007316094035760800350367678114188928080990603437368743592380903685493080816742286080078518285888348571040761261845186159810195000*x^387*y+1135889885073493916634887850125446886520575202378734907813738497146527765481569109816844231158297248957435379349769865981882341328496732222691972188581271699061053471218879217312272951614707151653367008735391305706466207507697533912501757760118876851102017574750802564508687438182160497500*x^386*y+712932513233119759058645057151906501133239070110880771408297658371641817033960449413498980857077622922878140534977509380498510167154046565787156528117676220874092097382906305500060746867117009005202707921725274801131635931660566000366956903099002381342079323339528113659111140062299109000*x^385*y+445582820770699849411653160719941563208274418819300482130186036482276135646225280883436863035673514326798837834360943362811568854471279103616972830073547638046307560864316440937537966791948130628251692451078296750707272457287853750229348064436876488338799577087205071036944462538936943125*x^384*y+277315726379171381157333571663626515837888779297587334097230855768547870483226106741069295633223062401119536026571478527260360518828154255735684873173812468411316212920417363565663823740855886809155024151076281932368869730305568622509026996343210002466935231120724063659945986409970480000*x^383*y+171863953403272878613687310594124523569435926328440046859610708348468987694297085569303463151334033818169550644299152550065886858755959676289267486125518083174003413508931796513995541250400978394670508494922679579445432211500053045988604109384222380169637853590998893821617010995175880000*x^382*y+106061438126090855622663251449039689827987922225305489338241180273207032793572676393334285821986431209274262271603031137520466526728233596676090758804439269422405983780956294456132951143219989897841897003328697252581833771878869569576166025500440951897902520309792532213017283037410640000*x^381*y+65176464396839703213281772261426002942682900593292566835274015619503031442501918880419940158349726275376603105614120747411770559166866129570307385652405422016026902936361851915784926428333574437222197997206828472957546237235240816142772993089787101085646548771017668988967072318150732000*x^380*y+39882538600320591338240053879777586341738328865460829947510669783270776083978629910723956940697094983322237005045677752039408715754282011653328190898412335533156558962669088128821694110735520589604565602155547213726034734540082947076092974837550883111989836607063952038337338938642960000*x^379*y+24301418214664797616065884920314638623020621607732563585380295575337016295543248771968456078013181669902134766740051234763562545451564119640854315676042243033868707149279074599394569241107334893022717625750727321546892547251915493475625783703266534886566154459931250518536738678047720000*x^378*y+14744680714515719901882671749404387479136107492332117231579055742339313257970061052655018294524851799715901993302952434575644690498701825400068910634902035099201238045630000318733783584492090833968839907758868262511597725298915018513301037303105538021062610571194241887651504366455920000*x^377*y+8908244598353247440720780848598484101978064943283987494079012844330001760023578552645740219608764628995024120953867095889452000509632352845874966841919979539100747985901458525901660915630638212189507444270982908600756959034761157018452710037292929221058660553429854473789450554733785000*x^376*y+5359199950369313660337621758516848035750003869879646876437934127148929058830184857271677316116632800803406511165846444887094323506594823472078380052099059690723009988318317449182439206843391948453207678473423317814215386555312312062301150358435426219388890188943400451431733453727845056*x^375*y+3210383356850627192694262235533255612470050241541322010645727312589214691791244922487027146236002077158590162439604499732684299225196579555957496037599276971279758379583656618919192815601073451549445494293184894856758418463645554350420018185963713789570022078041174391832108698319396000*x^374*y+1914965511103882886870261684353170014455819442322893830911486467158478938963198725694016192491650361813895886367483385805460810064152345700044822197866235386377399735190251316548290451411166620222476259753829586405785723294104365752882116110925724014831241239533332093373538521804552000*x^373*y+1137391935735268020386317847553714037248440528640827068359847853901453255148524083891509617514945198975450900660941565136045990690969466474708150764019276750189124365009496402981707545185294823794559944089456107849296297434237147174880619919387412511993714940041294380299888325848882000*x^372*y+672670588383974091548028997281369828070619835698549553942549128221527203362879108438221904158282375228724539023641116423861857769539970633690671040087712163863838257207524104784094128472066255089946421623652896852048048721043273050962783163771251914883405338148428472927755893824776000*x^371*y+396128235381673631689394853954584454308253903244701403988390042174899353091473252746952899115432954301360006313921990782940871797617982706506728501384986052053149195911097528372855431211327905775190670511706705923983850913503260796678083418665292794320227588020741211835234026363479200*x^370*y+232278046103358547900279074426618459737011005072170395365616981940907703080578610960970796628700781444537563131776761631835376489886931222515831292412749349064445645779882861327981790092220800533788507273108528037835221613306190958432473636935274697144982896303762675719550855395384000*x^369*y+135618036411612823062030029214275651333792817834858980838469408759802124108755549754111113854415487900370824043711432028714009374633350666310667321044785616779715891286039202262698228708907397780012593645216846275255058188781620986806301854476449941845725773316595612880560546900153000*x^368*y+78842713111332573280927410348899588769092823006679470692822657857199339134316022605865544863230489016329325826359884654923784280987477164616628079217189742456454104254758967508172114004546480857890417790584201310100886909117909199280756844308583852447436152575840735450310080978288000*x^367*y+45639236138578950148423280123101181511446476409229283508305860305350406091946341161439518871933106417969814792230406416966922446565306182041486601061054630097032580854095490655361460314934634818368743421363409906635686901650272359836021706405757529729036384850683832666031229840744000*x^366*y+26305449490897473628854992952842570760928205300437665770141645467335824613625765141869077019098451888152680651899730312771486008571499311223911962186371645063801456051336928472224085787820592666965291483809461457997891977959054619999974715818121662804452467488740602764988078931830400*x^365*y+15096680918518204205239107590860909320344017192861239003304560684870402490524220560978322503098954306880076160288367239247786781648737812259006078927713286868376621790468520271009105837349868433085426716337190931083695867853859962735834546027695608370479796593381006303806051588236000*x^364*y+8626674810581830974422347194777662468768009824492136573316891819925944280299554606273327144627973889645757805879066993855878160942135892719432045101550449639072355308839154440576631907057067676048815266478394817762111924487919978706191169158683204783131312339074860745032029478992000*x^363*y+4908280495675869347516163059097635542574902141521388050335472932026830366377332793224479237460743764798448406793262255124896195018801456202435473937089048932575650434339518905845325050566952298441567306789431534243960232898299298229384630728216306169712643227404662148035465048392000*x^362*y+2780590828536251492646089244747017318328817801613055515213523006875919550279490565175683073491062977867039629513553891009722101090463422762569079131809445561177442030095314309727711530994110691761889460184310196238362448058191464724627912869505951226034392563881827382768135129136000*x^361*y+1568427014221229357570684714615114456119848791222364126550127821065948371329525146919408733641052710953127041022488991647671372646339524402011621197786265386851650895100638227830787285451365562071940773635212470065701318357823623071235432102955700613435024555564593258092651221278275*x^360*y+880863845740472026092740245337661785028308213479018854224720773141562267829374497489839538394350976510336559700617842423029164044746066746839600048678713789807479441866193076472829052671593763410138344007295615013498400325766777387901334722408817817217798502345169380520053728019000*x^359*y+492570281340855852597030760243334238045424686353532349948091522675733417680288854515346408541389408983194431359068232756802912604460806794572299715694171729814462803161936626874993193004832026579812563081961255124370600805218493897595917703029229900905279847884604061692678019250500*x^358*y+274245973126013056344847608346988580435866310598078664512312231909661840636925987428451033060369219931545266604271271114985136411192797562141342609982136048636979290096381512319203053026018453367920525012973762573133242750028337193373776885978949151670435747344771779293901603253000*x^357*y+152027659015507237756382913322787147850317193918500129240520911167312542092208971291858724848682937135965310834976465509393934314900355170317483403359662374787890693422994099220427779394858055671347247561539803165541254133167882574587637186792678334078176338201993051565097627890250*x^356*y+83909839704683064560112119601414301759244838813931854278489061047384906953219215162638303947490117240935892491862979412936807156751203783927169134257426054921688506757497518329414402270650337703875380049470030894469281351019792552795657113950687576638497327751797715282441481440200*x^355*y+46111444419756173248978022381582162731473556933352644379045846241209972087295389137363154646066550496179940920451018098440505480877209509743258579971186144732506841948779595985978502795790820255225634547309382302688227367820474235669440054880021810691434290018402769234159018438500*x^354*y+25229445633065974235144080252055773735613043515311956893635594388544559689184803330180149528141512945359658556166399392346118918439771509194920459520556252295683805332098882502374636769258037666692232812592768678750591171883227016115891467430491067982639472436653138035382214107000*x^353*y+13743818377272050779330031371876061926962043766828890097921859288821341929447894406718507381842521712518455972726449051694722188594505158558344015757340057191938863089862508523670133919055690272133268800687110098146541178510461630692761864202103930552271194089719996491496792561375*x^352*y+7454274374113654659975610235593796299369244076924143781923720292242083758344620695169359935914588047467637137749938468715781526017358730065542517020930200510882095235179665639956682803555628622173976298677754629503208774785335121731667451770632640299536918828322709961489785796000*x^351*y+4025308162021373516386829527220650001659391801539037642238808957810725229506095175391454365393877545632524054384966773106522024049373714235392959191302308275876331426997019445576608713920039455973947201285987499931732738384080965735100423956141625761749936167294263379204484329840*x^350*y+2164144173129770707734854584527231183687845054590880452816563955812217865325857621178201271717138465393830136766111168336839797876007373244834924296399090470901253455374741637406778878451634116115025377035477150500931579776387615986613131159215927828897815143706593214626066844000*x^349*y+1158414595739708553680159892638042458753156325233462082872669970212368151838534217471153748204419209236881468913148462806069155611543824022158571440863930328749290576573289618795959859784693721662797326051198658780406627825090917146208562537678464436020456265573007717644934553000*x^348*y+617348054084867651884679391482448354741345177919211033445159494079485630688835999509895106240639946117051686342688614175363041581649694884703189680582921522825043063778720960705963294341613193167922617864957325046832322332513995661379142056833239852580580061898019426861312748000*x^347*y+327553172427292163920464447774326573540132686143679248632217652057464088454168336131396944748474099851096231132894417612921980778031260129957196971196137260581483093472807604533592145468715256925487994494098152586010421788046416658254682406301428484473182387581976668380543614000*x^346*y+173028088030294791933558318976972510602879251001088580193507339865469579549835487483150141806064180990044726674780867929879397479692848862542275041273074033833882672277238826211637988293397677704150910068638108083602451814754290326345221545924113367370566574203609049251401664800*x^345*y+90998003613493450026032957388804140484745947553926158790792732093882629488861651191595730065689241526776571193291767432634744101362854965818726965303674606208368173682389321711913271282350912816969609715975834281772630908674131345410215599609480353266532725762568783523984107250*x^344*y+47645834464294896208455612392311452552134864472679754374478995190708713157029540349937490323587669840504019011403908670968572253986030606151662216232060981028430215748465641809586248586192867593664453184620528147533919379884172272178255960830534614191304806183141037339802942000*x^343*y+24836658390962233129939627736417672075049025097460723024994369833454541964834547629222734317614849172177626931476505583802766387716122337249270729737989234791415750762498047326273682773653728852016576660068147651374064357599196184433346192347831873355041867052913944996280257000*x^342*y+12889434248420460895962598916319945143652149595343804665475075088075043022721419255226365912935172104529208513755637192201132176933101425400987237587848737934240040608155284044894688176918930603019224913115791345629635827464226244425196354754110016217639330094228481318251666000*x^341*y+6659541028350571462914009440098638324220277290927632410495455462172105561739399948533622388349838920673424398773745882637251624748769069790510072753721847932690687647546896756528922224741447478226599538443158861908645177523183559619684783289623508379113653882018048681096694100*x^340*y+3425482525929189557323393660565108971611035217723744356381928679483382588489252621030910154370567674779068525844286838270295843289835830149430294608570995910915028442005968074462683141319352711947116252754423620346353041388626944433725909710245072388651501240372369972122354000*x^339*y+1754136822190325166061375303522011996036466674937083590352679489946928546069269846721266680259248401435202764745035102981314638784372124502502824580522005458912680727855019905200679131279849802643613911908987322201531240227710776681318857087270512899928789910100050484213713000*x^338*y+894265830920557927795995252775927684253884971536552418611169936051375337211784627740253601700793302692456311438645346617925109968503436020883792923403375331994699978906480735984659949279923428798705131561444517200780632272950592033613534985667320301924481130639241423324638000*x^337*y+453866844909981960342244578592601851797528968987677959445729319953785374458390692091062445441517082842406290594613677425061388643653099305779876830100809468196105260378740975944021691125503306483680164662962051651601013668651128788144218810496817683356250212387687288645185250*x^336*y+229322195322938253646607787078367251434540952751668863719947445871386294463186876003905235591503368173005283668857437014767859525214197543972990398366724783930663710507153335213821486042359565381227872671812405045019459537844780861378131609514181566327368528364305156368093600*x^335*y+115349752902679151609029442449328872718575404161875479498772363914285898866618023215177558443173616123058213256857719819740590001421555821668095770950229433358517031561405956901847143880165847451518524542127861396518797214981984367209720854635511748828331016519582923998966000*x^334*y+57761345531476516697774863235496017223394580195002114171799054793660405129610824218694609475292335509897216233568933163108481350037180876217607177657236327948642711456536116349650593787069554795812874358426845137087373717847050642650744775784499136594696491030795647099932000*x^333*y+28794203685601317455627289606916427747590411983436682663486654560312746868503599498241474483940640306580498511644393328316054325692187472725244296646496552704937160052434920276098275046548146327852825091850508129715711748567466862279488039425506306116817262744393638449517000*x^332*y+14289500184782716584855396337064654726756377845292942667081568481351019372710306477453168204287432857675224971399011337819028454603596772712677289217693356499311116797321963425507664148660664545361940105372748429096586398392225707439148025544496403035550569852225243595276000*x^331*y+7059439643526983865055427145624478678442329950435767198214924130339085690100166334383580112864388471478357411243392168385221520110135122041636093628442538807868626358079955065437368407771164126141495783400566761240253877414666730093071636500340760305622744210576948701546800*x^330*y+3471855562390319933633816628995645251692949155952016654859798752625779847590245738221432842392322199087716759627897787730436813168918912479493160800873379741574734274465551671526574626772703668594178254131426276019796988892459047586756542541151193592929218464218171492564000*x^329*y+1699762619086927467508222724612451321141339690934841487275109805973038050382724476004243162421241076636694663567824958576359689780616550901418526642094258831812630321873759672518218827690802837749233103585177447634692275811933075381016223952438605196538263206440146459901125*x^328*y+828413282407893327403710332351982218921782197067797931391138211529207251895295138379482551670381980886828899926072193778671587292781914852400114024675953784300955045430301890915268611415428426124440502193073109694173947201060993647805826829717477718372288754401735570353000*x^327*y+401915642948636673681028603381451313927926970980964278286205037344288978293414703635149546581921228115716691803895559889652238938782917146490856507520826242531768990883840828381740112660007559855626178363701642240348487737013271398861283936673019605204359677580663993331500*x^326*y+194110369779637860177800481040523153097043248207102747735263469887760306553560286496383336571416770912183172634177707443002414657841823688527435883632280526022750653375010533411033002558759206685828346883802570919042380744098261445968560834600599098217216673913031795297880*x^325*y+93322293163287432777788692807943823604347715484184013334261283599884762766134753123261219505488832169318832997200820886058853200885492157945882636361673329818630121814908910293765866614788080137417474463366620634154990742354933387484885016634903412604431093227419132354750*x^324*y+44662367776817028389960910590507826953927119375000916278139816671141304484826676531664158227146797079555837357596847809576171989788625493610732605880623572911722539834609286462599912530563276166208658384240450643229271788069421591647123700723351116224277214484023336607000*x^323*y+21277204707834661017636245015831899861531651265671527961414691422977347121827457993698411662785273534950642280979029266214017039383076746956145474482951938127561032989054276589114707591994009147028608640279742710564977562752836539973482235005372286932804631678966869799500*x^322*y+10090220789282416565064611038435746326087174827019487486856451602649051212413021316599246767918789511419892215722013878823142101150737426391574142538307104679049562036046357970095634528162107430549649458276991388515350184398252379781238998043784795865041371724046144441000*x^321*y+4763177754940670172626088446085109662755857528637140416589589653603449175271440945041703253679310931126155001833480081032689138925568696870140146698230265591139572667016001335883380416970641890009466876627815052519746189987998549867320174076551352165703353416792371125825*x^320*y+2238203937710740756593756685385073556654734815218626921158103801986936470024759328066585963549749629897752717454792402247372282608196744491108438977141975020799799197423084328168401958047878714835579149076212653166400559172040434592573356394267889417070591913911246344000*x^319*y+1046901841832443257116434578647856986177214671634519043767500165445502542430935814740822466821657084952174658164338381696351551542543638552292656940921246380696680269762410411562639625538523915003738634245325273255251874451438267793300440894093045049920115572635905548000*x^318*y+487430140121108280765777739399734292246492336134373434726303151700834273049835415940822173425017500753721143918388880496983592079837301697846361503972117641378542204662439986642634554789532364525898807745261254604934254869044464360570337048787098573754900076278503608000*x^317*y+225899640962560416670689390920637091582073202565199384222570320305796000814031910604152966338787350495511114944633443154303799253374889821955111983566025281165201577307008005505431511503335905781739652127555289049362805253636104096930989538692266444269449304357142754000*x^316*y+104210637290757798055383719023242804291876105124967891115813461630119031032458516424689543595703361688440163974458639469722628560680971071150095455192502173501027296976663547065279354211757877703693036601908717284085615270290523933766704663104753571371016029455265854400*x^315*y+47851823245756131760125177102509450950351272761464847961342916054646493841435033052153361855169910979385789580108558940178758012557588757160758117180230589872920697591325098142220111627848005068022312725366247732488292726153812010403078671833815415425466544137622076000*x^314*y+21871139008977329508994622431132412807001891771615665589318305154525471712096943782206922303527441117215630171985571335103537141110746535296183477139144694643518339219324717345934665285508404063113546136484718759827254608460403160504463905321423639655890094409335272000*x^313*y+9950096671235325779528076774628554082255221111214685071884635920590803264369685179986579478203617833849552679987621842859603379604162304575153238872895769510786686301814878676275509061575771034526947588255402575328387634372247368078338956926752324436473255160060959500*x^312*y+4505704153012222994503280048888401848568402012625517768400589850833571289525895175842979386356355245516778572069866494879820398311318779430258070433009405061488310778180322419445513537317330279785787587134521920903420815564791261016606320117774637480667134412103076000*x^311*y+2030831871864929494623942166962743441890975399893530472424033976245276334844280289401690708922936929501040776686563014358875570833072667250449652035747717353801253118860985902097905376964767705816492666085270025218788222667608814748064587763228858342735476524875444400*x^310*y+911082315887305561987586210938423251789004882730816854488350987895854795660965108125215802845311791816675312261699760421492658405575292109463664444402015021242240907159053009624241196612269158904649387100482934613349275002834634691606399720117143395438491639235004000*x^309*y+406827219088406674355728524826550267056072989543095965371243432456386028698321124870941738553759167155133918336510442153527791108847926678936809701329801505150075780797900838112558569007501690898174365049204084964631395918895812311714418372133233105766609705958983000*x^308*y+180812097372625188602546011034022340913810217574709317942774858869504901643698277720418550468337407624503963705115752068234573826154634079527470978368800668955589235910178150272248252892222973732521940021868482206502842630620361027428630387614770269229604313759548000*x^307*y+79984602151867338474036924189401813631901638033769107504945074456683004041232523429637600855590178877122070399813452283786763925979067236909126210892250440013495526548162380595936907259239845728939820730134904953020709924496326852191051194521231228607332167614094000*x^306*y+35216242098520008018784602592743820102678994587529995534551356523374099621031873625135404117713085951653746104090527192573740663812366294236248374867667100207380764206816817931448480030686896105115949846649325058452283794094785635640952036724455763962364954373975200*x^305*y+15432404942598566732369689354578829211662490444248058387985867441995833885653335424807899792963349447204587014005187922033032905837315689284562865423331128682832087763044726248695095415746412804684432044867879515557394478734065544354152832185285931046726021672503500*x^304*y+6730919802797653208953207408596792080839881054593127331345342471114395267200306985855956294204961595337438238533109222809242472560321333633439183771438541061091757073121372711052093265978349343793496903357009143083856415401945373721180001412233748978772898979112000*x^303*y+2921874928721617367210346482528406877499260687022518024924983909380604249228786556897356385593271294251065596383283803024642505996815707866664860576999825131104301422859277838751839913454784887062220002460134341481960592932363106357474986286399464098235226920732000*x^302*y+1262383731722358290268275590448610696716418780373105069423955852121520004673953562493564561443730945441805164674895148087899909600913224285740755213524960214010728225613021326613813524840264715154206639117254035947857080208259882861169450441334246291369153834136000*x^301*y+542825004640614064815358503892902599588060075560435179852301016412253602009800031872232761420804306539976220810204913677796961128392686442868524741815732892024613137013599170443939815681313827516308854820419235457578544489551749630302863689773725905288736148678480*x^300*y+232307419960319856554361699240899828639683341894622758852625256667155607136861639888259384345565323768891392643454313984791852123420550546163420003630128199154613325398116620732071247795141438309404645429566006615226195929908024092854292591914575993704166682744000*x^299*y+98945752946062161125005908935938815902087349325487471363155201913788499336070698470925293332370415679342630199989800400929862941456901158551086297842462010751039008965864486608104420357189871872524200831111447262040787155331195446956457955815467552873996920428000*x^298*y+41942865402455937432790555992759270467741152345654717590640469659045480515148034344716553930364699676307402275386857068957466794529383421405723636923262701570141713615686510681671575058951040992904995515890769963141044910794731498140859844712673301218280344648000*x^297*y+17694646341661098604458515809445317228578298645823083983551448137409812092328076989177296189372607675942185334928830325966431303942083630905539659327001452224903535431617746693830195727994970418881794983266418578200128321741527350778175246988159048951462020398375*x^296*y+7429241584583950619744284651908955886041384963352670722171955530033055857204412466377985350431619676707640934948842236150444916265045042195801048455024723203647441826608302157976933241824838643956044418506184254109557423029066802596226770366659685800897529131800*x^295*y+3104286497807741406267087779480371085527207598001470060964202381529393028151985379010631272489132867746110588965876005190341714303382843410426783702878602471779030224999219740231154824841823512701180033228504752071273994041890519498423367221196327636352366988500*x^294*y+1290891414929951871913046403348273124674680387287740025351450495289450566162211741766797162817263172726105393431354378395983683175664152705325989262583181225888309598514527020690183194488679084489599617778190094920727799502568334840928528943467779809176231817000*x^293*y+534224836969598726653280503080570657527798521857779417271151123050577706053005706691626509470986030520831751801393831737320931031736718563079823240023830648566772192605588159692406322012970299089622440690691663575950911376062884333887088955418163113119542263250*x^292*y+220019255846435582768346836247569297599601083755249068890234313019419873296865537875817970050109902555829155889995767090687886969347139379999024521984426726913254556051948861255546750391801590034089919156109965816893746293103472814520493617746267459846130241000*x^291*y+90176906269454583923364689222595303664061852637714759221208711392466455111813903551919759555749269920769414597167279187873486067718334590957346670278124193706700106776221293838540992061991919295662206302011267679881803058159310688768258651780512437767920986100*x^290*y+36781016621859112992652264239877128076762218375439212621871345012398413477392450112597370282935707843914388513612533002086232010743061928801168121491780613466867835393958052339207999575214706885713136185067886958038991402062166103717011264439308870538251879000*x^289*y+14929373319827645582691719614219789345764439761940916359158453242392052661469688318175056196303960065858508820834300614610844172899922608740923577403264883836973039928165557761279651512973385238723449940287386700664702970780851129177270021661461044361734259875*x^288*y+6030378002959834400862854486529171573043700773406709553208463581779679055404306080833683288268640251005961487237417359057395682742184728355380070535961131199226136745177672700208330484903695580297831111925339929581254495911479838994465310292427462519185788000*x^287*y+2423975471777972651327225823016627789164624820683089134132813792676145502662515189354715831558963238149455107615040310993659048945387979044809636195827513521257564770120437065770015194912269792080500741068028795223837591493830131360520369823426725130260954000*x^286*y+969590188711189060530890329206651115665849928273235653653125517070458201065006075741886332623585295259782043046016124397463619578155191617923854478331005408503025908048174826308006077964907916832200296427211518089535036597532052544208147929370690052104381600*x^285*y+385940228746772181915228692491474256934032443516581230853548564755699144278668619534130732957712023951170226631444965716867502206388588842330025874754659974054975396359957856840477279636869771364772464359993411530052353952928261138406874524959003721857191000*x^284*y+152868933004300278471303973037069301212364315144643053782995526346748336088063999927047598549498207534354734119010279307657420678681114687896411922496964341187744787400596975373355017317811736495948925911071309448444725972986926308657674149356146523022932000*x^283*y+60253353816458187754009783244415894488995962654504156296083194924971837204626896907178928119091911883318091581726892819034888651903559131858892164438218535593498293641182373301754136352285127337539757705895794671183645474032451455919389671682158030662242000*x^282*y+23632052539973865016176298852469099090259890776870893011815661987262945885542120901007590722648009945891101287965206919287675382248683831966909026942388911039452738257042321656598979765430328107352172007041187896069246208174063018872417089588829714390476000*x^281*y+9223064949628688985479916635477523394948651817084334633778056970029010824774077740532129157033459437215832585997532144888662197794277995531529773015015672225119749236429017202089324047341558608563833797192463609438136367356821817087707225242307152421838550*x^280*y+3581772795972306402128122965233989667941224006634693061661381347584081873698670967196943361954741529015857314950497920345111533123991454575351368161171134844706698732593793088190028756249148974199547105705811110461412181497794880422410572909633845600714000*x^279*y+1384092257723370479492723417313411519883104567660774742664162598304652136789375623058098612168106491129396386248184099413138667232124121643383700439012114434450372501930288464826894768689075573132511970210417312768329638002610487033036772634055184103323000*x^278*y+532195916524338856568433070557577320231677828229177563569346061312162232403107086044469452534901251084332220438444231862866596805713009428576305286369803337174555401848713960196233396536048422310979706387961290386715960393811501238152451994837263043878000*x^277*y+203616393476853402305878398542056516165987235386025117553465274839045494994006440378892318165977412362375725222995928488969678612130529850435962105420491055797447301536041114605464987348736758259863782692631598670055692029124013595259985086422544009881500*x^276*y+77514654620153846946789569651872549602499968229714389578974366697346974645994175923550730777668642499332000222823277604076732823376587915476311091166973146758752352033030824318770119321725993489272281411263891355772925517294107244540352943244996064451440*x^275*y+29361611598543123843480897595406268788825745541558480901126654052025369184088703001344973779419940340656060690463362728816944251279010574044057231502641343469224375770087433454079590652168936927754652049721170968095805120187161835053163993653407600171000*x^274*y+11066136971115290141834616150125608869516168195855603530823525736251652209684050374647211575737364034855241580724843724478463170358251578112890896054640616383173973811559775469625595376470823546361450703746356045747249797704652466031041175049564900202000*x^273*y+4149801364168233803187981056297103326068563073445851324058822151094369578631518890492704340901511513070715592771816396679423688884344341792334086020490231143690240179334915801109598266176558829885544013904883517155218674139244674761640440643586837575750*x^272*y+1548348382789793682396612959276834162812961805181442469333332819064017181601883591514424664917985091296618163558208586964064805729138080888223417554970292004230103331658569407272717048559703706075264707520066278005788037538922567263602468936976158876000*x^271*y+574797824295937106752715221868523367290839245485165629026483827351162542759055415480012444099690355810114414142841817900358304592597835507819926243009519360474463017643112752562885370081753019652598268134161590876121312565819199627994889153315806925200*x^270*y+212305625936939834231509042277019574785946096143631627684200592865682471333714038549389001240651704608386992911856759005604298549933537054871928981686142581844192906653406899031435088812685793852532876055025485002124151016102850751790177936245236484000*x^269*y+78019417181744283344639251875024953029261611834203426020560053935612820749684530559816449909474465218109427723073043951513055068213280693661952043816355675568426081816620841310732293566410489817392545981969747903786060960835610453868248449248591003000*x^268*y+28525516786776900322460190317198754995691830793405891096193853280687907177238000259250762040571837214806721186607879643936560379646874796591273052855093207438387707949323854667498301058387464217000276020692895550088218741478777082723998068756647188000*x^267*y+10376448204454267555990287213476931313146755888064540766599126465863312283818182655613015619663052501843861794038561123884280138100429932819986246747016193986443485044236334054798428314154568046238520023876026037974869760183696840718402567245265394000*x^266*y+3755286016850115877406008705829746570472159273775738563150160054312436826524675627745662795687580905429207125461574501977168049979203213782471213108443955918903356492199816134117526437503557959591083437212276089933762389399814094736183786241143666400*x^265*y+1352107057697392265642109112832721251596633434171971085916837519555429020419890001837772609860338233612418326422985384543409691908272896266786510154534848258844279171783901189593946339590275624037550422365833103033216077705096107479740086078672651625*x^264*y+484336856488618125006128637432616567736107498807870239731404484616870096866826269315023024427583844876090146778382824314057203071620141946311585726997557585257652240639009381347085255972636044431361345325074544370107251715258307156921821878927517000*x^263*y+172602429886865266770476736646040863569913647949146169443576394924440156471511258577033950439640313282400689163570030886987865051268424568943017677778262391494258183317153749721251249757186015833940425231022500229455565313161158241558860642490429500*x^262*y+61193283667603112170317868743251293985544486823648574281755095358867822727382881931235311251942844492542598864486262641936157839556599779517010326898382606997964335628003088534462554041113310079150732625883484519779916254463089931378107562019611000*x^261*y+21583036536816773346558059110795388824631231163476051199375783633330407745739097546016778698320381638585970680582316958845050265032800733045864453135780892468200934593120008253371252168554829636024785426156201972516970462722792529850927126604214150*x^260*y+7572995276076060823353704951156276780572361811745982876973959169589616752890911419655010069586098820556480940555198932928087812292210783524864720398519611392351205120392985352060088480194677065271854535493404200883147530779927203456465658457619000*x^259*y+2643404011460511796831010218799832461143182896552843079698457445988828489216638891766371439383826946798016932080588306776785368441620745192641459007030430391669760277873023188926634658181160862406213375596754296534683572064691571017822918518225500*x^258*y+917898028205668968482369631830897409118359606070839185144282666305676649014660611138255493083482304540899553804564984049004878947426853646973750233935196555923012317215666194808979464079057203904176380759034466360630365535249563018301901719653000*x^257*y+317069614581796942069850800242662142665884971451889342180215921022256584404257764868994168981794290681466646945931721640583674582646104015150878777044819240419642695597347059228370594446663577155071679912730991740163983793762281849063963362837125*x^256*y+108952780312671163986418530016270481238210137841186136373335940646574074640926158129479875515891729415376458547863786228173719051217990104535067069695937886640843664527410533103977009635363591613017919540482058906687221276782743830013925665619200*x^255*y+37242572358888936751389712002880660476868076607912151173191239765249851251254920004044729566424116623218494543840838456011123804370760692568957242322338017551494818303605477133396967100559940832868055607001239974805953653591956671117360649776000*x^254*y+12663471725780173942239607561889809586512036758245898792490729451637834294268741206194593453643541663048858921198892861883300463601302832546874350133699941710950045313408020337192543030177007994040811411215950406426656262399406953766813393632000*x^253*y+4283233083719764715757514322403906183673188903524348120989511432171620423061485996212877197555903797795937576287860820931116333276911252184972206662869097931644868267770359819932771907265752703866745036146571460997251382870387646127010412552000*x^252*y+1441087766485154483806266500995706753385371967540902171547873005216619955422556036109005412261799408604240679872551304238506429887558925968775695699656892762048740725604980873996072791229599040553297395339033388746738783021812665986657708896000*x^251*y+482284039183698367247163855666563193466304485137021926744688165745828811748082086751147144636948868746219214197347169818486818535703053890883599494151840111032311896169133599164019027464839145571836861640129840767241912717966638883534779910528*x^250*y+160547283350099323317963999889002394629262478407796913030854915361460989263675794524349914992326520887556329626280682363011590724268659750627030457440692447081328860242720905181098211539560301455338502543318855115593179999323115473879753632000*x^249*y+53159938236934483385868398899417016306763772770666797000908077027930566923743713878408416001448542155587135740616874878178040013753851433385811946679165451227727242287815831635762572703923557263802243528306376228434443909350340097069226934000*x^248*y+17508186829694225603845103488785418385228971642928772451826298941469828150183852645212864765417315344735205396644070345004188477305385332642339127193138156579649875282042929941127646786949591236949477284223082741901383917023750788941790544000*x^247*y+5735440513175694594363051142877981884816387262338735803184477239447012669887813797569731561084982612930498319590298906122061742565557264141455920977062499569195648799289925325541815326759348853483449455176527105105625765921573534308517592000*x^246*y+1868766048001617046640146465096666945251432141106396036534280001197304790453512840002852932485967844742917333270481497888777733339241174806355174252128973369565734575662677655739452411103046116499243133739636646166866143598287535681980566400*x^245*y+605618626667190709559306724799845769294445601284480197024998148536163589498823605556480117009341431166686172819137522463955746915494825168726213878004759888285191760631423314360007725820431611828458422971178542739262176166111701378419628000*x^244*y+195206003839887097929287768627691370816175332514416338274900327929754182084165072332603895046604107271692769046062820979135009573818675483710959294891890901904341862079349126425154405680561840536517642278688988676855972238482503482608176000*x^243*y+62579233420966444322977477277742748164024545911613681003694960009142831459699356961507581129716092436703618572814334429986553201105459290952194074747664233723951282962113242376401742190470352573052489543168105868701848619988454282683096000*x^242*y+19952799061757417030514557972613629849399130580514506986685349568132496987150519610915460650054406284166371139158193586372524209048117455086206806441284248433723597466181033801171569973773155892857315506517367088571603907822405713319248000*x^241*y+6327137597215180926781590093947216833822619039347363399725222691999910228820098981882402653504094624321178216496216650415497808395521456152336632042565136674378140775460038350108353110104382329182385575093007195191784923401578653828866800*x^240*y+1995417901881265995305626310837492825384268816614148772580884948856739099759295342512190061551882667328623879052683306307121516445368133346334811682280726415046982636150340609758219114881802574249372586100291362478355297787620074794912000*x^239*y+625859420668796027399008777283675571216325783688689706885605646688662263572797358084532053426377896970526387261930853290553861457274257046947532797985687156425497178530093708309992609523295033130708724511771175370507764004253540519664000*x^238*y+195222204612284999372167875482981370838119969223994954441381577866188229004358808943798989142172921990806396026657330384209461371993804950424001056252416177233641321743331982408621547924697533270129326911928623510066641982978168602464000*x^237*y+60559767661140765512046840954799194880411561133621471469381458055348966327268373978639215218187149884582612379997103797195866943930015410013728076350553185869598158708337277265501710547321093435890903767182046821840044698908149160712000*x^236*y+18682490415724471452082424137689686263760952192855774204933364837990007912725929750273012799336166500341825518535054243317940651983638740866980164730366734464346621510023003182560004822441539935778108874581651045691830782931141440428800*x^235*y+5731573430411554557753746308560151791101597604857841955821593651341582061998163826780885127733680323211917750464409591618428267906207707707232818161404938118957514431926117164362403568242508988130359772228052213756632159254331904048000*x^234*y+1748615622837423424399448026340385292200487404871883986521842130917770798575710998000948005071292301996856262853548688968334047835792182012376114015343879426122631521604578117941072275057036640446550438984829488942701336721660580896000*x^233*y+530504479324374554537853372574622100368116621530141886535923458989375776130391487674766777580222794746442069329266724647945095241848409386567232507259275919643975448611805600885768020948293668260476890994095404848501837833524629360375*x^232*y+160048165413855522305308169619391843023931152399210556146078338732815578754552438414233930297284380209589805051222210817065360333041392688795315918965087143507675297890687775559815579791942953233329829272600954388624741713104959703000*x^231*y+48014449624156656691592450885817552907179345719763166843823501619844673626365731524270179089185314062876941515366663245119608099912417806638594775689526143052302589367206332667944673937582885969998948781780286316587422513931487910900*x^230*y+14323376671278898883354427631307441204476328813937131483890279341847308604493019780262180532441792781402978662171637543939701508404482614172343448000766553699130474130295014933368709475543532779636521685874793583417778441250638417000*x^229*y+4248773649900087881202284880271248751068755567864770867630665763319991801073706644663263396281308998628603774141586784406983996664023987882728820715253291187954505927250723341633982473963042754581299826509491879019004226744036525250*x^228*y+1253195850164579607909600197544430420755079261931652985536599992285844929682800924945956085837177815895629573744219646629744309494692715701503455527914295460353981049693615681620372579642398121661754670691027358882707585637309609000*x^227*y+367539351404857326867544243982668870169771308085898227024299997737579843718340839745131823624081865902206606253149689644640772939657941168270393287902513009690379455142701239958429684210367407774183863368040323599967211808358244500*x^226*y+107179217312900330157503224696881502784991375003113547493537805791861993135929070686967473727796776379224120017047522399598470560467993166489172752343184438954871944338387716426587236943926495686407165317647887914313019185405113880*x^225*y+31076448318817750367832764892781363565235901257346067250059286473155861411835104258463507202002931295522457479169706881326876129001673276366061687212907859233049210665125304376265629268535388568868056954214915954536635717417719875*x^224*y+8958976091911423529465301590711744451239178740856523891908983487756644731339849876313803878054899112222690444445320902724865190342824728321747513430748211670788961633189276937301803032370562470304304707521417212118669756372676000*x^223*y+2567932735856359186466275391682158113915600076106690010148718917441814620936743602079663579442471082295192762353864474688489636820629710045950765417810862728259560982263764469175195470718040399585938238788272542805222822199366000*x^222*y+731811383003994530674599662327906420140260868928992531775373041941056284785567496356463818531744005480786640876197578152560589697278299910399319541404379365434688752326772416119246976250840781396762887048776000645390842783388000*x^221*y+207346558517798450357803237659573485706407246196547884003022361883299280689244123967664748583994134886222881581589313809892167080895518307946473870064574153539828479825918851233786643271071554729082817997153200182860738788626600*x^220*y+58407481272619281790930489481569995973635843999027572958597848417830783292744823652863309460280037996119121572278679946448497769266343185337034892975936381278824923894625028516559617822837057670164174083705126812073447546092000*x^219*y+16357082351283405002830917131027914473435102091799281941090701794763352354362041406620287431971008083312132511929707043826369579884052631187737393301445099104939460783788850697092782996421119731158509110398238838675300527614000*x^218*y+4554079122068687472052541423453493429385507351228918854607628341326195163794284836070527024482349632390861925415933761882692935395559991824938380255063897324236018455767521650020723746130018009441321821285844274369368473844000*x^217*y+1260504042715440282443114143991591931347774356143718611543182844474214732835918124269520872847793201822470711499053094806816794618413926308688301677740900152243897965435653313845021751160987127613223004105903325941521631153250*x^216*y+346839328951000128672245420512336123784865300544004102029716553383987748143386388334033768834552014768985571571713972583786532022391602653091303391582209468642906956094396325847802163376781171419689387117038367392826334177200*x^215*y+94873353339014030107548047595613570755402086026667788723141296409106063423445386121904911322428350095842109272160946699127359268211443473809962123651622182898505083410044796510531126114513933658057529554915075050200587593000*x^214*y+25797836867279545670921578380509916317224963671800389817982512619502792341318059250429035607369335350076507476801070639915190449043518301645910920535510987471766312388500109851656494267479011185291373983166233876420490146000*x^213*y+6973273163363633537952152531787578903006240180321678973642481202987429909518713985204802772042726433459766614922116810027837012241458627221546987403634315141479980379125029693404610760118057591963277485297471847306553808500*x^212*y+1873680495098973776990945927425813342759598121962225529039551349852135793178665861677336106049503173502497493489846341857923252972356437225561421203511374917609322991602669575414166642769364017105468728622387872787058818000*x^211*y+500438714513776540436822266692210905471234435106366565351070044074431205519871514954326478957525531150667052058680478648128868831857225638725898574608734313437426773706535797990366027372576971657283419923194735643125836200*x^210*y+132859835711622090381457238944834753664929496045938026199399126745424213854833145563095525386953680836460279307614286366759876681024042204971477497683734773478962860276071450793902485143162027873615067236246389993750222000*x^209*y+35060234423900273850662326943775837772689728123233645802619214002264723100580968968039096977112776887399240372842658902339411901936900026311917672999874454112504088128407743959502044690556646244426198298453908470572975250*x^208*y+9196127062006629206731102149187104989557961474946530046588646295675992944414680385059435272685318527842423704352172826843124433294924597065421028983573627308197793607451211530361192049982071146079002832381353041461764000*x^207*y+2397478969565960007296395648465655834809191467649788060004848593457091359564028765374437155473376492775039932998614326393610526060515606539725633500755341124429399592874560184867464426128827112390873534386574407534742000*x^206*y+621233544315204731450386796960912077950557789101706088504401019185107949773823805870608872990585606932903429179515158788784614299957503078218214466862390278782963919663093582493959335575519981323924462998282173524725600*x^205*y+159991050985699710989107152483651979874201440660615261486686192126818002140243568094817611762650815855835682137940461748367896898858402174666751213199484933606165331069012794486509627880630145943975521249557594940413000*x^204*y+40951285321308332549282131877873279666671385062441045600105374145383779719711026212475273274254412088570237335181749305730302342995124270554601314294472931562933158767978180771954785555393412512636143456599434589516000*x^203*y+10417432230859137227448963372441448336258510235182371249149612721194119402382629475103358990819104829548569146669041490054199718832093016193714369425786973818640891265538309143742884044793061077775861054748978974526000*x^202*y+2633693755486290012446421278139139629442076429920949927820052277448325556046672282817119544612589706594256530196678824769647488365560437135332043334178934557403579518947106942473169683414516067222432957521018464148000*x^201*y+661715556065930365627163346132458831897321703017638669364788134708891795956726411057801285583913163781806953211915554723373931451847059830252175887712457307547649354135460619296383882957897161889636280577155889117185*x^200*y+165222361065151152466208076437567748288969214236613899966239234633930535819407343584969110008467706312561037006720488070755039064131600456991804216657292710998164632742936484218822442686116644666575850331374753837000*x^199*y+40996570887737006659320956622289254251252959642252077423044398618643611755688355827193083406091114159849933122615183448977871288980284901423153415355113777417250326578359551570505818072989042753925927950054334181500*x^198*y+10108743506565289313257222180838446253733606487130649227599988700487465912361512395746239743967671984620531454891415097008242235638974359255024129813589698541239806553568108606426092127586339309187215110972301579000*x^197*y+2476893620389753724765762151275091930330249350702410320693032057208993513352261121843295061643820125584881463449762156854009602513529787031392728324971605239582390411757359944609378294943418960086917135399929615750*x^196*y+603069751051418298203837741180022383036930276692760773733955631320450594555333142709671841095886639272840704144289916451411033655468122059817359939993086493115712448080052856078805150073180268542901563401721993400*x^195*y+145903972028568943113831711575811866863773454038571154935634426932367079327903179687823832523198380469235654228457237851147830723097126304794522566127359635431220753567754723244872213727382323034572958887513385500*x^194*y+35074808641316449769620014926527264153125216955988604780065773017198529602990355463987389726766401252827406344883152593708400446444662333494594024570889429087554927127812163952298896484649529948831665457469141000*x^193*y+8378017410611478719723592674281883640536097614487377131872146277622916105664775500680156209487519111133278990795109468546684760103737413817396840027452549274626362544143252033160503739526434752629345833281614125*x^192*y+1988355182740919547820679596368506376987553451151515957131584777878368222852455990272669953302353114137935186937776289197729881260713947407837074518258206997191917933838695167326102247205284885667286032126168000*x^191*y+468859061609278560041666423341215701240274949592517960261892213055269543907184066842074026025616598518945210747055890415760996692341190067773927448132490785757600401682951576492945097797789398965989669303824800*x^190*y+109843676579239120108405203988694184014367744047568942601429741652899153319808844266330536306864554523550665896350948432792342011769206057801536640129683414665775679802417755281947680125252756847765767161192000*x^189*y+25567062652064277956266728514609853175758009390382426295160370901967906376162403406818314485218473897722999820702375928322355468256625547936564562788805622379102959954011029246660235891222624438704100977174000*x^188*y+5912186689530238936996488266601048458846870560137633632829212459495653626960063764430311344675366657776044238981607225737518853668198773692588115380437216491108679546560976012757840525891578590991846228424000*x^187*y+1358205050297487323364058115300240862167524317869456375109413673127379887274609243179936389992989637597199352198477335642402979896748366929378350830640982166876318274209953948876801201894011297930559268692000*x^186*y+309970723135377475025416944105331043390379782973888203399203611290420440531383213780942538084289659623409913507873355128204851853736437115171010128219905132563184293255277833731392666935320369834458925124800*x^185*y+70275225220643177548654576788586082141201298835991810819672387363636987130276831555728394050972533125405433822250699385683698030565246159689505972696914766573761144916944116716063288459600819141390810230500*x^184*y+15826978507464314160284506889963083370845824952047115288641027264270753527504206862000029994343875269369155230470169751488127830629137446001063768636759261994580233371747512210227227755895410920460109036000*x^183*y+3540754360471845343926729536507633565849371596851616256505266490662038992094461926339860010959571117719505387745771472521182632035613878506350451907734651522014893284877499675393132859815232516435452266000*x^182*y+786834302327076743094828785890585236855415910411470279223392553480453109354324872519968891324349137271001197276838105004707251563469750779188989312829922560447754063306111038976251746625607225874544948000*x^181*y+173679278928293768902639036885604790086378389981068439682236648999953674137966831617212645524033163226891727691594752446160990893887835232967325689783190223708589616412690363481343373340530375467430043400*x^180*y+38078282834461484046863613446295812686416699386835955106945915736896055231222935068329203647169268429769197301445865335333712985261644752660315011158312107512236456704365731335739107431541373427694772000*x^179*y+8291986164682002000472733341711618577699013613435080248349536395260819813125189023395288872072139962200348317468138558424251367836781521564715799266834388375535676094989617894278954051394046038390954000*x^178*y+1793406485192462158060931391038478866136603187352909215317396693021173665536189120491326147301143272505057108759816115916788266676727959706584948079582650219739186324311241780293625542099805826043244000*x^177*y+385234160047409953855321427443945096245362577865855498921333998379548226698914410590976611738231018487129985740882830724844081555559282606875650254958894525356597062382390528048509370086972853409774500*x^176*y+82183287476780790155801904521374953865677349944715839769884586320970288362435074259408343837489283943921063624721670554633404065185980289466805387724564165409407373308243312650348665618554208727418560*x^175*y+17411713448470506388941081466392998700355370750999118595314531000205569568312515732925496575739255072864632123881709863269788996861436502005679107568763594366399867226322735731006073224269959476148000*x^174*y+3663407666304556362364870828479300814826885744466561832629659484928378603248340674158447888970532506261724292086357335198238555566976966564677345486051832430173611725973586477865848538117258704776000*x^173*y+765422133177159723054495958124298358653443519073327532662960254701219200920245092547598411584422854569176693877946641291419408349138907265325097547206481896642554140813321812404337919195997289766000*x^172*y+158808934748457747123490114351482892265853178866842383133931440058636553146299343689007149327528022902169350237643935225722724048313500663010756065282888885672520280120496202332383741980351669288000*x^171*y+32718467279501535853152782595305511539109510344855478934822019578345603118092997314241231969888303513579468542936280630841669653327239293222697936341414457168675864940487771805828457685108596925600*x^170*y+6693308589067702882113084285441560724005555666215922285101977531069497629453441087149229163515056073776786585197554401014541324988725246507651804065030635040523341804913262583623150188289363992000*x^169*y+1359578307154377147929220245480317022063628494700109214161339185998491705982730220827187173838995764985909775118253237706078706638334815696866772700709347742606303804123006462298452381996277060875*x^168*y+274200666989118080254632654550652172517034318258845555797244877848435302046937187393718421614587381173628862208723342058368814784201979636342878527874154166576061271419766009203049219898408819000*x^167*y+54905888953456498084560735383643780348135169243677707215995077458859347052564161024881266678220734599515611497430213577635002480769461150202950496588709527359954715020504704480706498468866034500*x^166*y+10915422235058417583277942603215410224898728256826945386652913602599582767336108658838670980340888555113283243800497549565760972224827007106215308303863211427248482267549438256044645204588936200*x^165*y+2154359651656266628278541303266199386493170050689528694734127684723601861974231972139211377698859583246042745486940305835347560307531646139384600323130896992220095184384757550535127343010974250*x^164*y+422120648592147822028292441739135841558996282333432145682672569049785788965082489164672241269549547971745531971156762433688171912108948586450507112298049111976219367071804346819308105440621000*x^163*y+82106999666491760131994830553077735291308346086335846952596215698228023390582870804106891798253671025530455502742902478151756589109497159416984080315730316529980616745470296577025323611958500*x^162*y+15853794929644416139908417818353507886998750972570211211347541052577997365046990548587981491438730281449265544033790466579957768099807556407093469619962230366933086904369711615587726370843000*x^161*y+3038644028181846426815780081851089011674760603075957148841612035077449494967339855146029785859089970611109229273143172761158572219129781644692915010492760820328841656670861392987647554411575*x^160*y+578101123078591472402526531624464021246090007719563785748701457327457692264892243547401623944654453386180114962785859264905316950131706377111612843851179228600017437654385045039267073372000*x^159*y+109166364096788650964372587325759832990651200982672971418103956906253887256672050741136411172446624808079142849267163447885920896758837665036515964575222680935157687158013327982474423594000*x^158*y+20460599676503685471377068561648936669659418452268481001257918376261108169103421135349410397682760047065841720265968949900326811017670641845515447690255259297455414674930137391733047364000*x^157*y+3806059418496538648111611094998676607981669072282170044072859224020135050413788054798409280137669819181679087774593750159184015793571434561310338018210990177370260786687241197277355967000*x^156*y+702657123414745596574451279076678758396615828729016008136527856742178778537930102424321713256185197387386908512232692337080125992659341765164985480285105878899125068311490682574281101600*x^155*y+128737416228469937906666605504592443914273585641840994398536427653708877864514380467813079851901543256554339030018992094855105826078248195745357859863110415164733316298204557682049138000*x^154*y+23406802950630897801212110091744080711686106480334726254279350482492523248093523721420559973073007864828061641821634926337291968377863308317337792702383711848133330236037192305827116000*x^153*y+4223161381422791702341335900986844751047139494683034335972571490355372708677251331812907636651144107687138480187158188360384046181383356335557408353142344236750471139285012291027769750*x^152*y+756090141314798985578189701943463371212208719896138067217704200864566138655997882727399247080063491600053061234921136196440959975936713972914871695733376117769224514924996311232298000*x^151*y+134317189810040760967419582345250551827110019652137468411615687447705278749477270931573278010693632037185896760556578312544217595725228011658994854183223286803709296180793462348325880*x^150*y+23675180342545375023634473973898452143438898881105311705925209303355807065125253395694467334434835259198454188112205342986642349422778145415804028351919498261523377705192737194182000*x^149*y+4140377782909930608593352842853133062643657198690952399275652800704243254347022014035769522101866729601607598625256568198368204300462375195956338291591555447144346570509058499921500*x^148*y+718377387890585850025575874258222383670880733184362198233055820051849943309917066913591898324825645933221482528180506557278422317079052202815402188927960382388468103675663139494000*x^147*y+123655124145100843037189289831333361123676191777636116089296493615482367291051298403159261187060152168833206008949103587728253021792295870976421688258091541230801886698269884667000*x^146*y+21115377924192658577110685748976223069072191812321488829283377857146696636834490721475148693930739434677950967610022367027280632961023622412348030977405105286195409892336144048400*x^145*y+3576787148373756417851693263553215356326481089703990514306179660381157724697431255390066075490604226668578142877865938339901509088023861273119701509023061058993381348584977671750*x^144*y+601000407661401311751042975439513432101532411805571334959264727065212033088016453647805734971583440653763421907132666418839926847929330248925597453091389489492470144919762876000*x^143*y+100166734610233551958507162573252238683588735300928555826544121177535338848002742274634289161930573442293903651188777736473321141321555041487599575515231581582078357486627146000*x^142*y+16558412473402985306295712555764630841757392797126722849091111999080347050542944590218939535499582571368724468531788636297103145596811194285493759863984731763277213926776548000*x^141*y+2714809486918396428125227291119549940334642307435892932234705571942242946658785101419616830820280398329058314026723485718478538987384160923551883884676566486769868794971503800*x^140*y+441432436897300232215484112377162591924332082509901289794261068608494788074599203482864525336630959077895660817353412311947729916647831044479981119459604306791848584548212000*x^139*y+71182260706177183617114027401885847189654477342083850674480613151485818494627945805241495384909168575205912823215921474896443687255276699747931990260887469424671639503714000*x^138*y+11382563125669120902852532771101097233108131950414335333810341384594487777819995968856693352395672379349265318196752217306731435505478777016471164143687683407421420917164000*x^137*y+1804874014139663846864348367639873056638673700470791598069463853807227807362661397839545126479406384225520079389994275197942368824364111633398784129265292392148998455615125*x^136*y+283772099332941367830695234680951139540878177183847002702250964298015007862799942319281083469594529774185815950334360031121574751576322753921658545179282965702039063541800*x^135*y+44236990080770305608711150902919634916880547251523493492845127228905341872376434426215873289140024849324578698955125409008559574437417519375778179675754272944313248935500*x^134*y+6837089585724591639639324361004880137095724719381947091166374911964608778429575793671196102358435213159738806989604157793710476325967644286452452222088895703042647471000*x^133*y+1047618565554574525428606152089457440361441690873040280098073575220383603146467258707683273748469911693830946232278056436133056856398268076149972517900717889982341144750*x^132*y+159131934008289801330927516773082142839712661904765612166795986109425357439969710183445560569387834687670523478320717433336666864263028062199995825503906514934026503000*x^131*y+23961245235730993073967246778475587025290067482211833556149740437166346924869002337967090154700926832281423650183924119272532596802823765687585578323002015467077553900*x^130*y+3576305259064327324472723399772475675416427982419676650171603050323335361920746617607028381298645795862899052266257331234706357731764741147400832585522688875683217000*x^129*y+529063507361580533092868484599368534551283497399241155816670634738199841385064579898287455490281316131094011172416508863849908425914738082585673627904159248810934625*x^128*y+77571739910976298093799731991659991320234006491526767404964308415222886251189308392876053038666676362863726724019831769270089666113501116347040348650323463743184000*x^127*y+11271866096903878555964034282540982720445902545107436453581770215941998345424533370589540887769642903985919100629884021392793349652648331551572224575047002168632000*x^126*y+1623148717954158512058820936685901511744209966495470849315774911095647761741132805364893887838828578173972350490703299080562242349981359743426400338806768312283008*x^125*y+231613686922682436081452759230294165488614435858371981922913086628945171481326028162798785365129648712039433574586657973824520883273595853799429272090006893876000*x^124*y+32748115368771518898631861054226313022335450452038911925246548166464311589149860310361515832697920684484480915905069086378837616335149242726487149075439971312000*x^123*y+4587720034577331235229748188690018794700752170388139142147295472067323833104137606121260190685471804318440948355721523490429415500254392773756172364782592792000*x^122*y+636748400703565882477849009124211937375986080531687116430000054143587608235159030656193109514934653159101018998177503829160851753163863388393916983507936656000*x^121*y+87552905096740308840704238754579141389198086073106978509125007444743296132334366715226552558303514809376390112249406776509617116060031215904163585232341290200*x^120*y+11925480830430008014624867934789440370832883460582108309982974907343014229148835420916216012481750030789065622553835202248756020348698917035754404344927304000*x^119*y+1608993445375477271814466308662067351620309673253141597378655344641517792821668271075997398509442464471540599868374590779594066237522869758792260903680668000*x^118*y+215018376618693451952556086548271741213133116017973622299752356362060135394062124560552313730593670223829887638129333762165458455297506943983563744772728000*x^117*y+28458314552474133346661834984330083395855853590614155892614282459684429684508222368308394464343279882565720422693588292051310677907022977880177554455214000*x^116*y+3730129365070055896285619048793547654146077984758465631122324028614004342828196378218953398716181317940817592127069199862092699025101316874689939341022400*x^115*y+484159003366880844326011501818575598450111702310636058215651538702720653978829101010360768456389222983288965118073316009188104275267100948746436821916000*x^114*y+62225621627761461390265288847032264062359339417601477606070209032818663532792015236957302823030858421753034975716299915498809343157214778080150842952000*x^113*y+7918350499929104884121596441120096665593024047510097938610285608905978580186371308306503625002800677542897468756691318075861999748609538201640816727000*x^112*y+997587464557997465716106638251350761019593580788673756045390312933036671519542841991370535433423707406979208662260323537116472409273642608080732816000*x^111*y+124418211871840133364593075107752735363117851087126726877571151388277607346819388158474302733831496092331114788214489789460593749920645314041529598400*x^110*y+15360273070597547328962108037994164859644179146558855170070512517071309548990047920799296633806357542263100591137591332032172067891437693091546864000*x^109*y+1876984041138041097373172394777313867378044312752147100378571596817009799147887021712021673861987636890894578961880555147429097982249673258944628000*x^108*y+227003669029012809088804724116405260556359222594884531736714146087611487466933704753525577577933555189492289504908286624773060002332547269838768000*x^107*y+27169342937476924577742847293574231408870734695360900330904265806906520312037926631574090381251555263171895947455466072539952371643828364510904000*x^106*y+3217821621645311737699152863819964837251729472299726743101510810650381176621251645750674391522530567481811140145563579541044638429324923618051200*x^105*y+377088471286559969261619476228902129365437047535124227707208298123091544135302927236407155256546550876774742985808231977466168565936514486490375*x^104*y+43720402468006952957868924780162565723528933047550635096487918622967135551919179969438510754382208797307216578064722548112019543876697331767000*x^103*y+5014700951230196163319041483693479142008329737079861263851064162767945391812556277118225621048293436662186311292501584026211595789866175024500*x^102*y+568964957759154625871570891364554919337986243806613847511466679201702369260156552020087890263543860444430482482575263148691415762587708401000*x^101*y+63850511926305130236698511142022274281262900693853331776286816221524376994750901948920974351797699894319420811933446197797592213357065053890*x^100*y+7086627294817439537924363056828221340872685981559748254859802022366745504411864811200996043484761364519358580680737646814383153535745289000*x^99*y+777800556748255559040478872090902342290904558951679686509002660991472067557399796351328834041010393666758868611300473430846931485630580500*x^98*y+84412463523066494779586854335446765830020649808709423342062304293648131362818582549756617647861593111121117523707028124277961556580063000*x^97*y+9057532037320188046039739901037982616716817512660192548871729553632598166143144366511495477701962977631358849335820495636020211270205875*x^96*y+960798978544461936375486221546570531718027051066716557670371311766551849668223048823318857303191652875812651421258306719401038985568800*x^95*y+100746029759077134608908599389541060169108796745406261565877786553887887106491379291628357001990294727596249321213619357994590180606000*x^94*y+10441154131591235560349954071242403148727923808233945520609164207348910019856879441469752544858972110688034659530408180431633381452000*x^93*y+1069413363698221263339808071173506049374115544235415124908207347228467656218821352485338091048330843936109276802123304823944828717000*x^92*y+108235455951855177367725349337692581454805973673991409781688752414762403049649685840100224836574738880222281040478926340817298396000*x^91*y+10823545595185517736772534933769258145480597367399140978168875241476240304964968584010022483657473888022228104047892634081729839600*x^90*y+1069285514343245440515398621338346029740124877130540821114378454152427692038251561537762923742231229332602117853249546725966724000*x^89*y+104349134623408820401173768968325434919814818053309356446468950021454018192329373878136951988002828301098232992257905327424384250*x^88*y+10057747915509283894089037972850644329620705354535841585201826508091953560706445675001151998843646101310673059494737862884278000*x^87*y+957356749069264440684623964593004438377463201142908334696453945518599518360460365125930223084679661722131899536151197014149000*x^86*y+89981071497220482949593072081965444481379054970808870802071081218141594075409389509103824246210328861315129355310385730291600*x^85*y+8349771918410197653619444461754435350346309686155844998008779370679078052849124572351337402759692088659155016595396055758500*x^84*y+764864603213147876667430027030940642779814627739466717374850018688159821635034311971114876588674084457327177092708035642000*x^83*y+69154424909249753554898357563799644172902629741150040895547441776816193023646893130285985573921513082743088996399528277000*x^82*y+6170470992990728826443596648782993277669222675488904628329586752664774567942377841875354534343377663530939388144462806000*x^81*y+543269728730705472762968835381980929881746779037610081407278833658529065221013701295547518784579989941310967869240747050*x^80*y+47189552984208944431093927069010287068989948233451472869253318884562785252639626605476440285305536585564470607534484000*x^79*y+4043356492139378102013470974459666679447077993972523163417583722213080298219664318690497594944834479674179151838638000*x^78*y+341692097927271388902546842912084508122288281180776605359232427229274391398844872002013881262943758845705280437068000*x^77*y+28474341493939282408545570242673709010190690098398050446602702269106199283237072666834490105245313237142106703089000*x^76*y+2339513463285822122215636041560218253810262105381893874531681483731968805974072997491266214052587898403027145334880*x^75*y+189485431691616262598458642674963681464114101407820778174812215205073067438504832410199747358470942095277576566000*x^74*y+15126129390700758826629924010730649868764232474842219616975300890156857594875250915161576380287863770281057892000*x^73*y+1189878712846072623215500487913079138383393287352890120731893281230011427183074694834908486811437559515643562625*x^72*y+92218802287316715685162578180561569390316810214648104082557929223423921159506327263846513509605494386573021000*x^71*y+7040360174623104100695207581526743469583326371225822999851196747164621937983816382508712321701064625211488700*x^70*y+529350389069406323360541923423063418765663636934272406003849379486061799848407246805166339977523656030939000*x^69*y+39190103911790811493430678880033665123209003163588836925177690112165519516673927070339568088464734191131750*x^68*y+2856299106111227418599449264568370019697976650722444706229456514069941401000886431707492636672670873523000*x^67*y+204895117890205821248568630327709626680689973874093999269136602187030057673511125186725917191722642961500*x^66*y+14463184792249822676369550376073620706872233449936047007233171919084474659306667660239476507651010091400*x^65*y+1004387832795126574747885442782890326866127322912225486613414716603088518007407476405519201920209034125*x^64*y+68602797544170865297614373893388453489255228032425220003477632724223762169129219306246775798178632000*x^63*y+4607650581324908863272607201794746876144007852924081940532079809835924324792260998180753598385132000*x^62*y+304232519746692598000960219926809697892362605837372822484546270723990743490010843330358597550456000*x^61*y+19742748621859838806445290867590842097270339314978449118678002674641952503075171748033908989976400*x^60*y+1258836256441647532823291659995165277190457342081516415643655855981420988506387146527135536024000*x^59*y+78844309055262425091904679341523090609593400406379478262182266988220635161228069686943733148000*x^58*y+4849384862359724979141539132352427630282520915768833233517042932467440974921768867277557288000*x^57*y+292812433426381698952402256932297007336974250210618108379736702490089126663708501519937251500*x^56*y+17351847906748545123105318929321304138487362975444036052132545332746022320812355645625911200*x^55*y+1008828366671427042041006914495424659214381568339769537914682868182908274465834630559646000*x^54*y+57525587962256663432116550562569093556047101045773553376338833032605118079361214414172000*x^53*y+3216092997889876753061368333139411348597569995175103722516833492329189090934751438767000*x^52*y+176224273856979548112951689487091032799866849050690614932429232456393922790945284316000*x^51*y+9460461017585217846063722277728044918729694001668654064793569321343252697198115263280*x^50*y+497395426792072441959186239628183223907975499561969193732574622573251981976767364000*x^49*y+25601235202533140394958115274980018877616386006866061442117811455976204954686555500*x^48*y+1289464102541018613806914515423967372639650082192624290893656820447909588483688000*x^47*y+63527057462712657074344845099503633662540412854353607622643470189781709285884000*x^46*y+3059942034853175105151688873902792825630218839057870105383873956785296991780800*x^45*y+144034928418821003903583681302955729239916158742263760190663523070437619906000*x^44*y+6622295559486023167980858910480723183444421091598333801869587267606327352000*x^43*y+297242911333923795640059429176065863139989673213703918037987737481286092000*x^42*y+13017937722653596889345668431068577947736774009359295680495813320348296000*x^41*y+555974423571664033815804589243553849851258056649719919687842027223208475*x^40*y+23141495257925662177556902778087569192560168851185012265883122881299000*x^39*y+938168726672661980171225788300847399698385223696689686454721197890500*x^38*y+37020157438796630577888452705537072885294536345248398842450057653000*x^37*y+1420898158957961962014390819610862755970848386695218627770386030250*x^36*y+53007599712421378893801108296363791932591235151324218238066214600*x^35*y+1920565206971789090355112619433470722195334606932036892683558500*x^34*y+67527628787012232752920195512655640697664298485717946588667000*x^33*y+2302078254102689752940461210658715023784010175649475451886375*x^32*y+76023224077694604844267036884498328958811481548795886956000*x^31*y+2429608192173745103270389838576750719302222606198631438800*x^30*y+75065134670661537691155195836562844056711306061749684000*x^29*y+2239597639351012955806070657675228886465666538879363000*x^28*y+64448852930964401605930090868351910401889684572068000*x^27*y+1786569845108869449035023052818790124077024110314000*x^26*y+47641862536236518640933948075167736642053976275040*x^25*y+1220334593653599350433758915859829319724743244750*x^24*y+29977513047785449754769922191029584107874962000*x^23*y+704992638138103624089681196721554636483767000*x^22*y+15842531194114688181790588690372014303006000*x^21*y+339482811302457603895512614793686020778700*x^20*y+6921158232466006195627168497322854654000*x^19*y+133912430159729244110912628766939143000*x^18*y+2452109606180189617493822296851378000*x^17*y+42363682220592706806295710413082750*x^16*y+688141030994399298376377021938400*x^15*y+10468676942105465999640624066000*x^14*y+148491871519226468080008852000*x^13*y+1953840414726664053684327000*x^12*y+23706860441577319154916000*x^11*y+263409560461970212832400*x^10*y+2658017764500203964000*x^9*y+24115080524699431125*x^8*y+194280608456793000*x^7*y+1368173298991500*x^6*y+8250291250200*x^5*y+41417124750*x^4*y+166167000*x^3*y+499500*x^2*y+1000*x*y+y");

  // Another bug
  x = may_parse_str ("y^4*(y + 4 * t) ^ 9 * (y + 2 * t) ^ 7 * (y + t) ^ 6 * (y + 5 * t) ^ 10 * (y + 3 * t) ^ 8");
  x = may_expand (x);
  check (x, "2149908480000000000*t^40*y^4+35294330880000000000*t^39*y^5+279386578944000000000*t^38*y^6+1421204565196800000000*t^37*y^7+5223918244884480000000*t^36*y^8+14791991611799347200000*t^35*y^9+33591871257937920000000*t^34*y^10+62891786195674567680000*t^33*y^11+99033256427425092416000*t^32*y^12+133147372944961552841600*t^31*y^13+154637765446388907186752*t^30*y^14+156580652661814944410208*t^29*y^15+139256593877877030214416*t^28*y^16+109433675558119494758568*t^27*y^17+76358550066942114807788*t^26*y^18+47495215175958435790082*t^25*y^19+26418344966123722464905*t^24*y^20+13173871308634296318792*t^23*y^21+5900730001037827016580*t^22*y^22+2377292513137472126340*t^21*y^23+862238433010102784234*t^20*y^24+281653796194070641520*t^19*y^25+82854892316358128960*t^18*y^26+21938059867530989790*t^17*y^27+5222788242376165695*t^16*y^28+1116208104497618640*t^15*y^29+213690433606026280*t^14*y^30+36541925421738520*t^13*y^31+5561574161796220*t^12*y^32+749959402654920*t^11*y^33+89095700342988*t^10*y^34+9259479876702*t^9*y^35+834384250519*t^8*y^36+64457631592*t^7*y^37+4206482452*t^6*y^38+227388228*t^5*y^39+9908010*t^4*y^40+334368*t^3*y^41+8200*t^2*y^42+130*t*y^43+y^44");

  may_keep (NULL);
}

#if 0
void test_matrix ()
{
  may_t a, b, det, trace, test;
  may_size_t n;
  char *s;
  static const char *trace_str[] =
    {"2+x", "4+2*x", "6+3*x", "8+4*x", "10+5*x"};
  static const char *det_str[] =
    {"2+x", "4*x+x^2", "6*x^2+x^3", "8*x^3+x^4", "10*x^4+x^5"};

  may_mark ();

  /* Check det && trace */
  for (n = 1 ; n < 6 ; n++)
    {
      a = may_mat_fill (n, n, may_set_ui (2));
      if (!may_mat_purenum_p(a))
	fail ("PureNumP", a);
      if (!may_mat_square_p(a))
	fail ("SquareP", a);

      b = may_mat_identity (n, may_set_str("x") );
      a = may_mat_add (a, b);
      if (may_mat_purenum_p(a))
	fail ("PureNumP2", a);

      trace = may_mat_trace (a);
      test  = may_parse_str (trace_str[n-1]);
      if (may_identical (trace, test))
	fail2 ("trace", trace_str[n-1], trace);

      det   = may_expand (may_mat_det (a));
      test  = may_parse_str (det_str[n-1]);
      if (may_identical (det, test))
	fail2 ("det", det_str[n-1], det);

      may_compact (NULL);
    }

  /* Check product of matrix */
  a = may_mat_fill (4, 4, may_set_ui (2));
  b = may_mat_identity (4, may_set_str ("x"));
  b = may_mat_add (a, b);
  a = may_mat_mul (a, b);

  s = may_get_string (NULL, 0, a);
  if (strcmp(s, "[16+2*x,16+2*x,16+2*x,16+2*x;16+2*x,16+2*x,16+2*x,16+2*x;16+2*x,16+2*x,16+2*x,16+2*x;16+2*x,16+2*x,16+2*x,16+2*x]"))
    fail ("mat_mul_symbolique", a);

  a = may_mat_fill (4, 4, may_set_ui (2));
  b = may_mat_identity (4, may_set_si (-5));
  b = may_mat_add (a, b);
  a = may_mat_mul (a, b);
  s = may_get_string (NULL, 0, a);
  if (strcmp(s, "[6,6,6,6;6,6,6,6;6,6,6,6;6,6,6,6]"))
    fail ("mat_mul_num", a);

  may_keep (NULL);
}
#endif

void test_eval ()
{
  may_mark ();

  may_t zero = may_set_ui (0);
  may_t one  = may_set_ui (1);
  may_t none = may_set_si (-1);
  may_t sum  = may_add_vac (zero, one, none, NULL);

  sum = may_eval (sum);
  if (may_identical (sum, zero) != 0)
    fail ("eval1: 0!=0+1-1", sum);

  // Test hold
  may_t z = may_num_add (MAY_DUMMY, one, one);
  z = may_hold (z);
  if (may_identical (z, may_set_ui (2)))
    fail ("hold integer", z);

  // Test add_c
  may_t x = may_add_c (z, may_set_ui (0));
  if (x != z)
    fail ("add_c", x);

  x = may_reeval (z);
  if (may_identical (z, x))
    fail ("reeval", x);

  zero = may_set_si_c (0);
  check_bool (!may_eval_p (zero));
  zero = may_eval (zero);
  check_bool (may_zero_p (zero));

  one = may_set_ui_c (1);
  check_bool (!may_eval_p (one));
  one = may_eval (one);
  check_bool (may_one_p (one));

  may_keep (NULL);
}

void test_set_get_ui ()
{
  may_mark ();
  may_t x;
  unsigned long u, v;
  int dummy;

  for (u = 0 ; u < 1000 ; u++) {
    x = may_set_ui (u);
    dummy = may_get_ui (&v, x);
    if (dummy != 0)
      fail ("get/set_ui error", x);
    if (u != v)
      fail ("get/set_ui fail", x);
  }
  x = may_set_ui (ULONG_MAX);
  x = may_sqr (x);
  dummy = may_get_ui (&v, x);
  if (dummy != -2)
    fail ("get/set_ui doesn't overflow", x);

  /* Check with double */
  x = may_set_d (42.0);
  dummy = may_get_ui (&v, x);
  if (dummy != 1 || v != 42)
    fail ("set_d / get_ui doesn't work", x);
  x = may_set_d (LONG_MAX*4.0);
  dummy = may_get_ui (&v, x);
  if (dummy != -2)
    fail ("set_d / get_ui doesn't overflow", x);

  /* Check with long double */
  x = may_set_ld ((long double) 42.0);
  dummy = may_get_ui (&v, x);
  if (dummy != 1 || v != 42)
    fail ("set_ld / get_ui doesn't work", x);

  /* Check with rat */
  x = may_set_si_ui (35, 17);
  dummy = may_get_ui (&v, x);
  if (dummy != 1 || v != 2)
    fail ("set_si_ui / get_ui doesn't work", x);
  
  /* Check with str */
  x = may_set_str ("x");
  dummy = may_get_ui (&v, x);
  if (dummy != -1)
    fail ("set_str / get_ui doesn't work", x);

  may_keep (NULL);
}

void test_set_get_si ()
{
  may_mark ();
  may_t x;
  long u, v;
  int dummy;

  for (u = -1000 ; u < 1000 ; u++) {
    x = may_set_si (u);
    dummy = may_get_si (&v, x);
    if (dummy != 0)
      fail ("get/set_si error", x);
    if (u != v)
      fail ("get/set_si fail", x);
  }
  x = may_set_si (LONG_MAX);
  x = may_sqr (x);
  dummy = may_get_si (&v, x);
  if (dummy != -2)
    fail ("get/set_ui doesn't overflow", x);

  /* Check with double */
  x = may_set_d (-42.0);
  dummy = may_get_si (&v, x);
  if (dummy != 1 || v != -42)
    fail ("set_d / get_si doesn't work", x);
  x = may_set_d (LONG_MAX*4.0);
  dummy = may_get_si (&v, x);
  if (dummy != -2)
    fail ("set_d / get_si doesn't overflow", x);

  /* Check with rat */
  x = may_set_si_ui (-35, 17);
  dummy = may_get_si (&v, x);
  if (dummy != 1 || v != -3)
    fail ("set_si_ui / get_si doesn't work", x);

  /* Check with str */
  x = may_set_str ("x");
  dummy = may_get_si (&v, x);
  if (dummy != -1)
    fail ("set_str / get_si doesn't work", x);

  may_keep (NULL);
}

void test_set_get_q ()
{
  may_mark_t mark;
  may_mark (mark);

  may_t x;
  long v;
  unsigned long u;
  int dummy;
  mpq_t q1, q2;

  mpq_init (q1);
  mpq_init (q2);
  for (v = -100 ; v < 100 ; v++)
    for (u = 1 ; u < 100 ; u++)
      {
	x = may_set_si_ui (v, u);
	dummy = may_get_q (q1, x);
	mpq_set_si (q2, v, u);
	if (dummy != 0)
	  fail ("get/set_q error", x);
	if (mpq_cmp (q1, q2))
	  fail ("get/set_q fail", x);
	may_compact (mark, NULL);
      }
  mpq_clear (q2);
  mpq_clear (q1);

  x = may_set_si_ui (1, 0);
  check (x, "INFINITY");
  x = may_set_si_ui (-1, 0);
  check (x, "-INFINITY");
  x = may_set_si_ui (0, 0);
  check (x, "NAN");

  may_compact (mark, NULL);
}

void test_set_str ()
{
  may_mark ();
  char *str;
  const char *s;
  may_t x;
  int dummy;

  str   = "STRING";
  x     = may_set_str (str);
  dummy = may_get_str (&s, x);
  if (dummy != 0)
    fail ("get/set_str error", x);
  if (strcmp (str, s) != 0)
    fail ("get/set_str fail", x);
  x     = may_set_ui (4);
  dummy = may_get_str (&s, x);
  if (dummy != -1)
    fail ("get/set_ui/str fail", x);
  may_keep (NULL);
}

void test_set_d ()
{
  may_mark ();
  double d, e;
  int dummy;
  may_t x;

  /* FIXME: Portability problems */
  for (d = -10.0 ; d < 10.0 ; d+=0.1)
    {
      x = may_set_d (d);
      dummy = may_get_d (&e, x);
      if (dummy != 0 || d != e)
	fail ("get/set_d", x);
    }

  x = may_set_ui (43);
  dummy = may_get_d (&e, x);
  if (dummy != 0 || e != 43.)
    fail ("set_ui/get_d", x);
  x = may_set_si_ui (1,2);
  dummy = may_get_d (&e, x);
  if (dummy != 0 || e != 0.5)
    fail ("set_si_ui/get_d", x);
  x = may_set_str ("x");
  dummy = may_get_d (&e, x);
  if (dummy != -1)
    fail ("set_str/get_d", x);

  may_keep (NULL);
}

void test_set_zqfr ()
{
  mpz_t  z1, z2;
  mpq_t  q1, q2;
  mpfr_t f1, f2;
  may_t  x;
  int dummy;

  may_mark ();

  /* MPZ check */
  mpz_init (z1); mpz_init (z2);
  mpz_set_str (z1, "1234567890123456789", 10);
  x = may_set_z (z1);
  dummy = may_get_z (z2, x);
  if (dummy != 0 || mpz_cmp (z1, z2) != 0)
    fail ("set/get_z", x);
  mpz_clear (z2); mpz_clear (z1);

  /* MPQ Check */
  mpq_init (q1); mpq_init (q2);
  mpq_set_str (q1, "1234567890123456789/123455667889", 10);
  x = may_set_q (q1);
  dummy = may_get_q (q2, x);
  if (dummy != 0 || mpq_cmp (q1, q2) != 0)
    fail ("set/get_q", x);
  mpq_clear (q2); mpq_clear (q1);

  /* MPFR Check */
  mpfr_init (f1); mpfr_init (f2);
  mpfr_set_str (f1, "12345678.455667889", 10, GMP_RNDN);
  x = may_set_fr (f1);
  dummy = may_get_fr (f2, x);
  if (dummy != 0 || mpfr_cmp (f1, f2) != 0)
    fail ("set/get_fr", x);
  x = may_set_ui (17);
  dummy = may_get_fr (f2, x);
  if (dummy != 1 || mpfr_cmp_ui (f2, 17) != 0)
    fail ("set_ui/get_fr", x);
  x = may_set_si_ui (1,2);
  dummy = may_get_fr (f2, x);
  if (dummy != 1 || mpfr_cmp_ui_2exp (f2, 1, -1) != 0)
    fail ("set_si_ui/get_fr", x);
  mpfr_clear (f2); mpfr_clear (f1);

  /* MPZ check 2 */
  mpz_init (z1); mpz_init (z2);
  mpz_set_str (z1, "-1234567890123456789", 10);
  x = may_set_z_c (z1);
  check_bool (!may_eval_p (x));
  x = may_eval (x);
  dummy = may_get_z (z2, x);
  if (dummy != 0 || mpz_cmp (z1, z2) != 0)
    fail ("set/get_z", x);
  mpz_clear (z2); mpz_clear (z1);

  /* */
  mpfr_init (f1);
  mpz_init (z1);
  mpq_init (q1);

  mpfr_set_str (f1, "12345678.", 10, GMP_RNDN);
  x = may_set_fr (f1);
  dummy = may_get_z (z1, x);
  if (dummy != 1 || mpz_cmp_ui (z1, 12345678) != 0)
    fail ("set_fr / get_z", x);

  mpfr_set_str (f1, "NAN", 10, GMP_RNDN);
  x = may_set_fr (f1);
  dummy = may_get_z (z1, x);
  if (dummy >= 0)
    fail ("set_fr (NAN)/ get_z", x);

  mpfr_set_str (f1, "+INF", 10, GMP_RNDN);
  x = may_set_fr (f1);
  dummy = may_get_z (z1, x);
  if (dummy >= 0)
    fail ("set_fr (INF)/ get_z", x);

  mpfr_set_str (f1, "+INF", 10, GMP_RNDN);
  x = may_set_fr (f1);
  dummy = may_get_z (z1, x);
  if (dummy >= 0)
    fail ("set_fr (INF)/ get_z", x);

  mpq_set_si (q1, 1742*2+1, 2);
  x = may_set_q (q1);
  dummy = may_get_z (z1, x);
  if (dummy != 1 || mpz_cmp_ui (z1, 1742) != 0)
    fail ("set_q / get_z", x);

  mpq_set_si (q1, 1742, 0);
  x = may_set_q (q1);
  dummy = may_get_z (z1, x);
  if (dummy >= 0)
    fail ("set_q(NAN) / get_z", x);

  may_keep (NULL);
}

void test_set_cx ()
{
  may_t x, r1, i1, r2, i2;
  int dummy;

  may_mark ();
  r1 = may_set_ui (17);
  i1 = may_set_ui (42);
  x  = may_set_cx (r1, i1);
  dummy = may_get_cx (&r2, &i2, x);
  if (dummy || may_identical (r1,r2) || may_identical (i1, i2))
    fail ("set/get_cx", x);
  dummy = may_get_cx (&r2, &i2, r1);
  if (dummy || may_identical (r1,r2) || may_identical (i2, MAY_ZERO))
    fail ("get_cx 1", r1);
  x = may_set_str ("x");
  dummy = may_get_cx (&r2, &i2, x);
  if (dummy != -1)
    fail ("get_cx 2", x);

  may_keep (NULL);
}

void test_set_list ()
{
  may_t x;
  may_t list[10];
  int i;

  may_mark ();
  for (i = 0 ; i < 10 ; i++)
    list[i] = may_set_ui (i*i-12);
  x = may_list_vc (10, list);
  if (may_nops(x) != 10
      || ({ for(i = 0 ; i < 10 ; i++)
	if (may_identical(list[i], may_op (x, i))) break;
      (i != 10); }))
    fail("set/get_list", x);
  if (may_op (x, 11) != NULL)
    fail("set/get_list 2", x);

  x = may_list_vac (may_set_ui (0),
                    may_set_ui (1),
                    may_set_ui (2), NULL);
  if (may_nops (x) != 3
      || ({ for(i = 0 ; i < 3 ; i++)
            if (may_identical(may_set_ui (i), may_op (x, i))) break;
          (i != 3); }))
    fail ("list_vac", x);

  may_keep (NULL);
}

void test_addmul2 ()
{
  may_t x;
  may_t list[10];
  unsigned long u;
  int i;

  may_mark ();

  for (i = 0 ; i < 10 ; i++)
    list[i] = may_set_ui (i*i+1);
  x = may_eval ( may_add_vc (10, list) );
  i = may_get_ui (&u, x);
  if (i != 0 || u != 295)
    fail("add_vc", x);
  x = may_eval ( may_mul_vc (8, list) );
  i = may_get_ui (&u, x);
  if (i != 0 || u != 81770000L)
    fail("mul_vc", x);

  /* Check for 0 */
  if (!may_one_p (may_mul_vc (0, list)))
    fail ("mul_vc0", x);
  if (!may_zero_p (may_add_vc (0, list)))
    fail ("add_vc0", x);

  may_keep (NULL);
}


void test_cmp ()
{
  may_t x, y;

  may_mark ();
  x = may_parse_str ("x");
  y = may_parse_str ("y");
  if (may_cmp (x, y) >= 0)
    fail ("cmp(x,y)", x);

  x = may_parse_str ("x");
  y = may_parse_str ("x^2");
  if (may_cmp (x, y) <= 0)
    fail ("cmp(x,x^2)", x);

  x = may_parse_str ("x^2*y");
  y = may_parse_str ("x^2*z");
  if (may_cmp (x, y) >= 0)
    fail ("cmp(x^2*y,x^2*z)", x);

  x = may_parse_str ("x^2");
  y = may_parse_str ("x^2*z");
  /*if (may_cmp (x, y) <= 0)
    fail ("cmp(x^2,x^2*z)", x); */
  if (may_cmp (x, y) > 0)
    printf ("Bug fixed. Updated test please!\n");

  may_keep (NULL);
}

void test_get_name ()
{
  may_mark ();

  may_t x = may_set_str ("x");

  if (strcmp (may_get_name(x), "IDENTIFIER") != 0)
    fail ("STRING ID", x);

  x = may_parse_str ("exp(x)");
  if (strcmp (may_get_name(x), "exp") != 0)
    fail ("exp", x);

  x = may_parse_str ("f(x)");
  if (strcmp (may_get_name(x), "f") != 0)
    fail ("exp", x);

  may_keep (NULL);
}

void test_add_c (void)
{
  may_t x, y, z;
  may_mark_t mark1, mark2;

  (may_mark) (mark1);
  x = may_set_str ("x");
  y = may_set_str ("y");
  (may_mark) (mark2);
  z = may_add_c (x, y);
  z = may_add_c (z, x);
  z = may_add_c (z, y);
  z = may_add_c (z, z);
  z = may_eval (z);
  z = (may_compact) (mark2, z);
  check (z, "4*x+4*y");
  check (x, "x");
  check (y, "y");

  z = may_add_c (x, y);
  z = may_add_c (z, z);
  may_t z1 = may_add_c (may_set_ui (0), z);
  z = may_add_c (z, z);
  z = may_eval (z);
  z1 = may_eval (z1);
  check (z, "4*x+4*y");
  check (z1, "2*x+2*y");
  check (x, "x");
  check (y, "y");

  (may_compact) (mark1, NULL);
}

void test_error_handler (void)
{
  may_error_e e;
  const char *s;

  may_mark ();

  MAY_TRY {
    MAY_THROW (MAY_DIMENSION_ERR);
    e = MAY_NO_ERR;
  } MAY_CATCH {
    e = MAY_DIMENSION_ERR;
  } MAY_ENDTRY;

  check_bool ( e == MAY_DIMENSION_ERR);
  may_error_get (&e, NULL);
  check_bool ( e == MAY_DIMENSION_ERR);
  may_error_get (NULL, &s);
  check_bool (s == NULL);
  check_bool (strcmp (may_error_what (e), "Dimension") == 0);

  may_keep (NULL);
}


void test_domain ()
{
  may_t x, y;

  may_mark ();

  // test identifer domain
  x = may_set_str_domain ("x", MAY_INT_NONNEG_D);
  y = may_eval (may_floor_c (x));
  if (may_identical (x, y))
    fail ("domain integer", y);
  y = may_eval (may_sign_c (x));
  if (!may_one_p (y))
    fail ("domain nonneg", y);

  // Test func domain
  x = may_func_domain_c ("f", x, MAY_INT_NONPOS_D);
  y = may_eval (may_floor_c (x));
  if (may_identical (x, y))
    fail ("domain integer 2", y);
  y = may_eval (may_sign_c (x));
  if (may_identical (y, may_set_si (-1)) != 0)
    fail ("domain nonneg 2", y);

  may_keep (NULL);
}


may_t my_f (may_t x)
{
  return may_neg_c (x);
}

may_t my_exp (may_t x)
{
  return may_func_c ("exp2", x);
}

void test_subs (void)
{
  may_t x, y, z, ref;
  const char *name [3];
  const void *val[3];

  may_mark ();
  x = may_parse_str ("x+y");
  name[0] = "x";
  val[0] = x;

  z = may_parse_str ("x+y+exp(x)*log(y)");
  y = may_eval (may_subs_c (z, 1, 1, name, val));
  ref = may_parse_str ("x + 2*y + exp(x + y) * log(y)");
  if (may_identical (ref, y))
    fail ("subs1", y);

  z = may_parse_str ("x+y+exp(x)*log(y)");
  y = may_eval (may_subs_c (z, 2, 1, name, val));
  ref = may_parse_str ("x + 3*y + exp(x + 2*y) * log(y)");
  if (may_identical (ref, y))
    fail ("subs2", y);

  name[1] = "f";
  val[1] = my_f;

  z = may_parse_str ("f(x)+x");
  y = may_subs_c (z, 1, 2, name, val);
  y = may_eval (y);
  ref = may_set_ui (0);
  if (may_identical (ref, y))
    fail ("subs3", y);

  name[0] = "f";
  val[0]  = my_f;
  name[1] = "exp";
  val[1]  = my_exp;
  z = may_parse_str ("f(exp(f(x)))+exp(x)");
  y = may_subs_c (z, 1, 2, name, val);
  y = may_eval (y);
  ref = may_parse_str ("exp2(x)-exp2(-x)");
  if (may_identical (ref, y))
    fail ("subs3", y);

  name[0] = "f";
  val[0]  = my_f;
  name[1] = "exp";
  val[1]  = my_exp;
  z = may_parse_str ("cos(x)+rphi(2)");
  y = may_subs_c (z, 1, 2, name, val);
  y = may_eval (y);
  if (may_identical (z, y))
    fail ("subs4", y);

  name[0] = "f";
  val[0]  = my_f;
  name[1] = "exp";
  val[1]  = my_exp;
  z = may_parse_str ("rphi(f(2))+exp(x)");
  y = may_subs_c (z, 2, 2, name, val);
  y = may_eval (y);
  ref = may_parse_str ("rphi(-2)+exp2(x)");
  if (may_identical (ref, y))
    fail ("subs5", y);

  z = may_parse_str ("rphi(f(2))+exp(x)");
  name[0] = "f";
  val[0]  = z;
  name[1] = "exp";
  val[1]  = z;
  name[2] = "x";
  val[2]  = my_exp;
  y = may_subs_c (z, 2, 3, name, val);
  y = may_eval (y);
  if (may_identical (z, y))
    fail ("subs6", y);

  x = may_replace (may_parse_str ("x+y+sin(2*z)"),
                   may_set_str ("x"),
                   may_set_str ("y"));
  ref = may_parse_str ("2*y+sin(2*z)");
  if (may_identical (ref, x))
    fail ("replace", x);

  x = may_parse_str ("1/((-1+x)^(1/2))");
  x = may_replace (x, may_set_str ("x"), may_set_ui(0));
  check (x, "-I");

  name[0] = "x";
  val[0]  = may_set_ui (2);
  y = may_subs_c (may_parse_str ("f(x)"), 1, 1, name, val);
  y = may_eval (y);
  check (y, "f(2)");

  may_keep (NULL);
}

void test_evalf (void)
{
  may_t x;

  may_mark ();
  may_kernel_num_presimplify (0);

  x = may_parse_str ("PI+10.2+2+2/5+x");
  x = may_evalf (x);
  may_kernel_num_presimplify (1);
  check (x, "1.57415926535897932384626433832795025E1+x");

  x = may_parse_str ("23*x^2");
  x = may_evalf (x);
  check (x, "2.3E1*x^2");

  may_keep (NULL);
}

void test_trig2exp (void)
{
  may_t x, y;

  may_mark ();

  x = may_parse_str ("cos(x)+I*sin(x)");
  y = may_trig2exp (x);
  check (y, "exp(I*x)");

  x = may_parse_str ("cosh(x)+sinh(x)");
  y = may_trig2exp (x);
  check (y, "exp(x)");

  x = may_parse_str ("tan(I*x)-I*tanh(x)");
  y = may_rationalize (may_eexpand (may_trig2exp (x)));
  check (y, "0");

  x = may_parse_str ("acos(x)+asin(x)");
  y = may_trig2exp (x);
  check (y, "-I*log((1-x^2)^(1/2)+I*x)-I*log(I*(1-x^2)^(1/2)+x)");
  //check (y, "PI/2");

  x = may_parse_str ("(acosh(x)+asinh(x))");
  y = may_trig2exp (x);
  check (y, "ln(sqrt(x^2+1)+x)+ln(sqrt(x^2-1)+x)");


  x = may_parse_str ("asin(2)");
  y = may_trig2exp (x);
  check (y, "-I*log(2*I+(-3)^(1/2))");

  x = may_parse_str ("atan(x)+atanh(x)");
  y = may_trig2exp (x);
  // normal form ?
  //check (y, "1/2*ln(-(x+1))+I/2*ln(-(x+I))-1/2*ln(x-1)-I/2*ln(x-I)");

  may_keep (NULL);
}


static const char *const
table[] = {
  "", "NAN",
  "1*2*34*5*6*7*8*9*10*11*12*13*14*15*16*17*18*19*20", "6893222356500480000",
  "1/2+1/3+1/4+1/5+1/6+1/7+1/8+1/9+1/10","4861/2520",
  "2*INFINITY-3", "INFINITY",
  "1/2+0.5", "1.",
  "1+2*I", "1+2*I",
  "1-I", "1-I",
  "2/-1", "-2",
  "1/-4", "-1/4",
  "(3/2)/(2/3)", "9/4",
  "3+3*I*I", "0",
  "NAN^2", "NAN",
  "(NAN+I)^2", "NAN+NAN*I",
  "NAN^0", "1",
  "0^NAN", "NAN",
  "NAN^x", "NAN",
  "0*x", "0",
  "0.0*x", "0.",
  "0^x", "0^x",
  "0^-2", "NAN",
  "0^(-2/3)", "NAN",
  "0^(-2.3)", "NAN",
  "0^(-2/3+I)", "NAN",
  "0^(-2+I)", "NAN",
  "0^I", "NAN",
  "1/0", "NAN",
  "0^(1+I)", "0",
  "2^(2.0)", "4.",
  "(2/3)^(-1)", "3/2",
  "argument(2)", "0",
  "(1/2)^2.0", "2.5E-1",
  "2^(3+2*I)", "2^(3+2*I)",
  "(-1)^(1/2)", "I",
  "-z^2", "-z^2",
  "(5/6)^(4/3)", "5/6*(5/6)^(1/3)",
  "(1+I)/(2+I)-I/(2+I)", "2/5-1/5*I",
  "sin(10^615*PI)", "0",
  "x^b/x^b", "1",
  "x^b*x^(-b)", "1",
  "x^x^x", "(x^x)^x",
  "x^(x^2)", "x^(x^2)",
  "x+2+3-4-5-6/3+2+3", "-1+x",
  "0^0", "1",
  "2/x/y", "2/(y*x)",
  "x/y", "x/y",
  "2+(-3)^(1/2)", "2+(-3)^(1/2)",
  "2+(-3/5)^(1/2)", "2+(-3/5)^(1/2)",
  "x-PI/2", "x-1/2*PI",
  "log(0)", "NAN",
  "log(0.0)", "-INFINITY",
  "log(I)", "1/2*I*PI",
  "log(-I)", "-1/2*I*PI",
  "cos(acos(x))", "x",
  "sin(asin(x))", "x",
  "tan(atan(x))", "x",
  "exp(log(x))", "x",
  "acos(cos(x))", "acos(cos(x))",
  "asin(sin(x))", "asin(sin(x))",
  "atan(tan(x))", "atan(tan(x))",
  "log(exp(x))", "log(exp(x))",
  "0.0", "0.",
  "x+x+x+x+x+-x-x-x-x", "x",
  "2^(-(3/2))", "1/4*2^(1/2)",
  "-sin(-x)", "sin(x)",
  "-tan(-2*x)", "tan(2*x)",
  "sin(2*PI)", "0",
  "sin(PI)", "0",
  "sin(PI/2)", "1",
  "sin(-PI/2)", "-1",
  "sin(PI/6)", "1/2",
  "sin(2^1000*PI)", "0",
  "tan(0)", "0",
  "tan(PI)", "0",
  "tan(asin(x))", "x/(1-x^2)^(1/2)",
  "tan(acos(x))", "(1-x^2)^(1/2)/x",
  "tan(PI/4)", "1",
  "tan(-PI/4)", "-1",
  "tan(PI/2)", "NAN",
  "tan(PI/3)", "3^(1/2)",
  "tan(PI/6)", "1/3*3^(1/2)",
  "tan(PI/12)", "2-3^(1/2)",
  "tan(5*PI/12)", "2+3^(1/2)",
  "tan(7*PI/12)", "-2-3^(1/2)",
  "tan(PI/5)", "(5-2*5^(1/2))^(1/2)",
  "tan(PI/10)", "1/5*(25-10*5^(1/2))^(1/2)",
  "tan(11*PI/21)", "-tan(10/21*PI)",
  "tan((2+3*I)*PI)", "tan((2+3*I)*PI)",
  "tan(-x)", "-tan(x)",
  "2*-PI", "-2*PI",
  "abs(x)","abs(x)",
  "abs(-2*x)", "2*abs(x)",
  "abs(-2)", "2",
  "abs(-2+2*I)", "8^(1/2)",
  "abs(-17.0)", "1.7E1",
  "abs(-17/42)", "17/42",
  "abs(acos(1/20))", "acos(1/20)",
  "abs(acos(20))", "abs(acos(20))",
  "conj(1+I)", "1-I",
  "conj(sqrt(2))", "2^(1/2)",
  "cosh(0)", "1",
  "cosh(1E-40)", "1.",
  "cosh(1E-40+1E-40*I)", "1.",
  "cosh(acosh(x^(1/2)))^2", "x",
  "cosh(-2*x)", "cosh(2*x)",
  "cosh(I*PI)", "-1",
  "cosh(x+y)", "cosh(y+x)",
  "sinh(0)", "0",
  "sinh(asinh(x^(1/2)))^2", "x",
  "sinh(-2*x)", "-sinh(2*x)",
  "sinh(I*PI)", "0",
  "sinh(x+y)", "sinh(y+x)",
  "sinh(I*x)", "I*sin(x)",
  "sinh(-I*x)", "-I*sin(x)",
  "tanh(0)", "0",
  "tanh(atanh(x^(1/2)))^2", "x",
  "tanh(-2*x)", "-tanh(2*x)",
  "tanh(2*I*PI)", "0",
  "tanh(x+y)", "tanh(y+x)",
  "asinh(x)", "asinh(x)",
  "asinh(0)", "0",
  "asinh(I*y)", "I*asin(y)",
  "asinh(sinh(t))", "t",
  "asinh(-x)", "-asinh(x)",
  "acosh(x)", "acosh(x)",
  "acosh(0)", "1/2*I*PI",
  "acosh(1)", "0",
  "acosh(2-3)", "I*PI",
  "acosh(cosh(x))", "acosh(cosh(x))",
  "atanh(0)", "0",
  "atanh(1)", "INFINITY",
  "atanh(-1)", "-INFINITY",
  "atanh(-2*x)", "-atanh(2*x)",
  "atanh(I*x)", "I*atan(x)",
  "atanh(tanh(x))", "atanh(tanh(x))",
  "1^320", "1",
  "0^300", "0",
  "(-x)^2", "x^2",
  "(2*sqrt(2))^2", "8",
  //"[1,2;3,4]^2", "[7,10;15,22]",
  "exp(2*log(x))", "x^2",
  "exp(0)", "1",
  "log(1)", "0",
  "log(exp(2))", "2",
  "floor(x)", "floor(x)",
  "floor(2.3)", "2",
  "floor(NAN)", "NAN",
  // "floor(log(9)/log(3))", "2",
  "ceil(x)", "-floor(-x)",
  "ceil(2.3)", "3",
  "sign(sqrt(2))", "1",
  "sign(x)", "sign(x)",
  "sign(1+2*I)", "1",
  "sign(2*I)", "1",
  "sign(-2*I)", "-1",
  "sign(-1+2*I)", "-1",
  "abs(sqrt(2)-2)", "2-2^(1/2)",
  //"2+[1,1;1,1]+[0,0;0,1]","[3,1;1,4]",
  //"2*[1,1;1,1]*[1,0;0,1]", "[2,2;2,2]",
  "x+0", "x",
  "0+x", "x",
  "x*1", "x",
  "1*x", "x",
  "x-0", "x",
  "0-x", "-x",
  "-0", "0",
  "x/1", "x",
  "(1+I)^-x", "1/((1+I)^x)",
  "mod(x,y)", "mod(x,y)",
  "gcd(x,y)", "gcd(x,y)",
  "conj(x)", "conj(x)",
  "real(x)", "real(x)",
  "imag(x)", "imag(x)",
  "argument(x)", "argument(x)",
  "argument(real(x))", "argument(real(x))",
  "argument(0)", "NAN",
  "argument(-1)", "PI",
  "argument(abs(x))", "argument(abs(x))",
  "argument(2*x)", "argument(x)",
  "argument(1+I)", "1/4*PI",
  "argument(1-I)", "-1/4*PI",
  "x!", "gamma(1+x)",
  "5!", "120",
  "gamma(6.)", "1.2E2",
  "gamma(0)", "INFINITY",
  "RANGE(2,3)", "RANGE(2.,3.)",
  "384/256", "3/2",
  "1+1/(2+1/(2+1/(2+1/(2+1/2))))", "99/70",
  "(8/125)^(2/3)", "4/25",
  "(9/127)^(2/3)", "(9/127)^(2/3)",
  "(-3)^x", "(-3)^x",
  "(-3.0)^x", "(-3.)^x",
  "(-INFINITY)^x", "(-INFINITY)^x",
  //"(-4)^(1/2)", "2*I",
  "{1}", "{1}",
  //"[1]", "[1]",
  "diff(asin(x)+acos(x)+atan(x)+asinh(x)+acosh(x)+atanh(x),y)", "0",
  "diff(tanh(x)+sinh(x)+cosh(x)+abs(x),z)", "0",
  "diff(diff(f(x),x,-1),x,1)", "f(x)",
  "diff(f(y),x)", "0",
  "diff(f(x),x)+diff(f(x),x)", "2*diff(f(_L0),_L0,1,x)",
  "diff(diff(f(x,y),x,1,y),y)", "diff(diff(f(_L0,_L1),_L0,1,_L1),_L1,1,y)",
  "diff(gamma(x),x)", "diff(gamma(_L0),_L0,1,x)",
  "5+sqrt(5)*(-sqrt(5))", "0",
  "5+sqrt(5)*(-sqrt(5))*sqrt(5)*sqrt(5)", "-20",
  "cos(0)", "1",
  "cos(PI)", "-1",
  "cos(2*PI)", "1",
  "cos(PI+x)", "-cos(x)",
  "cos(asin(x))", "(1-x^2)^(1/2)",
  "cos(atan(x))", "1/(1+x^2)^(1/2)",
  "cos(PI/2)", "0",
  "cos(3*PI/2)", "0",
  "cos(-PI/2)", "0",
  "cos(PI/4)", "1/2*2^(1/2)",
  "cos(PI/3)", "1/2",
  "cos(PI/6)", "1/2*3^(1/2)",
  "cos(50*PI/7)", "-cos(1/7*PI)",
  "cos(-53*PI/7)", "cos(3/7*PI)",
  "cos(5*PI/7)", "-cos(2/7*PI)",
  "cos(-3*x)", "cos(3*x)",
  "cos(-x)", "cos(x)",
  "cos(3*PI-x)", "-cos(x)",
  "cos((1+I)*PI)", "cos((1+I)*PI)",
  "sin(acos(x))", "(1-x^2)^(1/2)",
  "sin(atan(x))", "x/(1+x^2)^(1/2)",
  "sin(PI/3)", "1/2*3^(1/2)",
  "sin(2*PI/3)", "1/2*3^(1/2)",
  "sin(5*PI/3)", "-1/2*3^(1/2)",
  "sin(-PI/4)", "-1/2*2^(1/2)",
  "sin(-7*PI/6)", "1/2",
  "sin(PI/17)", "sin(1/17*PI)",
  "sin(16*PI/17)", "sin(1/17*PI)",
  "sin((1+I)*PI)", "sin((1+I)*PI)",
  "sin(-2*x)", "-sin(2*x)",
  "sin(x+3*PI)", "-sin(x)",
  "asin(0)", "0",
  "asin(1)", "1/2*PI",
  "asin(-1)", "-1/2*PI",
  "asin(1/2)", "1/6*PI",
  "asin(-1/2)", "-1/6*PI",
  "asin(-x)", "-asin(x)",
  "asin(-2*x)", "-asin(2*x)",
  "asin(-sqrt(2)/2)", "-1/4*PI",
  "asin(sqrt(3)/3)", "1/3*PI",
  "asin(I*x)", "I*asinh(x)",
  "asin(sin(sqrt(2)))", "2^(1/2)",
  "acos(0)", "1/2*PI",
  "acos(1)", "0",
  "acos(-1)", "PI",
  "acos(1/2)", "1/3*PI",
  "acos(-1/2)", "2/3*PI",
  "acos(-2*x)", "-acos(2*x)+PI",
  "acos(sqrt(2)/2)", "1/4*PI",
  "acos(sqrt(3)/3)", "1/6*PI",
  "acos(sin(x))", "-asin(sin(x))+1/2*PI",
  "acos(I*x)", "-I*asinh(x)+1/2*PI",
  "atan(0)", "0",
  "atan(1)", "1/4*PI",
  "atan(-1)", "-1/4*PI",
  "atan(NAN)", "NAN",
  "atan(INFINITY)", "1/2*PI",
  "atan(-2*x)", "-atan(2*x)",
  "atan(sqrt(3)/3)", "1/6*PI",
  "atan(sqrt(3))", "1/3*PI",
  "atan(I*x)", "I*atanh(x)",
  "exp(I*PI)", "-1",
  "exp(4*I*PI)", "1",
  "exp(I*PI/2)", "I",
  "exp(-I*PI/2)", "-I",
  "log(2*I)", "log(2*I)",
  "(1+I)^-2", "-1/2*I",
  "(1+I)^3", "-2+2*I",
  "-x^2", "-x^2",
  "abs(real(x))^2", "real(x)^2",
  "abs(x^2)", "abs(x)^2",
  "abs(y^(-3))", "1/abs(y)^3",
  "log(abs(exp(x)))", "real(x)",
  "gamma(3/2)", "1/2*PI^(1/2)",
  "gamma(-3/2)", "4/3*PI^(1/2)",
  "gamma(-5/2)", "-8/15*PI^(1/2)",
  "conj(real(x))", "real(x)",
  "real(real(x))", "real(x)",
  "imag(real(x))", "0",
  "real(2)", "2",
  "real(2+I)", "2",
  "real(x+2*y)", "2*real(y)+real(x)",
  "imag(2)", "0",
  "imag(2+3*I)", "3",
  "abs(abs(x))", "abs(x)",
  "(PI^2)^(1/2)", "PI",
  "(abs(x)^2)^(1/2)", "abs(x)",
  "(real(x)^3)^(1/3)", "real(x)",
  "(real(x)^2)^(1/3)", "real(x)^(2/3)",
  "abs(-2*x+2*y+1)-abs(2*x-2*y-1)", "0",
  "(a-I*b)/(I*b-a)", "-1",
  "2*((5)^(1/2))*2*((5)^(1/2))*2*((5)^(1/2))", "40*5^(1/2)",
  "((-1)^(1/4))", "(-1)^(1/4)",
  "((-1)^(1/4))*((-1)^(1/4))", "I",
  "(-4)^(1/2)", "2*I",
  "(8)^(1/2)", "8^(1/2)",
  "(-8)^(1/3)", "2*(-1)^(1/3)",
  "gcd(1,1)", "1",
  "gcd(1,2)", "1",
  "gcd(2,4)", "2",
  "gcd(2*I,3*I)", "I",
  "gcd(2*I,4*I)", "2*I",
  "gcd(x^2-1,x-1)", "-1+x",
  "gcd(x,y*x)", "x",
  "x^2.2/x^2.2", "1",
  "(1+2*I)^0", "1",
  "-1/2*(sin(3*x)-3*sin(x))/(1+cos(2*x))", "-1/2*(sin(3*x)-3*sin(x))/(1+cos(2*x))",

  /* Float checks: the values depends on the version of MPFR since
     the number of output digits has changed in 2.2.0 */
#if MPFR_VERSION_MAJOR == 2 && MPFR_VERSION_MINOR == 1
  "RANGE(2^(1/2),2^(1/2))", "RANGE(1.4142135623730950488016887242096979,1.4142135623730950488016887242096982)",
  "1.2/3.7", "3.2432432432432432432432432432432429E-1",
  "exp(2+3.*I)","-7.3151100949011025174865361510507896+1.0427436562359044141015039404625522*I",
  "log(2+3.*I)","1.2824746787307683680267437207826593+9.8279372324732906798571061101466604E-1*I",
  "sin(2+3.*I)", "9.154499146911429573467299544609833-4.1689069599665643507548130588537546*I",
  "cos(2+3.*I)","-4.1896256909688072301325550196159741-9.1092278937553365979791972627788632*I",
  "tan(2+3.*I)","-3.7640256415042482927512211303227444E-3+1.003238627353609801446358597821927*I",
  "asinh(1.2)", "1.0159731341796921059048534979380653",
  "asinh(1.2+3.2*I)", "1.9058866455715606336280558485459656+1.197243091803257231410227219545494*I",
  "acosh(1.4)", "8.6701472649056510395052026950146732E-1",
  "acosh(1.4+4.5*I)", "2.2526192625763138027487078248734492+1.2753900000787101262297184256686566*I",
  "atanh(-0.1)", "-1.0033534773107558063572655206003895E-1",
  "atanh(-0.1+0.6*I)", "-7.3517967637637984645116830248276413E-2+5.4367466261384881842327534258048541E-1*I",
  "exp(1.0)", "2.7182818284590452353602874713526623",
  "sinh(1E-50)", "1.E-50",
  "sinh(1E-50+1E-50*I)", "1.E-50*I",
  "tanh(1E-50)", "1.E-50",
  "tanh(1E-50+1E-50*I)", "1.E-50*I",

  "sin(1.7)", "9.9166481045246861534613339864787565E-1",
  "cos(1.7)", "-1.2884449429552468408764285733487348E-1",
  "tan(1.7)", "-7.6966021394591584141281929682986635",
  "asin(0.7)", "7.7539749661075306374035335271498708E-1",
  "acos(1.7)", "1.1232309825872958895311457962279851*I",
  "atan(1.7)", "1.0390722595360910276212503379072788",
  "sin(1.7+4.2*I)", "3.3072679208657454569377530459261426E1-4.2951172509399229694833898673576216*I",
  "cos(1.7+4.2*I)", "-4.2970493484520271114811373030444805-3.3057808622812150112857309047614557E1*I",
  "tan(1.7+4.2*I)", "-1.1497567357419482120178232897858386E-4+1.0004348905841315761149593339884275*I",
  "asin(1.7+4.2*I)", "3.7635245236802485059440201885598032E-1+2.2128286516457570924134170303950604*I",
  "acos(1.7+4.2*I)", "1.1944438744268717686369196727837749-2.2128286516457570924134170303950585*I",
  "atan(1.7+4.2*I)", "1.4846144872431840786280396661078955+2.0599040838747568650529663049941053E-1*I",
  "sin(1.2*PI)", "-5.8778525229247312916870595463907264E-1",
  "cos(1.2*PI)", "-8.0901699437494742410229341718281918E-1",
  "tan(1.2*PI)", "7.2654252800536088589546675748061844E-1",
#else /* MPFR >= 2.2.0 */
  "RANGE(2^(1/2),2^(1/2))", "RANGE(1.41421356237309504880168872420969798,1.41421356237309504880168872420969818)",
  "1.2/3.7", "3.24324324324324324324324324324324288E-1",
  "exp(2+3.*I)","-7.31511009490110251748653615105078963+1.04274365623590441410150394046255222*I",
  "log(2+3.*I)","1.28247467873076836802674372078265928+9.82793723247329067985710611014666038E-1*I",
  "sin(2+3.*I)", "9.154499146911429573467299544609833-4.16890695996656435075481305885375463*I",
  "cos(2+3.*I)","-4.18962569096880723013255501961597415-9.10922789375533659797919726277886318*I",
  "tan(2+3.*I)","-3.76402564150424829275122113032274436E-3+1.00323862735360980144635859782192701*I",
  "asinh(1.2)", "1.01597313417969210590485349793806525",
  "asinh(1.2+3.2*I)", "1.90588664557156063362805584854596555+1.19724309180325723141022721954549401*I",
  "acosh(1.4)", "8.67014726490565103950520269501467322E-1",
  "acosh(1.4+4.5*I)", "2.25261926257631380274870782487344923+1.27539000007871012622971842566865664*I",
  "atanh(-0.1)", "-1.00335347731075580635726552060038947E-1",
  "atanh(-0.1+0.6*I)", "-7.35179676376379846451168302482764132E-2+5.43674662613848818423275342580485406E-1*I",
  "exp(1.0)", "2.71828182845904523536028747135266231",
  "sinh(1E-50)", "1.00000000000000000000000000000000005E-50",
  "sinh(1E-50+1E-50*I)", "1.00000000000000000000000000000000005E-50*I",
  "tanh(1E-50)", "1.00000000000000000000000000000000005E-50",
  "tanh(1E-50+1E-50*I)", "1.00000000000000000000000000000000005E-50*I",

  "sin(1.7)", "9.9166481045246861534613339864787565E-1",
  "cos(1.7)", "-1.28844494295524684087642857334873477E-1",
  "tan(1.7)", "-7.69660213945915841412819296829866347",
  "asin(1.7)", "1.5707963267948966192313216916397514-1.12323098258729588953114579622798547*I",
  "acos(1.7)", "1.12323098258729588953114579622798509*I",
  "atan(1.7)", "1.0390722595360910276212503379072788",
  "sin(1.7+4.2*I)", "3.30726792086574545693775304592614255E1-4.29511725093992296948338986735762155*I",
  "cos(1.7+4.2*I)", "-4.29704934845202711148113730304448055-3.30578086228121501128573090476145573E1*I",
  "tan(1.7+4.2*I)", "-1.14975673574194821201782328978583855E-4+1.00043489058413157611495933398842749*I",
  "asin(1.7+4.2*I)", "3.7635245236802485059440201885598032E-1+2.21282865164575709241341703039506042*I",
  "acos(1.7+4.2*I)", "1.19444387442687176863691967278377493-2.21282865164575709241341703039505849*I",
  "atan(1.7+4.2*I)", "1.48461448724318407862803966610789549+2.0599040838747568650529663049941053E-1*I",
#endif

};

void test_table ()
{
  may_t x;
  char *s;
  int i;
  may_mark_t mark;

  /* Most of theses tests only works with NORMAL model */
  may_mark (mark);
  for (i = 0 ; i < numberof (table) ; i+=2) {
    //printf ("table[i]=%s\n", table[i]);
    x = may_parse_str (table[i]);
    s = may_get_string (NULL, 0, x);
    if (strcmp (s, table[i+1]) != 0) {
      printf ("%s: error in the reference table %d.\n"
	      "Input   : %s\n"
	      "Expected: %s\n"
	      "Got     : %s\n",
	      may_get_version (), i, 
	      table[i], table[i+1], s);
      test_fail ();
    }
    may_compact (mark, NULL);
  }
}

/* We don't check that the string is exactly the same for this table */
static const char *const
table2[] = {
  "-z*-x*-y", "-y*z*x",
  "-z*-x*-y*-a*-b*-d", "d*b*y*z*a*x",
  "1+y*-x^-x", "1-y/x^x",
  "1/2*s-1/2*c^2/s+1/-4*c*s2/s^2", "1/2*s-1/2*c^2/s-1/4*c*s2/s^2",
  "x+(y+z)", "y+z+x",
  "log(-22/11)", "log(2)+I*PI",
  "atan(tan(20*sqrt(2)))", "20*2^(1/2)-9*PI",
  "atan(tan(17*sqrt(2)))", "17*2^(1/2)-8*PI",
  "acos(cos(20*sqrt(2)))", "-20*2^(1/2)+10*PI",
  "acos(cos(18*sqrt(2)))", "18*2^(1/2)-8*PI",
  "gcd(2*x,y*x)", "gcd(2,y)*x",
  "x+-x*y^z", "-y^z*x+x",
  "(y + 2*z)^2*(2*y+4*z)", "2*(y+2*z)^3",
  "(y + 2.*z)^2*(2*y+4*z)", "2*(y+2*z)*(y+2.*z)^2",
  "(y + 2*z)^2*(2*y+4.*z)", "(y+2*z)^2*(2*y+4.*z)",
  "(x+2-1/2*x/y^z)/(x*z^(-2*z))", "(z^z)^2*(2+x-1/2*x/y^z)/x",
  "sqrt(1/z)-1/sqrt(z)", "(1/z)^(1/2)-1/z^(1/2)",
  "(x*y*z)^3", "y^3*z^3*x^3",
  "asin(sin(20*sqrt(2)))", "-20*2^(1/2)+9*PI",
  "asin(sin(18*sqrt(2)))", "18*2^(1/2)-8*PI",
  "asin(cos(x))", "-acos(cos(x))+1/2*PI",
  "diff(f(x)*x+2*sin(y)/tan(z)+cos(z)^2+exp(z)*log(y),x)","f(x)+diff(f(_L0),_L0,1,x)*x",
  "diff(u(x)^v(x),x)", "(v(x)*diff(u(_L0),_L0,1,x)/u(x)+diff(v(_L0),_L0,1,x)*log(u(x)))*u(x)^v(x)",
  "cos(PI/12)", "1/4*(1+1/3*3^(1/2))*6^(1/2)",
  "cos(5*PI/12)", "1/4*(1-1/3*3^(1/2))*6^(1/2)",
  "cos(7*PI/12)", "-1/4*(1-1/3*3^(1/2))*6^(1/2)",
  "cos(11*PI/12)", "-1/4*(1+1/3*3^(1/2))*6^(1/2)",
  "sin(PI/12)", "1/4*(1-1/3*3^(1/2))*6^(1/2)",
  "sin(5*PI/12)", "1/4*(1+1/3*3^(1/2))*6^(1/2)",
  "log(I)", "1/2*I*PI",
  "log(-I)", "-1/2*I*PI",
  "sqrt(exp(z)) - exp(z/2)", "-exp(1/2*z)+exp(z)^(1/2)",
  "(x*y)^(1/n) - x^(1/n)*y^(1/n)", "-y^(1/n)*x^(1/n)+(y*x)^(1/n)",
  "conj(x+y*z^2)", "conj(y)*conj(z)^2+conj(x)",
  "imag(x+2*y)", "2*imag(y)+imag(x)",
  "cosh(asinh(x))", "(1+x^2)^(1/2)",
  "cosh(atanh(x))", "1/(1-x^2)^(1/2)",
  "sinh(acosh(x))", "(1+x)^(1/2)*(-1+x)^(1/2)",
  "sinh(atanh(x))", "x/(1-x^2)^(1/2)",
  "tanh(acosh(x))", "(1+x)^(1/2)*(-1+x)^(1/2)/x",
  "tanh(asinh(x))", "x/(1+x^2)^(1/2)",
  "abs(exp(PI*sqrt(163))-262537412640768744)", "262537412640768744-exp(163^(1/2)*PI)",
  "(a-b*I)^3", "-(I*b-a)^3",
  "(x-y)^2*(-2*x+2*y)", "2*(y-x)^3",
  "abs(-2*x+2*y)", "2*abs(y-x)",
  "(2.3*x+1.3)*y", "(2.3*x+1.3)*y",
  "sinh(I*x+I*y)", "I*sin(x+y)",
  "sinh(I*x+I*y+2*I)", "I*sin(x+y+2)",
  "sinh(I*x+I*y-2)", "sinh(I*x+I*y-2)",
  "cosh(I*x+I*y)", "cos(x+y)",
  "cosh(I*x+I*y+2*I)", "cos(x+y+2)",
  "cosh(I*x+I*y-2)", "cosh(I*x+I*y-2)"
  };

// Test the table without being too strict about the order of the sum
void test_table2 ()
{
  may_t x;
  int i;
  may_mark_t mark;

  may_mark (mark);
  for (i = 0 ; i < numberof (table2) ; i+=2) {
    x = may_parse_str (table2[i]);
    check (x, table2[i+1]);
    may_compact (mark, NULL);
  }
}

void test_extract_pi_factor ()
{
  may_t s, x;

  may_mark ();
  s = may_parse_str ("x+y+2*PI");

  x = may_cos (s);
  check (x, "cos(y+x)");
  check (s, "y+x+2*PI");

  x = may_sin (s);
  check (x, "sin(y+x)");
  check (s, "y+x+2*PI");

  x = may_tan (s);
  check (x, "tan(y+x)");
  check (s, "y+x+2*PI");

  x = may_set_str ("x");
  s = may_asin (x);
  check (s, "asin(x)");
  s = may_acos (x);
  check (s, "acos(x)");
  s = may_atan (x);
  check (s, "atan(x)");
  s = may_asinh (x);
  check (s, "asinh(x)");
  s = may_acosh (x);
  check (s, "acosh(x)");
  s = may_atanh (x);
  check (s, "atanh(x)");
  s = may_sinh (x);
  check (s, "sinh(x)");
  s = may_cosh (x);
  check (s, "cosh(x)");
  s = may_tanh (x);
  check (s, "tanh(x)");

  may_keep (NULL);
}

void test_ifactor (void)
{
  may_mark ();

  may_t n = may_set_ui (76931800UL);
  may_t l = may_naive_ifactor (n);
  char *str = may_get_string (NULL, 0, l);
  if (strcmp (str, "{3,2,2,5,3,11,2,17}") != 0) {
    printf ("Error for ifactor\n Get: %s\n", str);
    test_fail ();
  }
  n = may_set_ui (3*10037);
  l = may_naive_ifactor (n);
  str = may_get_string (NULL, 0, l);
  if (strcmp (str, "{1,3,1,10037}") != 0) {
    printf ("Error for ifactor (2)\n Get: %s\n", str);
    test_fail ();
  }

  may_keep (NULL);
}


static int (*const predicate_cb[]) (may_t) = {
  may_zero_p, may_zero_fastp, may_nonzero_p, may_one_p, may_pos_p, may_neg_p,
  may_real_p, may_integer_p, may_even_p, may_odd_p, may_nan_p, may_inf_p,
  may_num_p, may_purenum_p, may_purereal_p, may_undef_p,
  may_nonneg_p, may_nonpos_p, may_cinteger_p, may_rational_p, may_crational_p,
  may_posint_p, may_nonnegint_p, may_negint_p, may_nonposint_p,
  may_prime_p
};
static const struct {
  const char *x;
  const char flag[32];
} predicate_table[] = {
  {"0",         "11000011100011101111101010"},
  {"0.",        "11000010000011101100000000"},
  {"1",         "00111011010011101011111000"},
  {"1.",        "00111010000011101000000000"},
  {"2",         "00101011100011101011111001"},
  {"-1",        "00100111010011100111100110"},
  {"xy",        "00000000000000000000000000"},
  {"NAN",       "00000010001011110000000000"},
  {"INFINITY",  "00101010000111101000000000"},
  {"sqrt(2)",   "00101000000010001000000000"},
  {"-sqrt(2)",  "00100100000010000100000000"},
  {"x+NAN",     "00000000000000010000000000"},
  {"1+I",       "00100000000011000010100000"},
  {"2/3",       "00101010000011101001100000"},
  {"2/3+I",     "00100000000011000000100000"},
  {"f(x)",      "00000000000000000000000000"},
  {"abs(x)",    "00000010000000001000000000"},
};

void test_predicate (void)
{
  may_t x;
  int i, j;

  may_mark ();
  for (i = 0; i < numberof (predicate_table); i++) {
    x = may_parse_str (predicate_table[i].x);
    for (j = 0; j < numberof (predicate_cb); j++) {
      int r = (*predicate_cb[j]) (x);
      if (predicate_table[i].flag[j] == '0' + (r != 0))
        continue;
      printf ("Error for predicate function %d, and entry %i\n"
              "Expected %c, got %d\n"
              "x=", j, i, predicate_table[i].flag[j], r);
      may_dump (x);
      test_fail ();
    }
  }

  x = may_parse_str ("1+x+f(y)+sin(h)");
  if (may_independent_p (x, may_set_str ("y")) != 0)
    fail ("independent_p (1)", x);
  if (may_independent_p (x, may_set_str ("t")) == 0)
    fail ("independent_p (2)", x);
  if (may_func_p (x, "f") == 0)
    fail ("func_p (1)", x);
  if (may_func_p (x, "g") != 0)
    fail ("func_p (2)", x);
  if (may_func_p (x, "sin") == 0)
    fail ("func_p (3)", x);

  x = may_parse_str ("1+x+y/exp(x)");
  if (may_exp_p (x, MAY_EXP_EXP_P) == 0)
    fail ("exp_p 1", x);
  if (may_exp_p (x, MAY_COS_EXP_P|MAY_SIN_EXP_P|MAY_TAN_EXP_P) != 0)
    fail ("exp_p 2", x);
  x = may_parse_str ("cos(x)");
  if (may_exp_p (x, MAY_EXP_EXP_P) != 0)
    fail ("exp_p 3", x);
  if (may_exp_p (x, MAY_COS_EXP_P|MAY_SIN_EXP_P|MAY_TAN_EXP_P) == 0)
    fail ("exp_p 4", x);
  x = may_parse_str("log(x)");
  if (may_exp_p (x, MAY_LOG_EXP_P) == 0)
    fail ("exp_p 5", x);
  x = may_parse_str("sin(x)");
  if (may_exp_p (x, MAY_SIN_EXP_P) == 0)
    fail ("exp_p 6", x);
  x = may_parse_str("tan(x)");
  if (may_exp_p (x, MAY_TAN_EXP_P) == 0)
    fail ("exp_p 7", x);
  x = may_parse_str("asin(x)");
  if (may_exp_p (x, MAY_ASIN_EXP_P) == 0)
    fail ("exp_p 8", x);
  x = may_parse_str("acos(x)");
  if (may_exp_p (x, MAY_ACOS_EXP_P) == 0)
    fail ("exp_p 9", x);
  x = may_parse_str("atan(x)");
  if (may_exp_p (x, MAY_ATAN_EXP_P) == 0)
    fail ("exp_p 10", x);
  x = may_parse_str("asinh(x)");
  if (may_exp_p (x, MAY_ASINH_EXP_P) == 0)
    fail ("exp_p 11", x);
  x = may_parse_str("acosh(x)");
  if (may_exp_p (x, MAY_ACOSH_EXP_P) == 0)
    fail ("exp_p 12", x);
  x = may_parse_str("atanh(x)");
  if (may_exp_p (x, MAY_ATANH_EXP_P) == 0)
    fail ("exp_p 13", x);
  x = may_parse_str("sinh(x)");
  if (may_exp_p (x, MAY_SINH_EXP_P) == 0)
    fail ("exp_p 14", x);
  x = may_parse_str("cosh(x)");
  if (may_exp_p (x, MAY_COSH_EXP_P) == 0)
    fail ("exp_p 15", x);
  x = may_parse_str("tanh(x)");
  if (may_exp_p (x, MAY_TANH_EXP_P) == 0)
    fail ("exp_p 16", x);

  may_keep (NULL);
}

void test_sqrtsimp ()
{
  may_t x, y;

  may_mark ();
  x = may_parse_str ("1/sqrt(2)");
  y = may_sqrtsimp (x);
  check (y, "1/2*2^(1/2)");

  x = may_parse_str ("((9997)^3)^(1/6)");
  y = may_sqrtsimp (x);
  check (y, "9997^(1/2)");

  x = may_parse_str ("sin(x)");
  y = may_sqrtsimp (x);
  check (y, "sin(x)");

  x = may_parse_str ("sin(1/sqrt(2))");
  y = may_sqrtsimp (x);
  check (y, "sin(1/2*2^(1/2))");

  x = may_parse_str ("1/(2+sqrt(3))");
  y = may_sqrtsimp (x);
  check (y, "2-3^(1/2)");

  x = may_parse_str ("1/(2+sqrt(3))^(1/2)");
  y = may_sqrtsimp (x);
  check (y, "-1/2*2^(1/2)+1/2*2^(1/2)*3^(1/2)");

  x = may_parse_str ("sqrt(3+sqrt(5))");
  y = may_sqrtsimp (x);
  check (y, "1/2*2^(1/2)+1/2*2^(1/2)*5^(1/2)");

  x = may_parse_str ("sqrt(3+sqrt(101))");
  y = may_sqrtsimp (x);
  check (y, "(3+101^(1/2))^(1/2)");

  x = may_parse_str ("sqrt(3+sqrt(PI))");
  y = may_sqrtsimp (x);
  check (y, "(3+PI^(1/2))^(1/2)");

  x = may_parse_str ("sqrt(18)");
  y = may_sqrtsimp (x);
  check (y, "3*2^(1/2)");

  x = may_parse_str ("(18)^(4/5)");
  y = may_sqrtsimp (x);
  check (y, "18^(4/5)");

  x = may_parse_str ("1/x");
  y = may_sqrtsimp (x);
  check (y, "1/x");

  x = may_parse_str ("1/3^(2/3)");
  x = may_sqrtsimp (x);
  check (x, "1/3*3^(1/3)");

  x = may_parse_str ("4^(1/3)/3^(2/3)");
  x = may_sqrtsimp (x);
  check (x, "1/3*3^(1/3)*4^(1/3)");

  /* Not really sqrtsimp, but more eval. */
  x = may_parse_str ("((1+sqrt(5))/2)^3/sqrt(5)");
  y = may_expand (x);
  check (y, "1+2/5^(1/2)");

  /* Other 0 to detects */
  x = may_parse_str ("4^(1/3)/2^(2/3)");
  x = may_sqrtsimp (x);
  check (x, "1");

  x = may_parse_str ("2^(1/3)*3^(1/3)-6^(1/3)");
  x = may_sqrtsimp (x);
  check (x, "0");

  x = may_parse_str ("6^(1/3)*5^(1/3)-2^(1/3)*15^(1/3)");
  x = may_sqrtsimp (x);
  check (x, "0");

  x = may_parse_str ("sqrt(2)+2^(1/10000000000)");
  x = may_sqrtsimp (x);
  check (x, "sqrt(2)+2^(1/10000000000)");

  x = may_parse_str ("sqrt(2)*sqrt(6)");
  x = may_sqrtsimp (x);
  check (x, "2*sqrt(3)");

  may_keep (NULL);
}

static const struct {
  const char *x, *p;
  int v;
} match_table[] = {
  {"cos(x+2)+sin(2*x+2)", "sin($0)+cos($0)", 0},
  {"2+sin(x)", "sin(x)+2", 1},
  {"2+sin(x)", "sin($0)+2", 1},
  {"cos(x+2)+sin(x+2)", "sin($0)+cos($0)", 1},
  {"cos(x)^2+sin(x)^2", "sin($0)^2+cos($0)^2", 1},
  {"cos(x)^2+sin(x)^2-1", "sin($0)^2+cos($0)^2", 0},
  {"cos(x)^2+sin(x)^2-1", "$0+sin($1)^2+cos($1)^2", 1},
  {"cos(x)^2+sin(x)^2", "$0+sin($1)^2+cos($1)^2", 1},
  {"2*x*y", "$0*x*y", 1},
  {"2*x", "$0*x", 1},
};

void test_match (void) {
  may_t x, p;
  may_t val[10];
  int i;
  int ret;

  may_mark ();

  for (i = 0; i < numberof (match_table); i++) {
    x = may_parse_str (match_table[i].x);
    p = may_parse_str (match_table[i].p);
    memset (val, 0, sizeof val);
    ret = may_match_p (val, x, p, 10, NULL);
    if (!!ret != !!match_table[i].v) {
      printf ("match error:\n"
              "x=%s\n"
              "pattern=%s\n"
              "Expected %d, got %d\n",
              match_table[i].x, match_table[i].p,
              match_table[i].v, ret);
      test_fail ();
    }
  }

  may_keep (NULL);
}

void test_rewrite (void)
{
  may_t x, y, z;
  may_mark ();

  y = may_parse_str ("sin($0)^2+cos($0)^2+$1");
  z = may_parse_str ("1+$1");
  x = may_parse_str ("log(sin(x)^2+cos(x)^2)");
  x = may_rewrite (x, y, z);
  if (!may_zero_p (x))
    fail ("rewrite", x);

  z = may_parse_str ("x+y+$0");
  y = may_parse_str ("1+$0*x");
  x = may_parse_str ("1+3*x");
  x = may_rewrite (x, y, z);
  check (x, "x+y+3");

  z = may_parse_str ("x+y+$0");
  y = may_parse_str ("1+$0*x");
  x = may_parse_str ("32+x");
  x = may_rewrite (x, y, z);
  check (x, "32+x");

  y = may_parse_str ("$0*$1^$2*$1^$3");
  z = may_parse_str ("$0*$1^($2+$3)");
  x = may_parse_str ("32+a*b^c*c^c*b^d");
  x = may_rewrite (x, y, z);
  check (x, "32+a*c^c*b^(c+d)");

  may_keep (NULL);
}

may_t naive_gcd (may_t x, may_t y)
{
  may_t tab[2];
  tab[0] = x;
  tab[1] = y;
  return may_naive_gcd (2, tab);
}

void test_naive_gcd (void)
{
  may_t z;
  may_mark ();

  z = naive_gcd (may_parse_str ("x^2"), may_parse_str ("x"));
  check (z, "x");

  z = naive_gcd (may_parse_str ("x^2*y^-1"), may_parse_str ("x^3*y"));
  check (z, "x^2*y^-1");

  z = naive_gcd (may_parse_str ("x^(n+1)"), may_parse_str ("x^n"));
  check (z, "x^n");

  z = naive_gcd (may_parse_str ("x^(n+2)"), may_parse_str ("2.0*x^(m+1)"));
  check (z, "x");

  z = naive_gcd (may_parse_str ("2*x^(n+2*a)"), may_parse_str ("2*x^(m+a)"));
  check (z, "2*x^a");

  z = naive_gcd (may_parse_str ("4"), may_parse_str ("6"));
  check (z, "2");

  z = naive_gcd (may_parse_str ("x^a"), may_parse_str ("x^b"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("x"), may_parse_str ("y"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("2*x"), may_parse_str ("8"));
  check (z, "2");

  z = naive_gcd (may_parse_str ("8"), may_parse_str ("6*x*y"));
  check (z, "2");

  z = naive_gcd (may_parse_str ("2/3"), may_parse_str ("2"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("2.3"), may_parse_str ("2"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("2.3+I"), may_parse_str ("2"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("2+4*I"), may_parse_str ("2"));
  check (z, "2");

  z = naive_gcd (may_parse_str ("2+4*I"), may_parse_str ("2+6*I"));
  check (z, "2");

  z = naive_gcd (may_parse_str ("2.0+4*I"), may_parse_str ("2+6*I"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("2.0"), may_parse_str ("3.0"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("2/3"), may_parse_str ("3/4"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("2/3"), may_parse_str ("2/3"));
  check (z, "2/3");

  z = naive_gcd (may_parse_str ("2/3+2/3*I"), may_parse_str ("2/3"));
  check (z, "2/3");

  z = naive_gcd (may_parse_str ("2.3+2.3*I"), may_parse_str ("2.3"));
  check (z, "2.3");

  z = naive_gcd (may_parse_str ("2/3+3/4*I"), may_parse_str ("2/3"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("2/3+3.4*I"), may_parse_str ("2/3"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("2.3+3.4*I"), may_parse_str ("2.3"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("2.3+4*I"), may_parse_str ("2.3"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("2.3+4.5*I"), may_parse_str ("6.7+8.9*I"));
  check (z, "1");

  z = naive_gcd (may_parse_str ("90*y+15*y^2"), may_parse_str ("90*c^2+90*y+15*y^2+15*y*c^2"));
  check (z, "15");

  z = may_gcd_c (may_set_str ("x"), may_set_str ("y"));
  z = may_eval (z);
  check (z, "gcd(x,y)");

  may_keep (NULL);
}

void test_divexact (void)
{
  may_t z;
  may_mark ();

  z = may_divexact (may_parse_str ("4"), may_parse_str ("2"));
  check (z, "2");

  z = may_divexact (may_parse_str ("4*x^2"), may_parse_str ("2*x"));
  check (z, "2*x");

  z = may_divexact (may_parse_str ("x^2"), may_parse_str ("x"));
  check (z, "x");

  z = may_divexact (may_parse_str ("x*y^2"), may_parse_str ("x"));
  check (z, "y^2");

  z = may_divexact (may_parse_str ("x*y^2*z^3*5"), may_parse_str ("x*y*z"));
  check (z, "5*y*z^2");

  z = may_divexact (may_parse_str ("y"), may_parse_str ("y"));
  check (z, "1");

  z = may_divexact (may_parse_str ("y*x"), may_parse_str ("y"));
  check (z, "x");

  z = may_divexact (may_parse_str ("y*x^2"), may_parse_str ("x^2"));
  check (z, "y");

  z = may_divexact (may_parse_str ("4.0*y^3"), may_parse_str ("2.0*y"));
  check (z, "2.0*y^2");

  z = may_divexact (may_parse_str ("2*x+3*y*x"), may_parse_str ("3*x"));
  check (z, "2/3+y");

  z = may_divexact (may_parse_str ("0"), may_parse_str ("3*x"));
  check (z, "0");

  z = may_divexact (may_parse_str ("1"), may_parse_str ("0"));
  check (z, "NAN");

  z = may_divexact (may_parse_str ("x"), may_parse_str ("0"));
  check (z, "NAN");

  z = may_divexact (may_parse_str ("1"), may_parse_str ("y"));
  check_bool (z == NULL);

  z = may_divexact (may_parse_str ("x^2+exp(y)"), may_parse_str ("x"));
  check_bool (z == NULL);

  z = may_divexact (may_parse_str ("exp(x)^2+exp(y)"), may_parse_str ("exp(x)"));
  check_bool (z == NULL);

  z = may_divexact ( may_parse_str ("(((1+x)^2*(2+x)^2+(3+x)^2*(2+x)^2+(3+x)^2*(1+x)^2)^2*(5+x)^2*(3+x)^4*(1+x)^4*(4+x)^4*(((1+x)^2*(2+x)^2+(3+x)^2*(2+x)^2+(3+x)^2*(1+x)^2)*(1+x)^2*(4+x)^2*(2+x)^2+((1+x)^2*(2+x)^2+(3+x)^2*(2+x)^2+(3+x)^2*(1+x)^2)*(3+x)^2*(1+x)^2*(2+x)^2+((1+x)^2*(2+x)^2+(3+x)^2*(2+x)^2+(3+x)^2*(1+x)^2)*(3+x)^2*(1+x)^2*(4+x)^2-(3+x)^2*(1+x)^2*(2+x)^2*((1+x)*(4+x)+(1+x)*(2+x)+(3+x)*(4+x))^2)*(2+x)^4)"),
                     may_parse_str ("(2*((1+x)^2*(2+x)^2+(3+x)^2*(2+x)^2+(3+x)^2*(1+x)^2)*(3+x)^2*(1+x)^2*(49+24*x^3+3*x^4+72*x^2+96*x)*(1+x^2+2*x)^2*(4+x^2+4*x)*(4+x)^2*(2+x)^2)"));
  check_bool (z != NULL);
  check (z, "525600+1716378*x^3+903761*x^4+2138354*x^2+13902*x^7+1551*x^8+325890*x^5+81419*x^6+102*x^9+3*x^10+1579440*x");
         //"(5+x)^2*(4+x)^2*(3+x)^2*(146+3*x^4+210*x+30*x^3+117*x^2)");

  z = may_divexact (may_parse_str("2*x*y+4*y*z"), may_parse_str("2*x*y+4*y*z"));
  check_bool (z!=NULL);
  check (z, "1");

  z = may_divexact (may_parse_str ("y*x^2"), may_parse_str ("y*x")),
  check_bool (z!=NULL);
  check (z, "x");

  z = may_divexact (may_parse_str ("y"), may_parse_str ("y")),
  check_bool (z!=NULL);
  check (z, "1");

  may_t sz = may_parse_str ("y^4*(y + 4 * t) ^ 9 * (y + 2 * t) ^ 7 * (y + t) ^ 6 * (y + 5 * t) ^ 10 * (y + 3 * t) ^ 8");
  z = may_divexact (sz, may_parse_str ("y + 4 * t"));
  check_bool (z!=NULL);
  check (z, "y^4*(y + 4 * t) ^ 8*(y + 2 * t) ^ 7 * (y + t) ^ 6 * (y + 5 * t) ^ 10 * (y + 3 * t) ^ 8");

  z = may_divexact (may_expand (sz), may_parse_str ("y + 4 * t"));
  check_bool (z!=NULL);
  check (z, "y^43+126*y^42*t+7696*y^41*t^2+303584*y^40*t^3+8693674*y^39*t^4+192613532*y^38*t^5+3436028324*y^37*t^6+50713518296*y^36*t^7+631530177335*y^35*t^8+6733359167362*y^34*t^9+62162263673540*y^33*t^10+501310347960760*y^32*t^11+3556332769953180*y^31*t^12+22316594341925800*y^30*t^13+124424056238323080*y^29*t^14+618511879544326320*y^28*t^15+2748740724198860415*y^27*t^16+10943096970735548130*y^26*t^17+39082504433415936440*y^25*t^18+125323778460406895760*y^24*t^19+360943319168475201194*y^23*t^20+933519236463571321564*y^22*t^21+2166653055183541730324*y^21*t^22+4507259087900129397496*y^20*t^23+8389308614523204874921*y^19*t^24+13937980717865616290398*y^18*t^25+20606627195479649646196*y^17*t^26+27007166776200896173784*y^16*t^27+31227926773073445519280*y^15*t^28+31668945569521162333088*y^14*t^29+27961983168304257854400*y^13*t^30+21299440271744521424000*y^12*t^31+13835495340447006720000*y^11*t^32+7549804833886540800000*y^10*t^33+3392651922391756800000*y^9*t^34+1221383922232320000000*y^8*t^35+338382555955200000000*y^7*t^36+67674341376000000000*y^6*t^37+8689213440000000000*y^5*t^38+537477120000000000*y^4*t^39");
  z = may_divexact (may_parse_str ("50*(5-x^2)*abs(x)"),
                    may_parse_str ("25-5*x^2"));
  check (z, "10*abs(x)");

  may_keep (NULL);
}

may_t naive_lcm (may_t x, may_t y)
{
  may_t tab[2];
  tab[0] = x;
  tab[1] = y;
  return may_naive_lcm (2, tab);
}

void test_naive_lcm (void)
{
  may_t z;
  may_mark ();

  z = naive_lcm (may_parse_str ("2*x^2"), may_parse_str ("4*x"));
  check (z, "4*x^2");

  z = naive_lcm (may_parse_str ("17*x^2*y^3*a"), may_parse_str ("42*x^3*y*b"));
  check (z, "17*42*x^3*y^3*a*b");

  z = naive_lcm (may_parse_str ("x^(n+1)"), may_parse_str ("x^n"));
  check (z, "x^(n+1)");

  z = naive_lcm (may_parse_str ("x^(n+2)"), may_parse_str ("x^(m+1)"));
  check (z, "x^(n+m+2)");

  z = naive_lcm (may_parse_str ("x^(n+2*a)"), may_parse_str ("x^(m+a)"));
  check (z, "x^(n+m+2*a)");

  z = naive_lcm (may_parse_str ("x*y*z"), may_parse_str ("2"));
  check (z, "2*x*y*z");

  z = naive_lcm (may_parse_str ("2*x+4"), may_parse_str ("2*x"));
  check (z, "2*(x+2)*x");

  z = naive_lcm (may_parse_str ("2*x+4"), may_parse_str ("3*x"));
  check (z, "6*(x+2)*x");

  z = naive_lcm (may_parse_str ("2*x*y+4*y*z"), may_parse_str ("2*y"));
  check (z, "2*(x+2*z)*y");

  z = naive_lcm (may_parse_str ("2"), may_parse_str ("3"));
  check (z, "6");

  may_keep (NULL);
}

void test_rectform (void)
{
  may_t x, y;
  may_mark ();

  may_rectform (&x, &y, may_parse_str ("1"));
  check (x, "1");
  check (y, "0");

  may_rectform (&x, &y, may_parse_str ("1+I"));
  check (x, "1");
  check (y, "1");

  may_rectform (&x, &y, may_parse_str ("x"));
  check (x, "real(x)");
  check (y, "imag(x)");

  may_rectform (&x, &y, may_parse_str ("1+x"));
  check (x, "1+real(x)");
  check (y, "imag(x)");

  may_rectform (&x, &y, may_parse_str ("(1+I)*real(x)"));
  check (x, "real(x)");
  check (y, "real(x)");

  may_rectform (&x, &y, may_parse_str ("2*real(x)"));
  check (x, "2*real(x)");
  check (y, "0");

  may_rectform (&x, &y, may_parse_str ("x*y"));
  check (x, "real(x)*real(y)-imag(x)*imag(y)");
  check (y, "real(x)*imag(y)+real(y)*imag(x)");

  may_rectform (&x, &y, may_parse_str ("exp(x)"));
  check (x, "cos(imag(x))*exp(real(x))");
  check (y, "sin(imag(x))*exp(real(x))");

  may_rectform (&x, &y, may_parse_str ("log(x)"));
  check (x, "log(abs(x))");
  check (y, "argument(x)");

  may_rectform (&x, &y, may_parse_str ("PI"));
  check (x, "PI");
  check (y, "0");

  may_rectform (&x, &y, may_parse_str ("x^3"));
  check (x, "(-imag(x)^2+real(x)^2)*real(x)-2*imag(x)^2*real(x)");
  check (y, "(-imag(x)^2+real(x)^2)*imag(x)+2*imag(x)*real(x)^2");

  may_rectform (&x, &y, may_parse_str ("x^-1"));
  check (x, "real(x)/(imag(x)^2+real(x)^2)");
  check (y, "-imag(x)/(imag(x)^2+real(x)^2)");

  may_rectform (&x, &y, may_parse_str ("cos(x)"));
  check (x, "cos(real(x))*cosh(imag(x))");
  check (y, "-sin(real(x))*sinh(imag(x))");

  may_rectform (&x, &y, may_parse_str ("sin(x)"));
  check (x, "sin(real(x))*cosh(imag(x))");
  check (y, "cos(real(x))*sinh(imag(x))");

  may_rectform (&x, &y, may_parse_str ("cosh(x)"));
  check (x, "cosh(real(x))*cos(imag(x))");
  check (y, "sinh(real(x))*sin(imag(x))");

  may_rectform (&x, &y, may_parse_str ("sinh(x)"));
  check (x, "sinh(real(x))*cos(imag(x))");
  check (y, "cosh(real(x))*sin(imag(x))");

  may_rectform (&x, &y, may_parse_str ("tan(x)"));
  check (x, "sin(2*real(x))/(cos(2*real(x))+cosh(2*imag(x)))");
  check (y, "sinh(2*imag(x))/(cos(2*real(x))+cosh(2*imag(x)))");

  may_rectform (&x, &y, may_parse_str ("tanh(x)"));
  check (x, "sinh(2*real(x))/(cosh(2*real(x))+cos(2*imag(x)))");
  check (y, "sin(2*imag(x))/(cosh(2*real(x))+cos(2*imag(x)))");

  may_rectform (&x, &y, may_parse_str ("(sqrt(2)+I*sqrt(6))/(2+2*I)"));
  check (x, "1/4*6^(1/2)+1/4*2^(1/2)");
  check (y, "1/4*6^(1/2)-1/4*2^(1/2)");

  may_keep (NULL);
}

void test_naive_factor ()
{
  may_t z;
  may_mark ();

  z = may_naive_factor (may_parse_str ("2*x*y+x"));
  check (z, "x*(2*y+1)");

  z = may_naive_factor (may_parse_str ("(2*x*y*z^2+x)^2"));
  check (z, "(x*(2*y*z^2+1))^2");

  z = may_naive_factor (may_parse_str ("2*(x*y+y)*y"));
  check (z, "2*y^2*(x+1)");

  may_keep (NULL);
}

void test_mod (void)
{
  may_t z;

  may_mark ();
  z = may_set_str ("z");
  z = may_pow (z, may_set_ui (2));
  z = may_add (z, may_sqrt (z));
  z = may_mod_c (z, may_set_str ("y"));
  z = may_eval (z);
  check (z, "mod(sqrt(z^2)+z^2,y)");

  may_keep (NULL);
}

void test_comdenom (void)
{
  may_t n, d;
  may_mark ();

  may_comdenom (&n, &d, may_parse_str ("x+1/x+2/3"));
  check (n, "3+2*x+3*x^2");
  check (d, "3*x");

  may_comdenom (&n, &d, may_parse_str ("y/x+3/2*I"));
  check (n, "2*y+3*I*x");
  check (d, "2*x");

  may_comdenom (&n, &d, may_parse_str ("1+I+x"));
  check (n, "1+I+x");
  check (d, "1");

  /* FIXME: More an integration test than an unit test */
  n = may_parse_str ("sin(x-y)/sin(y-x)");
  may_rectform (&n, &d, n);
  n = may_set_cx (n, d);
  may_comdenom (&n, &d, n);
  n = may_div (n, d);
  n = may_replace (n, may_parse_str ("imag(x)"), may_set_ui (0));
  n = may_replace (n, may_parse_str ("imag(y)"), may_set_ui (0));
  check (n, "sin(-real(y)+real(x))/sin(real(y)-real(x))");
  /*
  may_comdenom (&n, &d, may_parse_str ("(exp(a)+a)/(exp(a)-a)"));
  check (n, "exp(a)+a");
  check (d, "exp(a)-a");
  */
  /*
  n = may_parse_str ("-7/2*sign(2-x)/abs(2-x)+4/5/(5-x^2)-31/25/(1-1/5*x^2)+87/25*sign(5-x^2)*x/abs(5-x^2)-1/50*sign(x)/abs(x)+8/5*x^2/(5-x^2)^2+17/5*x/(5-x^2)^2");
  may_comdenom (&n, &d, n);
  check (n, "");
  check (d, "");
  */
  n = may_parse_str ("2*(1/2*(x/(a-x))^(1/2)/(1+x/(a-x))+1/2*atan((x/(a-x))^(1/2)))*a");
  may_comdenom (&n, &d, n);
  check (n, "(a-x)*(x/(a-x))^(1/2)+a*atan((x/(a-x))^(1/2))");
  check (d, "1");

  may_keep (NULL);
}

void test_replacefunc (void)
{
  may_mark ();

  check (may_trig2tan2 (may_parse_str ("sin(x)+cos(x)+tan(x)")),
             "(1-tan(1/2*x)^2)/(1+tan(1/2*x)^2)"
             "+2*tan(1/2*x)/(1+tan(1/2*x)^2)"
             "+2*tan(1/2*x)/(1-tan(1/2*x)^2)");

  check (may_tan2sincos (may_parse_str ("tan(x)")),
	 "sin(x)/cos(x)");

  check (may_tan2sincos (may_parse_str ("tanh(x)")),
	 "sinh(x)/cosh(x)");

  check (may_sin2tancos (may_parse_str ("sin(x)")),
         "tan(x)*cos(x)");

  check (may_exp2pow (may_parse_str ("exp(x)*exp(y)*exp(x)")),
         "(exp(1)^x)^2*(exp(1)^y)");

  check (may_pow2exp (may_parse_str ("x^y+3+z*cos(a^b)+x^3")),
         "exp(y*log(x))+3+z*cos(exp(b*log(a)))+x^3");

  may_keep (NULL);
}

void test_may_list (void)
{
  may_t z;
  may_list_t list;
  may_size_t i;
  may_mark_t mark;

  may_mark (mark);

  may_list_init (list, 10);

  check_si (may_list_get_size (list), 10);
  check_si ((may_list_get_size) (list), 10);

  for (i = 0; i <10; i++) {
    check (may_list_at (list, i), "0");
    check ((may_list_at) (list, i), "0");
  }

  may_list_set_at (list, 0, MAY_ONE);
  (may_list_set_at) (list, 9, MAY_ONE);
  check (may_list_at (list, 0), "1");
  check (may_list_at (list, 9), "1");

  may_list_resize (list, 5);
  check_si (may_list_get_size (list), 5);

  may_list_resize (list, 100);
  check_si (may_list_get_size (list), 100);

  may_list_compact (mark, list);

  may_list_push_back (list, MAY_ONE);
  check (may_list_at (list, 99), "0");
  check (may_list_at (list, 100), "1");

  z = may_list_quit (list);
  check_bool (MAY_TYPE (z) == MAY_LIST_T && MAY_NODE (z).size == 101);

  may_compact (mark, NULL);
}

void test_degree (void)
{
  may_t c, x, y, z, leader;
  mpz_srcptr d;
  int b;

  may_mark ();
  x = may_set_str ("x");
  y = may_set_str ("y");
  z = may_set_str ("z");

  b = may_degree (&c, &d, NULL, may_parse_str ("x"), 1, &x);
  check_bool (b);
  check (c, "1");
  check_z (d, "1");

  b = may_degree (&c, &d, NULL, may_parse_str ("12"), 1, &x);
  check_bool (b);
  check (c, "12");
  check_z (d, "0");

  b = may_degree (&c, &d, NULL, may_parse_str ("x-x"), 1, &x);
  check_bool (b);
  check (c, "0");
  check_z (d, "-1");

  b = may_degree (&c, &d, NULL, may_parse_str ("y*x^2+2*x+3"), 1, &x);
  check_bool (b);
  check (c, "y");
  check_z (d, "2");

  b = may_degree (&c, &d, NULL, may_parse_str ("5*x^2+2*x+3*y*z"), 1, &x);
  check_bool (b);
  check (c, "5");
  check_z (d, "2");

  b = may_degree (&c, &d, NULL, may_parse_str ("5*x^2+2*x+3*exp(x)"), 1, &x);
  check_bool (!b);

  b = may_degree (&c, &d, NULL, may_parse_str ("3*exp(x)*y+log(x)*z"), 1, &x);
  check_bool (!b);

  b = may_degree (&c, &d, NULL, may_parse_str ("5*x^2+y*x^2+x"), 1, &x);
  check_bool (b);
  check (c, "5+y");
  check_z (d, "2");

  b = may_degree (NULL,&d,&c, may_parse_str ("5*t*x^3*z+y*x^3+x+4+x^2"),1,&x);
  check_bool (b);
  check (c, "5*x^3*z*t+y*x^3");
  check_z (d, "3");

  b = may_degree (NULL, &d, NULL, may_parse_str ("5*t(x)*x^3*z+y*x^3+x+4+x^2"),
                  1, &x);
  check_bool (!b);

  b = may_degree (NULL, NULL, &c, may_parse_str ("5*t*x^3*z"), 1, &x);
  check_bool (b);
  check (c, "5*t*x^3*z");

  b = may_degree (&c, &d, NULL,
                  may_parse_str ("10*x^2+4*x"), 1, &x);
  check_bool (b);
  check (c, "10");
  check_z (d, "2");

  /* Multidegree tests */
  may_t var1[3] = {x, y, z};
  mpz_srcptr deg[3];

  b = may_degree (&c, deg, &leader,
		  may_parse_str ("5*t*x^4*z+y*x^3+x+4+x^2"), 2, var1);
  check_bool (b);
  check (c, "5*z*t");
  check (leader, "5*x^4*z*t");
  check_z (deg[0], "4");
  check_z (deg[1], "0");

  b = may_degree (NULL, deg, NULL,
		  may_parse_str ("5*t*x^4*z+y*x^3+x+4+x^2"), 2, var1);
  check_bool (b);
  check_z (deg[0], "4");
  check_z (deg[1], "0");

  b = may_degree (&c, NULL, &leader,
		  may_parse_str ("5*t*x^4*z+y*x^3*sin(1)+x+4+x^4+x^4*z*17"),
		  3, var1);
  check_bool (b);
  check (c, "5*t+17");
  check (leader, "5*t*x^4*z+17*x^4*z");

  b = may_degree (&c, deg, &leader, may_parse_str ("0"), 3, var1);
  check_bool (b);
  check (c, "0");
  check (leader, "0");
  check_z (deg[0], "-1");
  check_z (deg[1], "-1");
  check_z (deg[2], "-1");

  b = may_degree (NULL, NULL, NULL,
                  may_parse_str ("5*sin(x)*x^4*z+y*x^3*sin(1)+x+4"),
                  3, var1);
  check_bool (!b);
  b = may_degree (NULL, NULL, &leader,
                  may_parse_str ("5*sin(x)*x^4*z"), 3, var1);
  check_bool (!b);
  b = may_degree (NULL, NULL, &leader,
                  may_parse_str ("y*z*x*t(x)"), 3, var1);
  check_bool (!b);
  b = may_degree (NULL, NULL, &leader,
                  may_parse_str ("y*t(x,t)"), 3, var1);
  check_bool (!b);
  b = may_degree (NULL, NULL, &leader,
                  may_parse_str ("y*t(t,t)"), 3, var1);
  check_bool (b);
  b = may_degree (NULL, NULL, &leader,
                  may_parse_str ("x*z(x)+x"), 3, var1);
  check_bool (!b);
  b = may_degree (NULL, NULL, &leader,
                  may_parse_str ("toto(x)+y"), 3, var1);
  check_bool (!b);

  long degul;
  degul = may_degree_si (may_parse_str ("x^2+1"), x);
  check_si (degul, 2);

  degul = may_degree_si (may_parse_str ("1+y"), x);
  check_si (degul, 0);

  degul = may_degree_si (may_parse_str ("exp(x)"), x);
  check_bool ((degul == LONG_MAX));

  degul = may_degree_si (may_parse_str ("x^(2^200)"), x);
  check_bool ((degul == LONG_MAX));

  degul = may_degree_si (may_parse_str ("0"), x);
  check_bool ((degul == LONG_MIN));


  /* ldegree test */
  degul = may_ldegree_si (may_parse_str ("x^2+1"), x);
  check_si (degul, 0);

  degul = may_ldegree_si (may_parse_str ("x+x^2"), x);
  check_si (degul, 1);

  may_keep (NULL);
}

void test_taylor (void)
{
  may_t z;
  may_mark ();

  z = may_taylor (may_parse_str ("f(x)"), may_set_str ("x"), may_set_str ("a"),
                  3);
  check (z, "1/6*(-a+x)^3*diff(f(x),x,3,a)+f(a)+(-a+x)*diff(f(x),x,1,a)"
	 "+1/2*(-a+x)^2*diff(f(x),x,2,a)");

  z = may_taylor (may_parse_str ("x^2"), may_set_str ("x"), may_set_str ("x"),
                  5);
  check (z, "x^2");

  z = may_taylor (may_parse_str ("f(x)"), may_set_str ("x"), may_set_ui (0),5);
  check (z, "1/2*diff(f(x),x,2,0)*x^2+1/24*diff(f(x),x,4,0)*x^4"
	 "+diff(f(x),x,1,0)*x+1/6*diff(f(x),x,3,0)*x^3+f(0)"
	 "+1/120*diff(f(x),x,5,0)*x^5");

  z = may_taylor (may_parse_str ("acosh(x)"), may_set_str ("x"), may_set_ui (0), 3);
  check (z, "-1/6*I*x^3-I*x+1/2*I*PI");

  may_keep (NULL);
}

void test_divqr (void)
{
  may_t q, r, x, f, g;
  int b;

  may_mark ();
  x = may_set_str ("x");

  b = may_div_qr (&q, &r,
		  may_parse_str ("x^2+x+1"), may_parse_str ("x^2"), x);
  check_bool (b);
  check (q, "1");
  check (r, "x+1");

  b = may_div_qr (&q, &r,
		  may_parse_str ("x^2+2*x+1"), may_parse_str ("x+1"), x);
  check_bool (b);
  check (q, "x+1");
  check (r, "0");

  b = may_div_qr (NULL, &r,
		  may_parse_str ("-16+3*x^2+x^3+3*x"),may_parse_str ("x+1"),x);
  check_bool (b);
  check (r, "-17");

  b = may_div_qr (&q, NULL,
		  may_parse_str ("-16+3*x^2+x^3+3*x"),may_parse_str ("x+1"),x);
  check_bool (b);
  check (q, "x^2+2*x+1");

  b = may_div_qr (NULL, NULL,
		  may_parse_str ("-16+3*x^2+x^3+3*x"),may_parse_str ("x^x"),x);
  check_bool (!b);

  b = may_div_qr (NULL, NULL,
		  may_parse_str ("-16+3*x^x"),may_parse_str ("x^2"),x);
  check_bool (!b);

  f = may_parse_str ("-(y*x^4+4*y*x^3+6*y*x^2+y+4*y*x+4*x^2+4*x^4+6*x^3+x^5+x)*(1+x^2+2*x)");
  g = may_parse_str ("-y*x^2-y-2*y*x+2*x^2+x^3+x");
  b = may_div_qr (&q, &r, f, g, x);
  check_bool (b);
  f = may_expand (f);
  g = may_expand (may_add (may_mul (g, q), r));
  check_bool (may_identical (f, g) == 0);

  f = may_parse_str ("-(y*x^4+4*y*x^3+6*y*x^2+y+4*y*x+4*x^2+4*x^4+6*x^3+x^5+x)*(1+x^2+2*x)");
  g = may_parse_str ("-y*x^2-y-2*y*x+2*x^2+x^3+x");
  b = may_div_qr (&q, &r, f, g, may_set_str ("y"));
  check_bool (b);
  f = may_expand (f);
  g = may_expand (may_add (may_mul (g, q), r));
  check_bool (may_identical (f, g) == 0);

  b = may_div_qr (&q, &r,
                  may_parse_str ("x*z+1"), may_parse_str ("x*y"), x);
  check_bool (b);
  check (q, "z/y");
  check (r, "1");

  b = may_div_qr (&q, &r,
                  may_parse_str ("x*z+1"), may_parse_str ("x*y+x*a"), x);
  check_bool (b);
  check (q, "z/(y+a)");
  check (r, "1");

  b = may_div_qr (&q, &r,
                  may_parse_str ("x*z+1"), may_parse_str ("0"), x);
  check_bool (!b);

  b = may_div_qr (&q, &r,
                  may_parse_str ("2"), may_parse_str ("3"), x);
  check_bool (b);
  check (q, "2/3");
  check (r, "0");

  b = may_div_qr (&q, &r,
                  may_parse_str ("2*x*y+2*y*z"), may_parse_str ("x*y+y*z"),
                  may_set_str ("y"));
  check_bool (b);
  check (q, "2");
  check (r, "0");

  /* Multivariable test */
  b = may_div_qr (&q, &r,
                  may_parse_str ("x+y*x+z"), may_parse_str ("x"), may_parse_str("{x,y}"));
  check_bool (b);
  check (q, "1+y");
  check (r, "z");
  b = may_div_qr (NULL, NULL,
                  may_parse_str ("x+y*x+z"), may_parse_str ("x"), may_parse_str("{x,y}"));
  check_bool (b);

  b = may_div_qr (&q, &r,
                  may_parse_str ("-2*c^3+c^5+c"),
                  may_parse_str ("(s2-2*s*c)*(s^2+c^2-1)"),
                  may_parse_str ("{c,s}"));
  check_bool (b);
  check (q, "1/2*s-1/2*c^2/s+1/-4*c*s2/s^2"); // Bug parsing it // Bug getting it
  check (r, "-1/4*c*s2^2/s^2-c^3+1/2*s*s2-1/2*s^3*s2+s^4*c+c-1/2*s*c^2*s2-s^2*c+1/4*c^3*s2^2/s^2+1/4*c*s2^2");

  b = may_div_qr (&q, &r,
                  may_parse_str ("(6 * c ^ 2 + 6 * y + y ^ 2 + 6 * (c ^ 2 * x) + 6 * (y * x) + y * c ^ 2 * x + y ^ 2 * x + y * c ^ 2) * (6 + c ^ 2 + 2 * y)"),
                  may_parse_str ("6 + c ^ 2 + 2 * y + c ^ 2 * x + 2 * (y * x) + 6 * x"),
                  may_parse_str ("x"));
  check_bool (b);
  check (q, "6*c^2+y^2+(6+c^2)*y");
  check (r, "0");

  b = may_div_qr (&q, &r,
                  may_parse_str ("(6 * c ^ 2 + 6 * y + y ^ 2 + 6 * (c ^ 2 * x) + 6 * (y * x) + y * c ^ 2 * x + y ^ 2 * x + y * c ^ 2) * (6 + c ^ 2 + 2 * y)"),
                  may_parse_str ("6 + c ^ 2 + 2 * y + c ^ 2 * x + 2 * (y * x) + 6 * x"),
                  may_parse_str ("{x, y}"));
  check_bool (b);
  check (q, "6*c^2+y^2+(6+c^2)*y");
  check (r, "0");


  may_keep (NULL);
}

void test_divqr_xexp (void)
{
  may_t q, r, a, x, n;
  may_mark ();

  x = may_set_str ("x");
  a = may_parse_str ("x^2+x^4+x^3+x+1");
  may_div_qr_xexp (&q, &r, a, x, may_set_ui (2));
  check (q, "1+x^2+x");
  check (r, "x+1");
  may_div_qr_xexp (NULL, NULL, a, x, may_set_ui (2));

  a = may_parse_str ("4*y*x^2+5*a*x^4+exp(x)*x^3+1/x+1+x+x*y");
  may_div_qr_xexp (&q, &r, a, x, may_set_ui (2));
  check (q, "4*y+5*a*x^2+exp(x)*x");
  check (r, "1/x+1+x+x*y");

}

void test_gcd (void)
{
  may_t a, x;
  may_t temp[3];
  may_mark_t mark;

  may_mark (mark);

  x = may_set_str ("x");

  temp[0] = x;
  temp[1] = may_set_ui(0);
  a = may_gcd (2, temp);
  check_bool (a != NULL);
  check (a, "x");

  temp[0] = may_set_ui(0);
  temp[1] = x;
  temp[2] = may_parse_str("x^2");
  a = may_gcd (3, temp);
  check_bool (a != NULL);
  check (a, "x");

  temp[0] = may_set_ui(0);
  temp[1] = may_set_ui(0);
  temp[2] = may_set_ui(0);
  a = may_gcd (3, temp);
  check_bool (a != NULL);
  check (a, "0");

  temp[0] = may_parse_str ("(x+1)^7-(x-1)^6");
  temp[1] = may_diff (temp[0], x);
  a = may_gcd (2, temp);
  check_bool (a != NULL);
  check (a, "1");

  temp[0] = may_parse_str ("(x+1)^7-(x-1)^6");
  temp[1] = may_set_ui(0);
  temp[2] = may_diff (temp[0], x);
  a = may_gcd (3, temp);
  check_bool (a != NULL);
  check (a, "1");

  temp[0] = may_parse_str ("(x+1)^7");
  temp[1] = may_diff (temp[0], x);
  a = may_gcd (2, temp);
  check_bool (a != NULL);
  check (a, "(1+x)^6");

  temp[0] = may_parse_str ("(x+1)^7");
  temp[1] = may_diff (temp[0], x);
  temp[2] = may_set_ui(0);
  a = may_gcd (3, temp);
  check_bool (a != NULL);
  check (a, "(1+x)^6");

  temp[2] = may_expand (may_diff (temp[0], x));
  a = may_gcd (3, temp);
  check_bool (a != NULL);
  check (a, "1+15*x^2+15*x^4+20*x^3+x^6+6*x^5+6*x");

  temp[2] = may_expand (may_diff (temp[1], x));
  a = may_gcd (3, temp);
  check_bool (a != NULL);
  check (a, "x^5+5*x^4+10*x^3+10*x^2+5*x+1");

  temp[0] = may_parse_str ("x^2-4");
  temp[1] = may_parse_str ("x^2+4*x+4");
  a = may_gcd (2, temp);
  check_bool (a != NULL);
  check (a, "2+x");

  // Some heavier tests
  temp[0] = may_parse_str ("-2107-7967*x+19271*x^50+551*x^49-39300*x^48+23685*x^51-47193*x^61+22470*x^63-11981*x^64+4427*x^65+12796*x^66-11319*x^67-44213*x^68-18278*x^69-8897*x^70-15766*x^71+10258*x^72+13882*x^73-23195*x^74-8704*x^75+4815*x^76-13598*x^77+13217*x^78-16323*x^79-13612*x^80-14833*x^62+25634*x^52+17615*x^54+14299*x^55-19193*x^56-4719*x^57-51084*x^58-14141*x^59+22254*x^60+61350*x^53-7982*x^81-5434*x^82+16274*x^83+7039*x^84+4819*x^85+13739*x^86+7595*x^87+13020*x^88-1874*x^89-11233*x^90-24214*x^91-11353*x^92-2585*x^93+22786*x^94+18677*x^95+16977*x^96+1096*x^97-9941*x^98-5735*x^99-6630*x^100-11236*x^35+21918*x^34-7645*x^33+12569*x^32+44195*x^42+7111*x^41-5940*x^40+8721*x^39+8*x^38+3951*x^37-16955*x^36+8166*x^44+1255*x^43-5677*x^45+2889*x^46-8483*x^47-6954*x^31-21427*x^30-20019*x^29-13763*x^28-8902*x^27-8764*x^26+11613*x^12+1365*x^11-5613*x^10+7489*x^9+3585*x^8-11942*x^15+17166*x^18-12107*x^17-25122*x^16+17961*x^14-3205*x^13+13609*x^20-11190*x^24+3638*x^25+7883*x^23-6332*x^22+511*x^21+31863*x^19-2029*x^4+2383*x^3-4956*x^2+1663*x^7-6356*x^6+4493*x^5");
  temp[1] = may_parse_str ("-2401-3773*x-9484*x^50-4086*x^49-31296*x^48-21634*x^51+18525*x^61-1327*x^63+17211*x^64+6097*x^65+4392*x^66+5639*x^67+26737*x^68+5985*x^69-2186*x^70+3352*x^71+5962*x^72+5413*x^73+15009*x^74+7176*x^75+23806*x^76-860*x^77+16368*x^78+9044*x^79+4566*x^80-7929*x^62-13086*x^52-84580*x^54-17160*x^55-4502*x^56-56940*x^57+5464*x^58-25712*x^59-41935*x^60+2429*x^53+12607*x^81-4635*x^82+1625*x^83+11692*x^84-22514*x^85+14489*x^86+6490*x^87-6581*x^88+5116*x^89-9671*x^90-18639*x^91-1692*x^92-9911*x^93-3458*x^94+2083*x^95-10129*x^96+6425*x^97+986*x^98+1890*x^99+6205*x^100+4746*x^35+14296*x^34+33685*x^33+23430*x^32-20474*x^42+14725*x^41-39167*x^40+11129*x^39+34455*x^38+19625*x^37+6082*x^36+7384*x^44-9814*x^43+11853*x^45-23402*x^46+24318*x^47+10912*x^31+17976*x^30-11869*x^29+11588*x^28+11503*x^27+15005*x^26-11632*x^12-17281*x^11+4819*x^10-17526*x^9+1700*x^8-9352*x^15+5662*x^18-11328*x^17+9618*x^16-3703*x^14+9162*x^13-25581*x^20+13776*x^24-1141*x^25-22249*x^23-3267*x^22-2752*x^21+17790*x^19+13366*x^4-4768*x^3+1366*x^2+2927*x^7+5018*x^6-6514*x^5");
  a = may_gcd (2, temp);
  check_bool (a != NULL);
  check (a, "(85*x^50+55*x^49+37*x^48+35*x^47-97*x^46-50*x^45-79*x^44-56*x^43-49*x^42-63*x^41-57*x^40+59*x^39-45*x^38+8*x^37+93*x^36-92*x^35-43*x^34+62*x^33-77*x^32-66*x^31-54*x^30+5*x^29-99*x^28+61*x^27+50*x^26+12*x^25+18*x^24-31*x^23+26*x^22+62*x^21-x^20+47*x^19+91*x^18+47*x^17+61*x^16-41*x^15+58*x^14+90*x^13-53*x^12+x^11-94*x^10-83*x^9+86*x^8-23*x^7+84*x^6-19*x^5+50*x^4-88*x^3+53*x^2-85*x-49)");

  // Currently this test pass, but is too memory hungry with assertion on
  if (strstr (may_get_version (), "WANT_ASSERT") == NULL) {
    may_compact (mark, NULL);
  temp[0] = may_parse_str ("-1368+2517*x-62928*x^500+126728*x^499-139637*x^498+27580*x^50+22746*x^49-2531*x^48-14087*x^51-5358*x^61+46791*x^63-7435*x^64-40697*x^65+13424*x^66+30705*x^67+9285*x^68-11869*x^69+16191*x^70-17766*x^71-60286*x^72+12097*x^73+9105*x^74-31654*x^75+9021*x^76-53005*x^77+19426*x^78+53086*x^79-25232*x^80+12673*x^62-31816*x^52-21377*x^54+618*x^55+39344*x^56+4398*x^57-61128*x^58-463*x^59-8004*x^60+6777*x^53+896*x^81-14671*x^82-28762*x^83-35402*x^84-5453*x^85+40572*x^86+24269*x^87-44167*x^88-22203*x^89-50138*x^90-26643*x^91+19350*x^92+19349*x^93+40256*x^94+13649*x^95-47903*x^96-41079*x^97-5645*x^98-4755*x^99-11582*x^100-40612*x^35+2800*x^34+2779*x^33-7942*x^32-69752*x^426+94597*x^425+47154*x^424+2012*x^423+31277*x^422+33408*x^421+84807*x^420+28863*x^419+68957*x^418+68073*x^417+59170*x^416+90828*x^415-16413*x^414-47850*x^413+139517*x^412-127726*x^411+5282*x^410+22398*x^409+77991*x^408+17422*x^407+142163*x^406+61882*x^405+89785*x^404+94572*x^403-2414*x^402-56973*x^401-101640*x^400-149488*x^434-98144*x^433+5076*x^432+40468*x^431+136215*x^430+151022*x^429+29277*x^428-186747*x^427-11950*x^438+46996*x^437-87072*x^436+50978*x^435-53867*x^439+34216*x^465-34398*x^464+186429*x^463+115912*x^462+50371*x^461-94614*x^460-6364*x^459+208358*x^458-37065*x^457-95153*x^456+134795*x^455-4809*x^454-55455*x^453-122445*x^452+43069*x^451-72494*x^450+79409*x^449-68950*x^448-16066*x^447-50587*x^446-102418*x^445-74453*x^444+167227*x^443-43709*x^442+19063*x^441-33367*x^440+65259*x^476+43375*x^475-62053*x^474+57480*x^473+16073*x^472+196982*x^471+29802*x^470-49897*x^469-9948*x^468+67615*x^467-19190*x^466-7500*x^42+18420*x^41-23816*x^40-4876*x^39+7627*x^38+16788*x^37-6120*x^36+5817*x^44+26641*x^43-20495*x^45-3115*x^497-60587*x^496-201311*x^495-46792*x^494+31565*x^493+22845*x^492-40022*x^491-106106*x^490-69754*x^489+16080*x^488+66696*x^487+188430*x^486+143305*x^485+3480*x^484+19886*x^483-69468*x^482+54910*x^481-6406*x^480+26650*x^479+64640*x^478-11401*x^477-2757*x^46+3794*x^47+45223*x^266+94576*x^265+42457*x^264-45418*x^263+54839*x^262+62286*x^261-43253*x^260+74164*x^259-22360*x^258+367*x^267+87386*x^297+1033*x^293-34984*x^292+15107*x^291+29227*x^290-60676*x^289+37316*x^288+67439*x^287+28665*x^286-30395*x^285-12669*x^284-118959*x^283-76727*x^282+40578*x^281+12478*x^280+139518*x^279-64661*x^278-70425*x^277-51423*x^276-75396*x^275-69331*x^274+14651*x^273+65687*x^272-33259*x^271-5551*x^270+26596*x^269-48554*x^268+30609*x^31+13234*x^30-24923*x^29-18067*x^28-12113*x^27-23210*x^26+89410*x^323+102031*x^322+57892*x^321-20669*x^320+60640*x^319-1516*x^318+2552*x^317-34122*x^333-49281*x^332-16718*x^331+96104*x^330-15874*x^329-42731*x^328-33928*x^327-29381*x^326+25084*x^325-91237*x^324-11027*x^334+21081*x^345-6588*x^344-85155*x^343+81994*x^342-27868*x^341-18945*x^340+94970*x^339+74905*x^338+84612*x^337-2698*x^336+24174*x^335-961*x^371-62269*x^370+77449*x^369-8992*x^368-45317*x^367+50793*x^366+100850*x^365+49600*x^364+418*x^363-38448*x^362+111756*x^361-59212*x^360+83470*x^359+38710*x^358+41701*x^357+4635*x^356+34246*x^355-47782*x^354+63743*x^353+14401*x^352-2942*x^351-15056*x^350+909*x^349-42355*x^348-27510*x^347-125124*x^346-25863*x^373+80195*x^372+52262*x^316+149353*x^315-55819*x^314+103411*x^313+158668*x^312-959*x^311+41724*x^310+62321*x^309+40000*x^308-48054*x^307-43199*x^306+111954*x^305+10527*x^304-22207*x^303-44732*x^302-62568*x^301-57920*x^300-5505*x^299+4394*x^298-18533*x^296-3782*x^295-68145*x^294+31258*x^399+2445*x^398-23728*x^397-17216*x^396-63084*x^395-4385*x^394-95754*x^393+63983*x^392-53024*x^391-44658*x^390-17576*x^389-41740*x^388+6318*x^387+6705*x^386-84829*x^385+53323*x^384+853*x^383+21916*x^382+57967*x^381-2589*x^380+71462*x^379-52141*x^378-7581*x^377-107109*x^376+58294*x^375-9185*x^374+16162*x^145-45541*x^144-5390*x^143+16722*x^142-30871*x^141+8858*x^140-35953*x^139-64934*x^138-6993*x^137-47839*x^136+34376*x^135+1714*x^134-55912*x^133+60713*x^132-66513*x^131+3055*x^130-52112*x^129-16474*x^128+68069*x^127-20608*x^126+796*x^125+37824*x^124-61304*x^123-43055*x^122-17539*x^121+680*x^120-18792*x^12+22675*x^11+23056*x^10+12449*x^9+769*x^8-70461*x^156-13959*x^155-3582*x^154+38179*x^153+509*x^152+11510*x^151+21141*x^150-23384*x^149-39509*x^148+91242*x^147+2943*x^146+33565*x^177-16919*x^176+54064*x^175-48549*x^174+61080*x^173-7615*x^172-781*x^171+37221*x^170+67845*x^169-16675*x^168+48196*x^167+30887*x^166+17310*x^165-135726*x^164+67870*x^163+21560*x^162+19914*x^161+11556*x^160+70610*x^159-47532*x^158+25493*x^157-4417*x^15+20459*x^18+40952*x^17+29850*x^16-13653*x^14-41563*x^13-45800*x^119+61140*x^118+8669*x^117+21847*x^116-79064*x^115-12970*x^114+14648*x^113-14101*x^112-30609*x^111+34569*x^110+18287*x^109-12075*x^107+7847*x^106+22346*x^105-23549*x^104-3279*x^103+44478*x^102-1076*x^101-48244*x^20+6836*x^201+1668*x^200-48353*x^199-25269*x^198-75510*x^197+50290*x^196+75028*x^195+15221*x^194+82848*x^193-7631*x^192-80664*x^191-88334*x^190+18015*x^189+84791*x^188+46134*x^187-4242*x^186-17111*x^185-95674*x^184+54257*x^183-64039*x^182+35669*x^181+17814*x^180-75848*x^179-42986*x^178-27965*x^228-94680*x^227+63443*x^226-78487*x^225+8800*x^224-58462*x^223-85978*x^222-20545*x^221+38209*x^220+42553*x^219+62882*x^218-78547*x^217+73749*x^216-53425*x^215+11205*x^214+32703*x^213+116004*x^212+8114*x^211+788*x^210+15187*x^209-9315*x^208-80863*x^207+20671*x^206-56079*x^205-15767*x^204+31905*x^203+78466*x^202-12265*x^234-35331*x^233-91707*x^232+109440*x^231-19376*x^230+26171*x^229+30379*x^24+63028*x^252+9160*x^251-2817*x^250+49242*x^249-119901*x^248-153407*x^247-11239*x^246+23438*x^245+26296*x^244-22390*x^243-11429*x^242+25379*x^241-106807*x^240+12136*x^239+15332*x^238+39203*x^237+15547*x^236-20061*x^235-20118*x^257+19291*x^256+101761*x^255-3630*x^254-7337*x^253+13929*x^25+21267*x^23-5025*x^22-16945*x^21-27666*x^19-2191*x^4+11365*x^3+7985*x^2-3553*x^7-18692*x^6-11338*x^5-21056*x^108+6546*x^501-191302*x^502-79510*x^503-65806*x^504+8111*x^505+47722*x^506+78838*x^507+101061*x^508+63259*x^509-103564*x^510-74851*x^544-174966*x^546-109470*x^547+160718*x^548+28397*x^549+21235*x^550-103816*x^551+15599*x^552-149336*x^553-98258*x^545+28210*x^554-6175*x^555+23644*x^556+94624*x^557-89878*x^558-113748*x^559-49343*x^560+91912*x^561+34420*x^562-50036*x^563+8764*x^564+17462*x^565-10665*x^566+4433*x^567+42567*x^568-39896*x^511+34392*x^512-1252*x^513+87985*x^514+126695*x^515+22222*x^516-142289*x^517+20542*x^518+77779*x^519-199249*x^520+102289*x^521+51103*x^522-13558*x^524+33990*x^525-1402*x^526+52050*x^527+39174*x^528-17971*x^529+20449*x^530+47878*x^531-52137*x^532-2594*x^533-43235*x^534+13613*x^535-50267*x^536-79596*x^537-92795*x^538-32565*x^539+54227*x^540-71137*x^541+32592*x^542-20287*x^543+129714*x^523-11030*x^616+166341*x^617-89703*x^618+45481*x^619-36199*x^620+51414*x^621-99641*x^622-42368*x^623+94675*x^624+88105*x^625+62598*x^626+137212*x^627+73363*x^628+56976*x^629-8849*x^630-89311*x^631-52982*x^632+35761*x^633+84848*x^634-26340*x^635-5634*x^636+40135*x^637-17527*x^638-139991*x^639-66912*x^640+155655*x^641+13263*x^642-56872*x^643-199551*x^644-50946*x^645+15816*x^646-40485*x^647-2615*x^648+80392*x^649+32051*x^650-38389*x^651-111528*x^652+34080*x^653-72650*x^654+61913*x^655-70919*x^656-78982*x^657-63906*x^658+4987*x^659+85398*x^660+92332*x^661+10581*x^662-9507*x^608-8820*x^609-54207*x^610-5942*x^611+59155*x^612+65755*x^613+10436*x^614+15343*x^615+12258*x^585-11797*x^586-6093*x^587-254632*x^588-96200*x^589+36506*x^590+36557*x^591-94285*x^592-20095*x^593-26461*x^594+72331*x^595+95209*x^569+45876*x^570+69083*x^571+124047*x^572-38339*x^573+219104*x^574+35058*x^575+25981*x^576-11144*x^577+64388*x^578+114*x^579-19025*x^580+34757*x^581-3375*x^582-10633*x^583+94099*x^584-132267*x^596-100986*x^597+45027*x^598+19888*x^599+2646*x^600-103029*x^601-132542*x^602+25637*x^603+26933*x^604+50175*x^605-70515*x^606+3540*x^607+33620*x^663-104162*x^664-73999*x^665-94129*x^666+94415*x^667+31830*x^668-5005*x^669+27726*x^670+44480*x^671+54227*x^672+50650*x^673-36618*x^674+73854*x^675-83478*x^676+83430*x^677-12361*x^678+15746*x^679-67222*x^680+25389*x^681-59494*x^682+51009*x^683+5516*x^684+26886*x^685+39793*x^686-124334*x^687-104667*x^688-22802*x^689-59040*x^690-35282*x^691+20231*x^692-10146*x^693+10496*x^694-70312*x^695-23451*x^696+25991*x^697+55521*x^698-113503*x^699-134070*x^700-113543*x^701-137007*x^702+61726*x^703-33606*x^704+53903*x^705-40448*x^706+4894*x^707-107277*x^708-86139*x^709-68962*x^710-16212*x^711+6491*x^712-79377*x^713-27934*x^714-7561*x^715-67725*x^716+45970*x^717+216735*x^718-137176*x^719-83295*x^720+5746*x^721-48014*x^722+39883*x^723+60457*x^724+49097*x^725-3439*x^726-11151*x^727-2207*x^728+70984*x^729-16530*x^730-66203*x^731-13069*x^732+56823*x^733-28636*x^734+35635*x^735+7024*x^736-13647*x^737-141812*x^738+3551*x^739+40067*x^740+106430*x^741-60146*x^742-84142*x^743-56336*x^744-35018*x^745-28114*x^746+48357*x^747+83943*x^748-622*x^749-91288*x^750-60778*x^751+35656*x^752-29322*x^753+72416*x^754-33714*x^755-53150*x^756-61948*x^757-26773*x^758-14013*x^759+44150*x^760+102179*x^761-45084*x^762-21220*x^763-48465*x^764+76913*x^765-44733*x^766-14166*x^767+55934*x^768+51115*x^769+46692*x^770-50553*x^771+30072*x^772+84302*x^773+105576*x^774+15638*x^775-83937*x^776+57*x^777+50156*x^778+40385*x^779+1895*x^780+72976*x^781+98109*x^782-17711*x^783+36796*x^784-11557*x^785-53085*x^786+33967*x^787-43234*x^788-7853*x^789+80450*x^790+148118*x^791-13683*x^792-47086*x^793-45213*x^794+7230*x^795-35*x^796+64324*x^797-41719*x^798-3774*x^799-87888*x^800-2764*x^801+34199*x^802+28067*x^803+57152*x^804+6485*x^805-10129*x^806-52006*x^807-79247*x^808-31185*x^809+16696*x^810+49617*x^811-12422*x^812-113676*x^813-45421*x^814-21624*x^815+71790*x^816+85697*x^817-31184*x^818-19922*x^819-42000*x^820-64032*x^821-127714*x^822+75365*x^823+51420*x^824-35002*x^825+1706*x^826+33387*x^827-9716*x^828-19761*x^829-25702*x^830-37203*x^831+7663*x^832-75159*x^833+5565*x^834-9322*x^835-31941*x^836+21325*x^837-2845*x^838-34434*x^839-10368*x^840+2185*x^841-27934*x^842-16267*x^843-41163*x^844+25895*x^845+40393*x^846+3941*x^847-21991*x^848-6784*x^849+18050*x^850-90653*x^851+60321*x^852+8423*x^853+13572*x^854-156301*x^855-21806*x^856+35085*x^857-15085*x^858+54547*x^859-47800*x^860-5254*x^861-81781*x^862-29440*x^863+22184*x^864-46117*x^865-59070*x^866+17927*x^867+33855*x^868-3151*x^869-29846*x^870-65956*x^871+52289*x^872-35287*x^873+54512*x^874-83059*x^875-37459*x^876-2672*x^877-8033*x^878+35411*x^879+15932*x^880+17557*x^881-102416*x^882-758*x^883+33596*x^884+88484*x^885+12872*x^886+2572*x^887+46754*x^888+1808*x^889+39711*x^890-17346*x^891+93893*x^892+27086*x^893+41025*x^894+55423*x^895+10557*x^896+15809*x^897-41601*x^898+15383*x^899+56744*x^900+48292*x^901+12460*x^902+47575*x^903+7133*x^904-4963*x^905-36508*x^906+40056*x^907+24227*x^908+21353*x^909+27255*x^910+38360*x^911+17237*x^912+10553*x^913+85724*x^914+41962*x^915+63340*x^916-14715*x^917-15090*x^918-10060*x^919+55422*x^920+39387*x^921+18064*x^922+49105*x^923+30516*x^924+67754*x^925-39611*x^926-1869*x^927+17979*x^928+56851*x^929+26558*x^930+20527*x^931+24521*x^932-9309*x^933+8971*x^934+14831*x^935+68598*x^936+33539*x^937+4457*x^938+20848*x^939+14657*x^940-7254*x^941-41834*x^942+22774*x^943+18000*x^944+48189*x^945+6981*x^946+26315*x^947+1341*x^948-16146*x^949-27103*x^950+21402*x^951+38473*x^952+10482*x^953-23425*x^954+19125*x^955+3086*x^956+5820*x^957+27199*x^958+15683*x^959+14548*x^960+16778*x^961-21663*x^962-10322*x^963-56927*x^964+19710*x^965-16079*x^966+2508*x^967-9465*x^968+20640*x^969-10146*x^970-22758*x^971-12289*x^972-19208*x^973-37003*x^974+2892*x^975-17318*x^976-2994*x^977-10592*x^978-11231*x^979+20238*x^980+23234*x^981-2207*x^982-11822*x^983-15862*x^984+2933*x^985-7984*x^986-9333*x^987+9463*x^988+21888*x^989+5011*x^990+12439*x^991+2736*x^992-17243*x^993-22202*x^994+6028*x^995+16742*x^996+8153*x^997-8041*x^998-941*x^999+11*x^1000");
  temp[1] = may_parse_str ("-6336-11784*x+4932*x^500+50975*x^499+97099*x^498-15185*x^50-35458*x^49-3044*x^48+13767*x^51+19234*x^61-17894*x^63+18735*x^64-19974*x^65-74185*x^66+24966*x^67+20533*x^68-4341*x^69+7991*x^70+55079*x^71-50140*x^72-62180*x^73+39666*x^74+26552*x^75+427*x^76+35692*x^77-22108*x^78+1224*x^79-29889*x^80-7251*x^62+43*x^52+19785*x^54-25981*x^55-17850*x^56-8708*x^57-5910*x^58-17200*x^59+33737*x^60-1968*x^53-27426*x^81+76764*x^82+37352*x^83-7468*x^84-32889*x^85-2395*x^86+34476*x^87-30999*x^88+26660*x^89-23550*x^90-11040*x^91-3042*x^92+55160*x^93+34254*x^94-40638*x^95-21061*x^96-40800*x^97+5796*x^98+60663*x^99+50901*x^100+22063*x^35-31440*x^34-27849*x^33+7755*x^32+110306*x^426-115497*x^425+20415*x^424+67595*x^423+28928*x^422+42735*x^421+9196*x^420+65649*x^419+14523*x^418-122033*x^417-60312*x^416-80286*x^415-68129*x^414-41662*x^413-99035*x^412+90631*x^411+18885*x^410+20285*x^409+103847*x^408+49*x^407-78236*x^406+146563*x^405+51046*x^404+28921*x^403+8513*x^402+93930*x^401+57512*x^400-151248*x^434-32796*x^433-6320*x^432-13153*x^431+39244*x^430+1207*x^429-125494*x^428+75594*x^427+71100*x^438-104332*x^437+33349*x^436+17541*x^435-36707*x^439+3468*x^465-37426*x^464+4923*x^463-68389*x^462+37190*x^461-153141*x^460+9355*x^459+58415*x^458-26337*x^457-33589*x^456-60647*x^455-96122*x^454+3772*x^453+48878*x^452+47805*x^451-9815*x^450-19456*x^449+41357*x^448+109367*x^447-70859*x^446-26327*x^445-56764*x^444+112709*x^443+84028*x^442-114405*x^441+13469*x^440+75154*x^476-13917*x^475+89884*x^474+40702*x^473-102123*x^472-52269*x^471-54844*x^470-31354*x^469+35506*x^468-26591*x^467+40325*x^466+25113*x^42-38860*x^41-32253*x^40-18775*x^39+24889*x^38-3913*x^37+9886*x^36+25961*x^44+20718*x^43+53782*x^45+148743*x^497-71348*x^496+38970*x^495-69447*x^494+119334*x^493+209256*x^492+16894*x^491+17755*x^490-13091*x^489+61269*x^488+60877*x^487-34956*x^486-3555*x^485+7309*x^484+44127*x^483+2004*x^482-77975*x^481+37224*x^480-53310*x^479-99789*x^478-8866*x^477-26350*x^46+2664*x^47-28517*x^266-9030*x^265-56243*x^264+2532*x^263-148743*x^262-47669*x^261-2217*x^260+76152*x^259+33687*x^258-76494*x^267+27600*x^297-78630*x^293-68075*x^292-10407*x^291-40497*x^290-53532*x^289+22767*x^288-919*x^287-26600*x^286-19633*x^285+19684*x^284-11088*x^283-44447*x^282+29042*x^281+49415*x^280-23335*x^279+34497*x^278-7428*x^277-9528*x^276+2950*x^275-12454*x^274+80274*x^273-11356*x^272-8544*x^271+52315*x^270-86635*x^269-11482*x^268+18471*x^31+2029*x^30+6484*x^29+11951*x^28-17347*x^27+7705*x^26+15541*x^323-46446*x^322-86117*x^321-7060*x^320+9858*x^319-102444*x^318+112000*x^317+62657*x^333-21036*x^332-47391*x^331-25301*x^330+1843*x^329-329*x^328+46388*x^327-45259*x^326+95490*x^325+22935*x^324-79440*x^334+52306*x^345-56880*x^344+61293*x^343-74447*x^342-18731*x^341+25963*x^340+37668*x^339+10946*x^338-6400*x^337-56723*x^336+18902*x^335-6560*x^371+67845*x^370+30644*x^369-33743*x^368-93463*x^367-15362*x^366+15340*x^365+46588*x^364-33466*x^363+73487*x^362-54280*x^361-173369*x^360-28330*x^359-52960*x^358+30366*x^357+20899*x^356+59310*x^355+85864*x^354+36605*x^353+40710*x^352-51058*x^351+7116*x^350+37369*x^349+98835*x^348+80705*x^347+62567*x^346-678*x^373+18456*x^372-65734*x^316+32304*x^315-47716*x^314-157705*x^313-60347*x^312+35891*x^311-52771*x^310-54138*x^309-62416*x^308-18320*x^307+42911*x^306-51152*x^305+42328*x^304-64533*x^303+37605*x^302+30404*x^301+32050*x^300+76679*x^299-99895*x^298+59164*x^296-106932*x^295+106931*x^294+24565*x^399+101405*x^398+76345*x^397+34939*x^396+84599*x^395-129439*x^394+127572*x^393+27839*x^392+7132*x^391+58203*x^390-42361*x^389-9644*x^388+17227*x^387-90659*x^386+56714*x^385+19616*x^384-9827*x^383-49144*x^382+51479*x^381+548*x^380+101139*x^379-51395*x^378-9396*x^377+40716*x^376+15155*x^375-85129*x^374+45539*x^145-49187*x^144-84561*x^143+14981*x^142-78934*x^141+38549*x^140+46695*x^139+30137*x^138+18715*x^137+7978*x^136-105944*x^135-38644*x^134-25305*x^133+69515*x^132+49336*x^131+38646*x^130+10128*x^129+7578*x^128-45907*x^127-34515*x^126+745*x^125+51674*x^124-13931*x^123+47163*x^122-39146*x^121-38791*x^120+12741*x^12+2341*x^11-19627*x^10-21728*x^9+2501*x^8-62303*x^156+99627*x^155+43811*x^154-35273*x^153-25451*x^152-71627*x^151+14225*x^150+10629*x^149+101280*x^148+30343*x^147-107015*x^146+41606*x^177-5407*x^176+1429*x^175-36378*x^174+59320*x^173-47308*x^172+99354*x^171-31347*x^170-61573*x^169-11818*x^168-45715*x^167-58723*x^166+12914*x^165-56185*x^164-7108*x^163-84181*x^162+24633*x^161+94005*x^160-126837*x^159-1879*x^158-9439*x^157+21163*x^15-8494*x^18-25839*x^17-1561*x^16-5662*x^14+3974*x^13+44403*x^119-33107*x^118-21042*x^117+33494*x^116-17715*x^115-53687*x^114-13088*x^113+140*x^112-15245*x^111-33628*x^110+13219*x^109+40524*x^107+67247*x^106-8625*x^105-59664*x^104-49230*x^103-51057*x^102+51292*x^101-9917*x^20+6081*x^201+46891*x^200-47454*x^199-62696*x^198+41710*x^197+18905*x^196+46484*x^195+39763*x^194+5584*x^193-80122*x^192-68819*x^191-61225*x^190+65582*x^189-76743*x^188+31261*x^187+24627*x^186-43917*x^185+29632*x^184-5935*x^183-78409*x^182+54977*x^181+7914*x^180+18263*x^179+36344*x^178+17764*x^228+39636*x^227+72602*x^226-37408*x^225+24660*x^224-19506*x^223-57444*x^222-18625*x^221+75768*x^220-54505*x^219-1311*x^218-24631*x^217-66612*x^216-123582*x^215-91923*x^214+35544*x^213-42258*x^212-21479*x^211+78630*x^210-69830*x^209-68336*x^208-7103*x^207+21333*x^206+26090*x^205-2844*x^204+22078*x^203+89181*x^202+121555*x^234+87427*x^233+39975*x^232+19237*x^231-13255*x^230-37761*x^229-6025*x^24+58779*x^252+93840*x^251+16027*x^250+137937*x^249-70759*x^248+29186*x^247-32044*x^246-7642*x^245+41519*x^244+2409*x^243-51568*x^242-25984*x^241-60334*x^240+30556*x^239-7279*x^238-58845*x^237-15424*x^236-51954*x^235+63439*x^257+48527*x^256+66625*x^255-86121*x^254-16328*x^253+21835*x^25+20507*x^23+28454*x^22-11264*x^21-2287*x^19+5863*x^4+2059*x^3-6862*x^2+5251*x^7+17382*x^6+5646*x^5+53693*x^108-2220*x^501-178171*x^502+35562*x^503+86592*x^504+115360*x^505-21838*x^506-5983*x^507+133517*x^508-62778*x^509+14689*x^510+21219*x^544+78574*x^546+70049*x^547-2225*x^548+6810*x^549+1272*x^550+54800*x^551-15350*x^552+14022*x^553-14642*x^545-12293*x^554+34987*x^555-80945*x^556+65048*x^557+4396*x^558+5987*x^559+59996*x^560-25921*x^561-20674*x^562-73724*x^563-86051*x^564+57444*x^565+30358*x^566-47115*x^567+83676*x^568+32647*x^511-56107*x^512+160040*x^513+37815*x^514+73084*x^515+19498*x^516+80908*x^517-69505*x^518+57795*x^519+44871*x^520+117699*x^521-17791*x^522+77150*x^524-47800*x^525-88448*x^526-124619*x^527-162383*x^528+108698*x^529-72699*x^530+62382*x^531-17169*x^532-15565*x^533-75333*x^534-106966*x^535-37986*x^536+44573*x^537-36631*x^538+42208*x^539-5662*x^540+57347*x^541+90315*x^542+154755*x^543+5844*x^523+25893*x^616+6367*x^617-70773*x^618+32464*x^619-10751*x^620+9236*x^621-45776*x^622-85970*x^623+60399*x^624+13210*x^625+22899*x^626-101228*x^627+67283*x^628-49406*x^629-93221*x^630-17611*x^631-81861*x^632-80851*x^633-40863*x^634+101734*x^635+84815*x^636+93493*x^637-93707*x^638+60197*x^639+34746*x^640+71174*x^641+116940*x^642+19311*x^643-38383*x^644+27902*x^645-90481*x^646+35257*x^647+70607*x^648+62710*x^649+86763*x^650+37765*x^651-24756*x^652+79081*x^653-36572*x^654+71500*x^655+10004*x^656+10307*x^657-26116*x^658+40391*x^659+29403*x^660+48382*x^661+40371*x^662+17964*x^608-66405*x^609+84023*x^610+108021*x^611-57828*x^612+17896*x^613-30899*x^614+5573*x^615+80902*x^585+40394*x^586-95297*x^587-39918*x^588-21629*x^589-102865*x^590+18055*x^591+17661*x^592+70475*x^593+18799*x^594-62172*x^595+6542*x^569-63042*x^570-61611*x^571-804*x^572+30421*x^573-827*x^574+84135*x^575-93535*x^576-36838*x^577-31079*x^578-10732*x^579-152986*x^580+25187*x^581-78780*x^582-104612*x^583-82337*x^584-32116*x^596-18344*x^597+111809*x^598-3677*x^599-73528*x^600-74039*x^601+60655*x^602-7301*x^603-52559*x^604-50970*x^605-147067*x^606-87111*x^607+68281*x^663+8767*x^664+66218*x^665-1593*x^666+51951*x^667-42881*x^668-58854*x^669-41820*x^670+121478*x^671-29364*x^672+67316*x^673-108952*x^674-63036*x^675-63736*x^676-60490*x^677+26749*x^678+26778*x^679-6816*x^680-56263*x^681-103299*x^682+56220*x^683-56912*x^684+105806*x^685+18461*x^686-208*x^687-135502*x^688+40890*x^689-78405*x^690+6671*x^691+101024*x^692+20806*x^693-73654*x^694-4110*x^695+34849*x^696+83609*x^697-91702*x^698-10490*x^699-49417*x^700+72413*x^701-115808*x^702+67997*x^703-28999*x^704-15903*x^705-18810*x^706-64148*x^707-51673*x^708+71240*x^709+11167*x^710-25747*x^711+38696*x^712+33736*x^713-78299*x^714-72221*x^715+16201*x^716+102446*x^717-79882*x^718-64009*x^719-18865*x^720+20202*x^721-83188*x^722+13949*x^723-58788*x^724+27369*x^725-127822*x^726-47406*x^727-96352*x^728+77832*x^729-7570*x^730-68492*x^731-70303*x^732-33187*x^733+11249*x^734-19185*x^735-24464*x^736-27739*x^737-91427*x^738-2873*x^739+112169*x^740+52050*x^741+49872*x^742-27512*x^743-68687*x^744-21489*x^745+48460*x^746+25736*x^747+19330*x^748+36807*x^749-111154*x^750-40184*x^751-85849*x^752+30258*x^753+64448*x^754+18727*x^755+19609*x^756-101909*x^757-110682*x^758-56025*x^759+15190*x^760+60986*x^761-25770*x^762+1145*x^763-29242*x^764-40822*x^765+27055*x^766+69747*x^767-41493*x^768-38067*x^769+10638*x^770-12425*x^771+32665*x^772-50799*x^773-24851*x^774-37999*x^775+81972*x^776+13712*x^777+21142*x^778+80045*x^779+10989*x^780-18919*x^781+20331*x^782+1981*x^783+81220*x^784+93490*x^785+86203*x^786-6184*x^787-34669*x^788+2625*x^789+36488*x^790+98399*x^791+55871*x^792-12580*x^793+29981*x^794-11495*x^795-4120*x^796-11600*x^797+120764*x^798+27277*x^799-12511*x^800-22539*x^801+65689*x^802+59797*x^803-42505*x^804+54730*x^805+81002*x^806+122534*x^807-34284*x^808+35794*x^809+23785*x^810+93547*x^811+81437*x^812-75417*x^813-14919*x^814+92631*x^815+39042*x^816+41033*x^817+9437*x^818-2449*x^819-73120*x^820+21390*x^821-4992*x^822+113498*x^823-2845*x^824-66107*x^825-32548*x^826+1570*x^827+30833*x^828+12646*x^829-11235*x^830-50766*x^831+89265*x^832-43042*x^833-1536*x^834+77853*x^835-45779*x^836-22918*x^837-52773*x^838-6208*x^839-26906*x^840+88824*x^841-31306*x^842+47841*x^843-50295*x^844+10093*x^845-44660*x^846-14687*x^847+54329*x^848-76237*x^849-10938*x^850-31387*x^851+52803*x^852+52921*x^853-73013*x^854+61459*x^855-54150*x^856-14971*x^857-79759*x^858+57004*x^859-6337*x^860+33853*x^861-75884*x^862-29897*x^863-24566*x^864-33404*x^865+20385*x^866-10007*x^867+30032*x^868+25466*x^869-132998*x^870-7827*x^871-42954*x^872+74252*x^873-68670*x^874+37229*x^875-30501*x^876+39224*x^877-41751*x^878+40396*x^879-11037*x^880-33488*x^881-44421*x^882-23593*x^883+15648*x^884+49701*x^885-36712*x^886+45061*x^887-23462*x^888+35585*x^889-47581*x^890-24803*x^891+16116*x^892-15421*x^893-72965*x^894+8773*x^895+4501*x^896+21015*x^897-947*x^898-10723*x^899-4722*x^900-30308*x^901-25646*x^902+11311*x^903-7746*x^904+63211*x^905-56647*x^906-72333*x^907+11126*x^908+19959*x^909+15870*x^910-27557*x^911+24503*x^912+23184*x^913-51436*x^914-78049*x^915+20443*x^916+16088*x^917-1045*x^918+826*x^919-34245*x^920+24401*x^921-43788*x^922+46959*x^923-48763*x^924-10333*x^925-27312*x^926-4637*x^927+6597*x^928-16296*x^929+59310*x^930-4379*x^931+6115*x^932-45278*x^933+16605*x^934-35307*x^935-32384*x^936+43345*x^937-6473*x^938+18593*x^939-54643*x^940+30580*x^941-43187*x^942-37941*x^943+16887*x^944+34660*x^945+57995*x^946-39793*x^947-15020*x^948-28081*x^949+29576*x^950-20978*x^951-2490*x^952+39058*x^953+31933*x^954+23666*x^955-45171*x^956+13372*x^957+33063*x^958+24160*x^959+14389*x^960+1634*x^961-7927*x^962-28327*x^963+14751*x^964-23861*x^965+68400*x^966+11323*x^967+490*x^968-29450*x^969-19011*x^970-14395*x^971+19216*x^972+24061*x^973+34798*x^974+485*x^975-11013*x^976-17932*x^977+4632*x^978+6966*x^979+19961*x^980+16933*x^981+2852*x^982-5265*x^983-12204*x^984-8039*x^985+25412*x^986+12248*x^987-4338*x^988+6385*x^989+6283*x^990-20229*x^991-19051*x^992+1166*x^993+19521*x^994-1835*x^995-6702*x^996+483*x^997+5937*x^998-7163*x^999-924*x^1000");
  a = may_gcd (2, temp);
  check_bool (a != NULL);
  check (a, "11*x^500+93*x^499-14*x^498-99*x^497-67*x^496+68*x^495+45*x^494+76*x^493+6*x^492+72*x^491-28*x^490-61*x^489-59*x^488+6*x^487-87*x^486+72*x^485-46*x^484-68*x^483-42*x^482-47*x^481-32*x^480+37*x^479-93*x^478-58*x^477-90*x^476-53*x^475-69*x^474-84*x^473+46*x^472+59*x^471-56*x^470-83*x^469-91*x^468+92*x^467-93*x^466+91*x^465-54*x^464+10*x^463-77*x^462-63*x^461-90*x^460+61*x^459-3*x^458-82*x^457+16*x^456-40*x^455+21*x^454-94*x^453-98*x^452+75*x^451+39*x^450+95*x^449-68*x^448+98*x^447-36*x^446-95*x^445+8*x^444+92*x^443+8*x^442-95*x^441-18*x^440+44*x^439+66*x^438-62*x^437+40*x^436+68*x^435-67*x^434+68*x^433-65*x^432+43*x^431+6*x^430+39*x^429-67*x^428+8*x^427+20*x^426+93*x^425+45*x^424+81*x^423+8*x^422-44*x^421-80*x^420-5*x^419+23*x^418+34*x^417-81*x^416-95*x^415+63*x^414-24*x^413-63*x^412-36*x^411+85*x^410+35*x^409-67*x^408+19*x^407+60*x^405+95*x^404+95*x^403-18*x^402+46*x^401-20*x^400+52*x^399+31*x^398+65*x^397+46*x^396-24*x^395+63*x^394+73*x^393+76*x^392-68*x^391+67*x^390-38*x^389+23*x^388+14*x^387+58*x^386-92*x^385-8*x^384+36*x^383-90*x^382+96*x^381-19*x^380-4*x^379+38*x^378-61*x^377-58*x^376-91*x^375+45*x^374-79*x^373-27*x^372+32*x^371-24*x^370-46*x^369+12*x^368+81*x^367+63*x^366-85*x^365-36*x^364-35*x^363+11*x^362+90*x^361+31*x^360+47*x^359-50*x^358-54*x^357+71*x^356-71*x^355-17*x^354+64*x^353-34*x^352+60*x^351+99*x^350+48*x^349+36*x^348+73*x^347-40*x^346+49*x^345+55*x^344+24*x^343+65*x^342+6*x^341-65*x^340+52*x^339+99*x^338-39*x^337+49*x^336-76*x^335-15*x^334+44*x^333-98*x^332+27*x^331+36*x^330+18*x^329+32*x^328+5*x^327+8*x^325+63*x^324-80*x^323-97*x^322+25*x^321-76*x^320-53*x^319-57*x^318-99*x^317+90*x^316+86*x^315-84*x^314+8*x^313-10*x^312+63*x^311-39*x^310-48*x^309-94*x^308+10*x^307-20*x^306+70*x^305+98*x^304+58*x^303+45*x^302-12*x^301+28*x^300+3*x^299+13*x^298-5*x^297-45*x^296-41*x^295+84*x^294-49*x^293-79*x^292+93*x^291-81*x^290-47*x^289+28*x^288+73*x^287-24*x^286+8*x^285+27*x^284-18*x^283+62*x^282+74*x^281+24*x^280+3*x^279+32*x^278+65*x^277-26*x^276+87*x^275+33*x^274+76*x^273-57*x^272-80*x^271-92*x^270+81*x^269+77*x^268-87*x^267+50*x^266+74*x^265-60*x^264+19*x^263-68*x^262+78*x^261+34*x^260+66*x^259-53*x^258+59*x^257+28*x^256+38*x^255+6*x^254-50*x^253-47*x^252+75*x^251+67*x^250-79*x^249+53*x^248+63*x^247-24*x^246-29*x^245-4*x^244-21*x^243-75*x^241+93*x^240+23*x^239-16*x^238+83*x^237-17*x^236+51*x^235-15*x^234+22*x^233-8*x^232+53*x^231+84*x^230-98*x^229-5*x^228+20*x^227-72*x^226+10*x^225-49*x^224-31*x^223+33*x^222-80*x^221-44*x^220-21*x^219-5*x^218+21*x^216-92*x^215-70*x^214-95*x^213-47*x^212+51*x^211+47*x^210+5*x^209+33*x^208+9*x^207+26*x^206-55*x^205+37*x^204-94*x^203-65*x^202+90*x^201-38*x^200-46*x^199+28*x^198+88*x^197+64*x^196-22*x^195-37*x^194+68*x^193-84*x^192+80*x^191+23*x^190-20*x^189-7*x^188+4*x^187-77*x^186+40*x^185+21*x^184+55*x^183+61*x^182+5*x^181+66*x^180-83*x^179-26*x^178+27*x^177+7*x^176+97*x^175+31*x^174-45*x^173+24*x^172-75*x^171+16*x^170-44*x^169-70*x^168-23*x^167-67*x^166-97*x^165-22*x^164+67*x^163+77*x^162+67*x^161+9*x^160-90*x^159+17*x^158-7*x^157+18*x^156+58*x^155+35*x^154-50*x^153-93*x^152+14*x^151-27*x^150-50*x^149+58*x^148+46*x^147-84*x^146-10*x^145-84*x^144-34*x^143-66*x^142+83*x^141-86*x^140+39*x^139+70*x^138+65*x^137+95*x^136+90*x^135-52*x^134-69*x^133+32*x^132+54*x^131+56*x^130+93*x^129-67*x^128+94*x^127-84*x^126-84*x^125-68*x^124+70*x^123-79*x^122-3*x^121-34*x^120+78*x^119-82*x^118+39*x^117-89*x^116-22*x^115+77*x^114-22*x^113+90*x^112+20*x^111-26*x^110+27*x^109-32*x^108+85*x^107+74*x^106+46*x^105-50*x^104-2*x^103-45*x^102-74*x^101-60*x^100-x^99-31*x^98-43*x^97+73*x^96-82*x^95+75*x^94-66*x^93-90*x^92+76*x^91-63*x^90+63*x^89+69*x^88-89*x^87+99*x^86-9*x^85+44*x^84-18*x^83+37*x^82-2*x^81-43*x^80+8*x^79+96*x^78-89*x^77-58*x^76-81*x^75+58*x^74+35*x^73-78*x^72-69*x^71+95*x^70-68*x^69-67*x^68-2*x^67-83*x^66+21*x^65-54*x^63-18*x^62-94*x^61-37*x^60+66*x^59+57*x^58+28*x^57-6*x^56+11*x^55-34*x^54-75*x^53-12*x^52+11*x^51-56*x^50+6*x^49+55*x^48+14*x^47-92*x^46+38*x^45-15*x^44+10*x^43-18*x^42+78*x^41-26*x^40-64*x^39-8*x^38-x^37-79*x^36+3*x^35+35*x^34+18*x^33+7*x^32-52*x^31+28*x^30-7*x^29+95*x^28-42*x^27+3*x^26+96*x^25-87*x^24+92*x^23+56*x^22-46*x^21-95*x^20+33*x^19+40*x^18+32*x^17-24*x^16-96*x^15-95*x^14-17*x^13+80*x^12+80*x^11+67*x^10-60*x^9-99*x^8-85*x^7+44*x^6+59*x^5-13*x^4+83*x^3+5*x^2-57*x-72");
	may_compact (mark, NULL);
  }

  temp[0]=may_parse_str ("-(7+x)^2*(5+x)^8*(3+x)^8*(1+x)^8*((5+x)^2*(3+x)^4*(1+x)^4*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))^2*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-((5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))*(1+x)^2*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)^2*(2+x)*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^2*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^2*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2)*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(6+x)^4*(4+x)^6*(2+x)^8*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^4+(5+x)^6*(3+x)^8*(1+x)^8*((5+x)^2*(3+x)^4*(1+x)^4*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))^2*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-((5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))*(1+x)^2*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)^2*(2+x)*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^2*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^2*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2)*((7+x)*(3+x)*(1+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(6+x)*(2+x))^2*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(6+x)^2*(4+x)^8*(2+x)^8*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^3-(7+x)^2*(5+x)^6*(3+x)^8*(1+x)^8*((5+x)^2*(3+x)^4*(1+x)^4*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))^2*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-((5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))*(1+x)^2*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)^2*(2+x)*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^2*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^2*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2)*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(6+x)^4*(4+x)^8*(2+x)^8*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^4-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2*((5+x)^2*(3+x)^4*(1+x)^4*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))*((7+x)*(3+x)*(1+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(6+x)*(2+x))*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(7+x)*(5+x)^3*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(7+x)*(5+x)^3*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^3*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-((5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))*(1+x)^2*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)^2*(2+x)*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((7+x)*(5+x)*(3+x)*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(7+x)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)*(2+x)*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))*(1+x)^2*((7+x)*(3+x)*(1+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2+(7+x)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(7+x)*(5+x)^4*(3+x)^3*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^3*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2)^2-(7+x)^2*(5+x)^8*(3+x)^8*(1+x)^8*((5+x)^2*(3+x)^4*(1+x)^4*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))^2*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-((5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))*(1+x)^2*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)^2*(2+x)*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^2*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^2*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2)*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(6+x)^2*(4+x)^8*(2+x)^8*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^4-(5+x)^8*(3+x)^8*(1+x)^8*((5+x)^2*(3+x)^4*(1+x)^4*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))^2*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-((5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))*(1+x)^2*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)^2*(2+x)*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^2*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^2*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2)*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(6+x)^4*(4+x)^8*(2+x)^8*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^4-(5+x)^6*(3+x)^6*(1+x)^6*((5+x)^2*(3+x)^4*(1+x)^4*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))^2*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-((5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))*(1+x)^2*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)^2*(2+x)*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^2*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^2*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2)*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^6*((7+x)*(5+x)*(3+x)*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(7+x)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)*(2+x)*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))*(1+x)^2*((7+x)*(3+x)*(1+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2+(7+x)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(2+x)^6*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^3");
  temp[1]=may_parse_str("-(7+x)^2*(5+x)^8*(3+x)^8*(1+x)^8*((5+x)^2*(3+x)^4*(1+x)^4*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))^2*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-((5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))*(1+x)^2*((1+x)*(6+x)*(4+x)*(2+x)+(5+x)*(3+x)*(1+x)*(6+x)+(5+x)*(3+x)*(1+x)*(2+x)+(5+x)*(6+x)*(4+x)*(2+x))*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)*(1+x)^2*(6+x)*(4+x)^2*(2+x)*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)+(5+x)*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^4*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^2*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2-(5+x)^4*(3+x)^2*(1+x)^4*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))*(6+x)^2*(4+x)^4*(2+x)^4*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^2)*((3+x)^2*((5+x)*(1+x)*(4+x)+(5+x)*(1+x)*(2+x)+(5+x)*(3+x)*(4+x)+(3+x)*(1+x)*(2+x))^2*(1+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)-(5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2))^2*(6+x)^4*(4+x)^8*(2+x)^8*((3+x)^2*(1+x)^2*(4+x)^2+(1+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(4+x)^2*(2+x)^2+(3+x)^2*(1+x)^2*(2+x)^2)^4");
  a = may_gcd (2, temp);
  check_bool(a != NULL);
  a = may_expand (a);
  check (a, "(59684528608115397673535017858577203200000000000000+5899694138249710068513804370349753568683867116917893584941580288*x^50+1011331325875654043890745344*x^99+438284133159440046668823487492422328765850162096405317833700966400*x^42+356296224624822018943663786511302502847741952*x^83+9004110592278129402022964429271971687519131174626860862873272320*x^18+2618883576389839156608169399724106947276333532869558272*x^70+28473853805880041887829455452797342956283694779522404319232*x^63+7820926505193751028077080739840*x^96+2292906414273036113928102379649841203845599485426495639974171377664*x^31+2454709673858593523950590095417191956894432061436866314577447485440*x^34+178324312736628324200053613704985294866273106586695499776*x^67+203577510422319367772124804092974935343243729710394627850240000*x^14+114021207549157301231738908296831503386056310173297747866943488*x^55+15414067200*x^109+19035006703017356502193577492779606971776872305298453299200000*x^12+39767786910138791568582190461472857512964962095542754335637897216*x^47+29034194935347003536436515601121280*x^93+132963736890789960992494269595899914856323219456*x^80+850982918140733962942531477031222609622181285615146538893312*x^60+1085976895599265764822761316385174495012044673035825370294313287680*x^27+15287947181424640*x^106+1227523605688662101410409639414541417256897111640978227200000*x^10+1144078806794266380249142708878535703383101420146312394497156448256*x^39+32484832455257938990378057497035360790480066969600*x^77+8493594779627147969204038617343951249986607569052394455040*x^64+1364483680736284907787960587373461629922965175292528467911376896*x^52+1568087349787649638400*x^103+228479297211629105671575272656354633526732944381126966544816930816*x^23+57480387421272331499324090935133175808*x^90+21880683811850914201771009991748819703865961586705819371450531840*x^48+130594465644119363292511623970816*x^95+75214230621475387839604013956044867485041774881937061034799923200*x^21+2693102232678717608293289627228861221686247424*x^82+284964392386444258667129171044518893033293719588719188508672*x^61+3892829508322627072090227839952796661817341932641346801945804800*x^17+672325033430520887858324497036273211273449696501503459328*x^66+8697984445289054024106287664058965490165880224481280000000*x^7+2241436057600*x^108+1273312972689523436584626376896398839964317935206400000000*x^6+389343639338413947890162446882570240*x^92+865562141711376130211773159415332631064301703076128737266191695872*x^40+870234381995517766547152456308263887740041691136*x^79+619684127761801676974197995324647285319084206624656652345016320*x^53+860334874091847680*x^105+157856754254372216373069708744234630066475106304000000000*x^5+185566485316117332050251680189562190688830678368256*x^76+2454681040096028330461506944217816655350878758224560528074983079936*x^32+2044419449711546508449006546021233293854227071731471731066081902592*x^30+796056649823353757353041268956808714644353236256703677330450546688*x^26+53316613654340532830208*x^102+116887161700246050028132198527464514430875918090171759059880378368*x^45+636015455947707115404313875380125040640*x^89+16113951405591619186527820711323586268862873600000000000*x^4+3064786867547754407911447182957053214720000000000000*x+6766536492942785439577303118143263361784155340784494162247680*x^58+134952156473759606754148924513490365847779732637201709489973100544*x^22+607670175205522863893746197683018427400192*x^86+1764392012771280026965718136338722064063500204331763219793234624512*x^37+26648976560870467183844444510442193094636962945433600*x^73+39478344137349330880405480447727250618703909694986403077802688512*x^20+5438777042290476092535550306316701929231426453504*x^78+39897267977511239680*x^104+1300337714963905735204741201546204526778777600000000000*x^3+69523492754498485797183707299265428305109075968341156552239808512*x^46+4884489024502914611026330975671025664*x^91+1450853651015648334023576412509092358954694973765149102198185132032*x^38+1014514159725441035161188038182931559319336759459840*x^75+587447559717907390546679950066853938773514990166578267422720000*x^15+2445861039365785422898142510730609026887084781428491438915584*x^59+1568900156507294756912368294349968391283496513061543301087232000*x^16+65010674035642969725462416457727017197483472032252087500800000*x^13+2892019691423244460727596736644282312847367302119212630250749952*x^51+1593006559329833188065280*x^101+6631222738036419166932751783066849640448*x^88+77773057744599252388809628366622964056064000000000000*x^2+1738429583192869926913946773118285992503036371357379361588645986304*x^29+5080392740830507057212709365358588925170773378560163840000000*x^11+293564935983738098498343868531019532974608723170622038696559247360*x^43+5358896805837420945218694939706223938502656*x^85+128184418709939320163127423878288323958065609143615488*x^72+46198344507197332424270522617790308986059003507105410479685632*x^56+554060206507882566661368418012669958561776460058457413157289721856*x^25+21882400727392774440963047424*x^98+266308226068944592984665226565453081334232060174270464000000*x^9+2296338020699767948549784628627551506342621880360788962945548681216*x^35+11133305671625169802834252406100580241518012999250214912*x^69+19472384488017205632943408215209535734422907532251058338518794240*x^19+5311697032687670838572122043084510625247830418489344*x^74+51361035043136473399219151520753845822745825113014272000000*x^8+188942923163581466671195673715523017154453100093521095230398398464*x^44+65273831148339524263493008681313639333888*x^87+18020103291972780753438456715780639562424079811366273426128896*x^57+42343572409750001007198208*x^100+44832838692720529563180081253707068395225088*x^84+2056584876809668966393987770745733959398368944583165426302996447232*x^36+591444621042250306577720010480337194257942087885127680*x^71+1408400809625812544872525607003181999808895049883820719539857915904*x^28+52428800*x^110+11582518126272594083580507388562324225897939634038695469184712704*x^49+431889768618467030033506762752*x^97+45457429023717040502074083111050799002136355307420909568*x^68+91830573593132745665890624560818976994910118559908404854784*x^62+365655246583840357474621045263867770707906704841081409876407615488*x^24+2019568429395361196753421899988992*x^94+628550506740355704087594176245000259764085460706635153093843484672*x^41+19382518180653455735443062966244059379195969536*x^81+2510473276082672374888634967166379935926288222053079149987611279360*x^33+2436888638488344075156379781727687683410605423438115635200*x^65+270917155366616472980056220590073227557828125284620975767158784*x^54+214930869452800*x^107)");
  temp[0] = may_divexact (temp[0], a);
  temp[1] = may_divexact (temp[1], a);
  check_bool (temp[0] != NULL);
  check_bool (temp[1] != NULL);
  may_compact (mark, NULL);

  // Another heavy test
  temp[0] = may_parse_str("(9+x)^2*(7+x)^8*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^8*(5+x)^16*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(3+x)^16*(1+x)^16*((7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(6+x)^8*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^6*(2+x)^8-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(5+x)^6*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^6*(1+x)^6*(6+x)^6*(4+x)^6*(2+x)^6*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)*(2+x)+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2)^2-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))^2*(1+x)^8*(6+x)^6*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^6*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^6*(4+x)^8*(2+x)^8+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^8*(2+x)^8-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^3*(4+x)^4*(2+x)^4+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^3*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^3*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)*(2+x)+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2)*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^3*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^4*(4+x)^3*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^3*(1+x)^4*(8+x)*(6+x)^4*(4+x)^3*(2+x)^4)^2*(6+x)^4*(4+x)^4*(2+x)^4)*(8+x)^2*(6+x)^16*(4+x)^16*(2+x)^16+(9+x)^2*(7+x)^6*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^8*(5+x)^16*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(3+x)^16*(1+x)^16*((7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(6+x)^8*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^6*(2+x)^8-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(5+x)^6*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^6*(1+x)^6*(6+x)^6*(4+x)^6*(2+x)^6*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)*(2+x)+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2)^2-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))^2*(1+x)^8*(6+x)^6*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^6*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^6*(4+x)^8*(2+x)^8+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^8*(2+x)^8-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^3*(4+x)^4*(2+x)^4+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^3*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^3*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)*(2+x)+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2)*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^3*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^4*(4+x)^3*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^3*(1+x)^4*(8+x)*(6+x)^4*(4+x)^3*(2+x)^4)^2*(6+x)^4*(4+x)^4*(2+x)^4)*(8+x)^4*(6+x)^16*(4+x)^16*(2+x)^16-(7+x)^6*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^6*(5+x)^12*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(3+x)^12*(1+x)^12*((9+x)*(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^3*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^4*(4+x)^4*(2+x)^4+(9+x)*(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^3*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)*(2+x)+(9+x)*(7+x)*(3+x)*(8+x)*(6+x)*(4+x)*(2+x)+(9+x)*(7+x)*(5+x)*(3+x)*(1+x)*(6+x)*(2+x)+(9+x)*(7+x)*(5+x)*(3+x)*(1+x)*(8+x)*(4+x)+(9+x)*(5+x)*(1+x)*(8+x)*(6+x)*(4+x)*(2+x))*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(9+x)*(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^3*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^3*(1+x)^4*(8+x)*(6+x)^4*(4+x)^4*(2+x)^4-((9+x)*(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)*(4+x)^2*(2+x)^2+(9+x)*(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)-(5+x)^2*(3+x)^2*(1+x)^2*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)*(2+x)+(9+x)*(7+x)*(3+x)*(8+x)*(6+x)*(4+x)*(2+x)+(9+x)*(7+x)*(5+x)*(3+x)*(1+x)*(6+x)*(2+x)+(9+x)*(7+x)*(5+x)*(3+x)*(1+x)*(8+x)*(4+x)+(9+x)*(5+x)*(1+x)*(8+x)*(6+x)*(4+x)*(2+x))*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2+(9+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)*(2+x)^2+(9+x)*(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(8+x)*(6+x)*(4+x)^2*(2+x)^2)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^4*(4+x)^4*(2+x)^4+(9+x)*(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^3*(4+x)^3*(2+x)^4)^2*((7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(6+x)^8*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^6*(2+x)^8-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(5+x)^6*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^6*(1+x)^6*(6+x)^6*(4+x)^6*(2+x)^6*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)*(2+x)+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2)^2-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))^2*(1+x)^8*(6+x)^6*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^6*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^6*(4+x)^8*(2+x)^8+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^8*(2+x)^8-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^3*(4+x)^4*(2+x)^4+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^3*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^3*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)*(2+x)+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2)*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^3*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^4*(4+x)^3*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^3*(1+x)^4*(8+x)*(6+x)^4*(4+x)^3*(2+x)^4)^2*(6+x)^4*(4+x)^4*(2+x)^4)*(8+x)^2*(6+x)^12*(4+x)^12*(2+x)^12+(9+x)^2*(7+x)^8*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^8*(5+x)^14*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(3+x)^16*(1+x)^16*((7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(6+x)^8*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^6*(2+x)^8-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(5+x)^6*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^6*(1+x)^6*(6+x)^6*(4+x)^6*(2+x)^6*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)*(2+x)+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2)^2-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))^2*(1+x)^8*(6+x)^6*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^6*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^6*(4+x)^8*(2+x)^8+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^8*(2+x)^8-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^3*(4+x)^4*(2+x)^4+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^3*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^3*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)*(2+x)+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2)*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^3*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^4*(4+x)^3*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^3*(1+x)^4*(8+x)*(6+x)^4*(4+x)^3*(2+x)^4)^2*(6+x)^4*(4+x)^4*(2+x)^4)*(8+x)^4*(6+x)^16*(4+x)^16*(2+x)^16+(7+x)^8*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^8*(5+x)^16*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(3+x)^16*(1+x)^16*((7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(6+x)^8*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^6*(2+x)^8-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(5+x)^6*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^6*(1+x)^6*(6+x)^6*(4+x)^6*(2+x)^6*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)*(2+x)+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2)^2-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))^2*(1+x)^8*(6+x)^6*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^6*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^6*(4+x)^8*(2+x)^8+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^8*(2+x)^8-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^3*(4+x)^4*(2+x)^4+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^3*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^3*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)*(2+x)+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2)*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^3*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(8+x)*(6+x)^4*(4+x)^3*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^3*(1+x)^4*(8+x)*(6+x)^4*(4+x)^3*(2+x)^4)^2*(6+x)^4*(4+x)^4*(2+x)^4)*(8+x)^4*(6+x)^16*(4+x)^16*(2+x)^16+(9+x)^2*(7+x)^8*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^8*(5+x)^16*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(3+x)^16*(1+x)^16*((7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(6+x)^8*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*(1+x)^8*(8+x)^2*(6+x)^8*(4+x)^6*(2+x)^8-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(5+x)^6*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^6*(1+x)^6*(6+x)^6*(4+x)^6*(2+x)^6*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)*(2+x)+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(8+x)*(6+x)*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)*(1+x)^2*(8+x)*(6+x)^2*(4+x)^2*(2+x)^2-(5+x)^2*(3+x)^2*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2)^2-(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^3*(5+x)^8*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))^2-((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2*((7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)*(4+x)^2*(2+x)^2+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)*(2+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))*(4+x)^2*(2+x)^2*((7+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(6+x)+(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(1+x)*(2+x)+(7+x)*(5+x)*(6+x)*(4+x)*(2+x))+(7+x)*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)*(1+x)^2*(6+x)^2*(4+x)^2*(2+x))^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^2*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^4*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^4*(4+x)^2*(2+x)^4+(7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(5+x)^4*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^4*(1+x)^4*(6+x)^2*(4+x)^4*(2+x)^4)*(((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2-(5+x)^2*(3+x)^2*(1+x)^2*((3+x)*(1+x)*(6+x)*(2+x)+(5+x)*(3+x)*(6+x)*(4+x)+(5+x)*(1+x)*(6+x)*(4+x)+(3+x)*(1+x)*(4+x)*(2+x)+(5+x)*(1+x)*(6+x)*(2+x))^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(3+x)^2*(1+x)^2*(6+x)^2*(4+x)^2*(2+x)^2+((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)*(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^2*(3+x)^8*((7+x)*(3+x)*(1+x)*(8+x)*(6+x)*(4+x)+(5+x)*(3+x)*(1+x)*(8+x)*(6+x)*(2+x)+(7+x)*(5+x)*(1+x)*(8+x)*(4+x)*(2+x)+(7+x)*(3+x)*(1+x)*(6+x)*(4+x)*(2+x)+(7+x)*(5+x)*(3+x)*(8+x)*(6+x)*(2+x))^2*(1+x)^8*(6+x)^6*(4+x)^8*(2+x)^8+(7+x)^4*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(1+x)^2*(4+x)^2+(3+x)^2*(1+x)^2*(4+x)^2*(2+x)^2)^4*(5+x)^6*((7+x)^2*((5+x)^2*(1+x)^2*(4+x)^2*(2+x)^2+(5+x)^2*(3+x)^2*(4+x)^2*