using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;
using Fireball.Windows.Forms;
using Fireball.Syntax;
using Fireball.CodeEditor.SyntaxFiles;

namespace C8Asm
{
    public partial class C8Asm : Form
    {
        public static int MAJOR_VERSION = 1;
        public static int MINOR_VERSION = 0;

        private List<PageInfo> mInfos;
        private C8Assembler mAssembler;
        private PageInfo mCurrentInfo;

        public C8Asm()
        {
            InitializeComponent();

            mCurrentInfo = null;
            mInfos = new List<PageInfo>();
            mAssembler = new C8Assembler();
        }

        #region Utility Functions
        private void NewTab()
        {
            CodeEditorControl newBox = new CodeEditorControl();
            newBox.Dock = DockStyle.Fill;
            CodeEditorSyntaxLoader.SetSyntax(newBox, SyntaxLanguage.Chip8);

            tabs.TabPages.Add("Untitled");
            TabPage newTab = tabs.TabPages[tabs.TabPages.Count - 1];

            newTab.Controls.Add(newBox);
            newBox.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom)
                        | System.Windows.Forms.AnchorStyles.Left)
                        | System.Windows.Forms.AnchorStyles.Right)));

            newBox.SetBounds(0, 0, newTab.Bounds.Width, newTab.Bounds.Height);

            PageInfo newInfo = new PageInfo();

            if (tabs.TabPages.Count == 1)
                newInfo.index = 0;
            else
                newInfo.index = tabs.SelectedIndex + 1;

            newInfo.changed = false;
            newInfo.tab = newTab;
            newInfo.file = "";
            newInfo.text = newBox;
            
            newBox.CaretChange += caretChanged;
            newBox.TextChanged += textChanged;

            mInfos.Add(newInfo);
            mCurrentInfo = newInfo;

            UpdateMenus();

            maxLine.Text = mCurrentInfo.text.Document.Lines.Length + "";
            currentLine.Text = "1";
        }

        private void Open()
        {
            if (openFileDlg.ShowDialog() == DialogResult.OK)
            {
                NewTab();

                StreamReader sr = new StreamReader(openFileDlg.OpenFile());
                string text = sr.ReadToEnd();

                ((CodeEditorControl)mCurrentInfo.tab.Controls[0]).Document.Text = text;

                string[] parts = openFileDlg.FileName.Split("\\".ToCharArray());
                mCurrentInfo.tab.Text = parts[parts.Length - 1];
                mCurrentInfo.changed = false;
                mCurrentInfo.file = openFileDlg.FileName;

                tabs.SelectedIndex = tabs.TabPages.Count - 1;
                maxLine.Text = mCurrentInfo.text.Document.Lines.Length + "";
                currentLine.Text = "1";
            }
        }

        public bool Save()
        {
            if (mCurrentInfo.changed)
            {
                if (mCurrentInfo.file.Length != 0)
                {
                    StreamWriter sw = new StreamWriter(mCurrentInfo.file);
                    sw.Write(mCurrentInfo.text.Document.Text);
                    sw.Close();
                    mCurrentInfo.changed = false;
                    string[] parts = mCurrentInfo.file.Split("\\".ToCharArray());
                    mCurrentInfo.tab.Text = parts[parts.Length - 1];
                    saveToolStripMenuItem.Enabled = false;

                    return true;
                }
                else
                {
                    if (saveFileDlg.ShowDialog() == DialogResult.OK)
                    {
                        StreamWriter sw = new StreamWriter(saveFileDlg.OpenFile());
                        sw.Write(mCurrentInfo.text.Document.Text);
                        sw.Close();
                        mCurrentInfo.changed = false;
                        string[] parts = saveFileDlg.FileName.Split("\\".ToCharArray());
                        MessageBox.Show(saveFileDlg.FileName);
                        mCurrentInfo.tab.Text = parts[parts.Length - 1];
                        mCurrentInfo.file = saveFileDlg.FileName;
                        saveToolStripMenuItem.Enabled = false;

                        return true;
                    }

                    return false;
                }
            }

            return true;
        }

        private int CloseTab()
        {
            if (mCurrentInfo.changed)
            {
                string file = mCurrentInfo.tab.Text;

                if (file[0] == '*')
                    file = file.Substring(1, file.Length - 1);

                AskSave ask = new AskSave("Save " + file + "?", "Save file \"" + file + "\"?");
                ask.Visible = false;
                ask.StartPosition = FormStartPosition.CenterParent;

                ask.ShowDialog();

                if (ask.result == 0)
                {
                    if (!Save())
                        return 1;
                }
                else if (ask.result == 2)
                        return 2;
            }

            PageInfo removeInfo = mCurrentInfo;

            tabs.TabPages.Remove(tabs.SelectedTab);
            mInfos.Remove(removeInfo);

            if(tabs.TabPages.Count != 0)
                tabs.SelectedIndex = tabs.TabPages.Count - 1;

            UpdateMenus();

            return 0;
        }

        private void RemovePageInfoByIndex(int index)
        {
            for (int i = 0; i < mInfos.Count; i++)
                if (mInfos[i].index == index)
                    mInfos.RemoveAt(i);
        }

        private PageInfo GetPageInfoByIndex(int index)
        {
            for (int i = 0; i < mInfos.Count; i++)
                if (mInfos[i].index == index)
                    return mInfos[i];

            return null;
        }

        private void UpdateMenus()
        {
            if (tabs.TabPages.Count == 0)
            {
                closeToolStripMenuItem.Enabled = false;
                saveToolStripMenuItem.Enabled = false;

                selectAllToolStripMenuItem.Enabled = false;
                findToolStripMenuItem.Enabled = false;
                cutToolStripMenuItem.Enabled = false;
                copyToolStripMenuItem.Enabled = false;
                pasteToolStripMenuItem.Enabled = false;
                deleteToolStripMenuItem.Enabled = false;

                assembleToolStripMenuItem.Enabled = false;
            }
            else
            {
                closeToolStripMenuItem.Enabled = true;

                selectAllToolStripMenuItem.Enabled = true;
                findToolStripMenuItem.Enabled = true;
                cutToolStripMenuItem.Enabled = true;
                copyToolStripMenuItem.Enabled = true;
                pasteToolStripMenuItem.Enabled = true;
                deleteToolStripMenuItem.Enabled = true;

                assembleToolStripMenuItem.Enabled = true;
            }
        }
#endregion

        #region Event Callbacks
        private void tabs_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (tabs.TabPages.Count != 0)
            {
                mCurrentInfo = GetPageInfoByIndex(((TabControl)sender).SelectedIndex);

                if (mCurrentInfo.changed)
                    saveToolStripMenuItem.Enabled = true;
                else
                    saveToolStripMenuItem.Enabled = false;

                maxLine.Text = mCurrentInfo.text.Document.Lines.Length + "";
                currentLine.Text = (mCurrentInfo.text.Caret.CurrentRow.Index + 1) + "";
                undoToolStripMenuItem.Enabled = mCurrentInfo.text.CanUndo;
                redoToolStripMenuItem.Enabled = mCurrentInfo.text.CanRedo;
            }
            else
            {
                saveToolStripMenuItem.Enabled = false;
                mCurrentInfo = null;
                maxLine.Text = "";
                currentLine.Text = "";
            }
        }

        private void textChanged(object sender, EventArgs e)
        {
            mCurrentInfo.changed = true;
            saveToolStripMenuItem.Enabled = true;

            if (mCurrentInfo.tab.Text[0] != '*')
                mCurrentInfo.tab.Text = "*" + mCurrentInfo.tab.Text;

            undoToolStripMenuItem.Enabled = mCurrentInfo.text.CanUndo;
            redoToolStripMenuItem.Enabled = mCurrentInfo.text.CanRedo;
        }

        private void caretChanged(object sender, EventArgs e)
        {
            currentLine.Text = (mCurrentInfo.text.Caret.CurrentRow.Index + 1) + "";
        }

        private void C8Asm_FormClosing(object sender, FormClosingEventArgs e)
        {
            int ret = -1;
            while (tabs.TabPages.Count != 0)
            {
                ret = CloseTab();
                if (ret == 2)
                    break;
            }

            if (ret == 2)
                e.Cancel = true;
        }
        #endregion

        #region Menu Callbacks
        private void newToolStripMenuItem_Click(object sender, EventArgs e)
        {
            NewTab();
            tabs.SelectedIndex = tabs.TabPages.Count - 1;
        }

        private void assembleToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (mCurrentInfo.text.Document.Text.Length == 0)
            {
                MessageBox.Show("No code to assemble.", "No Code To Assemble", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            if (!Save())
                return;

            mAssembler.asm(mCurrentInfo.text.Document.Text, mCurrentInfo.file);
        }

        private void saveToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Save();
        }

        private void openToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Open();
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            int ret = -1;
            while (tabs.TabPages.Count != 0)
            {
                ret = CloseTab();
                if (ret == 2)
                    break;
            }

            if (ret != 2)
                Application.Exit();
        }

        private void closeToolStripMenuItem_Click(object sender, EventArgs e)
        {
            CloseTab();
        }

        private void selectAllToolStripMenuItem_Click(object sender, EventArgs e)
        {
            mCurrentInfo.text.SelectAll();
        }

        private void undoToolStripMenuItem_Click(object sender, EventArgs e)
        {
            mCurrentInfo.text.Undo();
            undoToolStripMenuItem.Enabled = mCurrentInfo.text.CanUndo;
        }

        private void redoToolStripMenuItem_Click(object sender, EventArgs e)
        {
            mCurrentInfo.text.Redo();
            redoToolStripMenuItem.Enabled = mCurrentInfo.text.CanRedo;
        }

        private void cutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            mCurrentInfo.text.Cut();
        }

        private void copyToolStripMenuItem_Click(object sender, EventArgs e)
        {
            mCurrentInfo.text.Copy();
        }

        private void pasteToolStripMenuItem_Click(object sender, EventArgs e)
        {
            mCurrentInfo.text.Paste();
        }

        private void deleteToolStripMenuItem_Click(object sender, EventArgs e)
        {
            mCurrentInfo.text.Delete();
        }

        private void findToolStripMenuItem_Click(object sender, EventArgs e)
        {
            mCurrentInfo.text.ShowFind();
        }

        private void aboutC8AsmToolStripMenuItem_Click(object sender, EventArgs e)
        {
            MessageBox.Show("C8Asm Version " + C8Asm.MAJOR_VERSION + "." + C8Asm.MINOR_VERSION + "\n\n" +
                            "Les Harris (orpheanjmp@gmail.com)\n" + "For the Emutalk.net boards.", "About C8Asm...", MessageBoxButtons.OK, MessageBoxIcon.Information);
        }
        #endregion
    }

    public class PageInfo
    {
        public int index;
        public bool changed;
        public TabPage tab;
        public string file;
        public CodeEditorControl text;

        public PageInfo()
        {
            index = -1;
            changed = false;
            tab = null;
            file = "";
            text = null;
        }
    }

}