using System;
using System.Collections.Generic;
using System.Collections;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Media;

namespace C8Asm
{
    public class C8Assembler
    {
        #region Structure Definition
        private struct Symbol
        {
            public string mName;
            public int mLine;
        }

        private enum Tokens
        {
            OP_REGISTER,
            OP_LITERAL,
            OP_I,
            OP_LABEL,
            OP_DT,
            OP_ST,
            OP_KEY,
            OP_MEM,
            OP_FONT,
            OP_HFONT,
            UNKNOWN
        }

        private struct Instruction
        {
            public string mName;
            public int mNumberOfOperands;
            public Tokens[] mOperandTypes;
            public string mOpcode;
        }
        #endregion

        #region Fields
        private List<string> mLines;
        private Hashtable mSymbolTable;
        private List<Instruction> mInstructionTable;

        private string mCurrentLexeme;
        private int mCurrentIndex;
        private int mPeek;
        private int mStartingAddress = 0x200;
        private bool mError = false;
        private string mFilename = "";
        private Stream mStream;
        private int mBytes;
        #endregion

        #region Constructor
        public C8Assembler()
        {
            mLines = new List<string>();
            mSymbolTable = new Hashtable();
            mInstructionTable = new List<Instruction>();

            mCurrentLexeme = "";
            mCurrentIndex = 0;
            mPeek = 0;
            mBytes = 0;
        }
        #endregion

        #region API Methods
        public void asm(string code, string filename)
        {
            mLines = new List<string>();
            mSymbolTable = new Hashtable();
            mInstructionTable = new List<Instruction>();

            mCurrentLexeme = "";
            mCurrentIndex = 0;
            mPeek = 0;
            mBytes = 0;

            mFilename = filename + ".o";

            defineInstructions();
            fillLines(code);
            removeComments();
            removeWhitespace();
            fillSymbolTable();

            mStream = File.Open(mFilename, FileMode.Create, FileAccess.Write);
            assemble();
            mStream.Close();

            string symbols = "";
            foreach (string key in mSymbolTable.Keys)
            {
                Symbol s = (Symbol)mSymbolTable[key];
                symbols += s.mName + ": " + s.mLine + "\n";
            }

            AssemblyStats stats = new AssemblyStats(filename, mFilename, mLines.Count, mBytes, symbols);
            stats.Visible = false;
            stats.StartPosition = FormStartPosition.CenterParent;
            
            SystemSounds.Asterisk.Play();
            stats.ShowDialog();
        }
        #endregion

        #region Main Assembler Methods
        private void assemble()
        {
            bool found_inst;

            for (int i = 0; i < mLines.Count; i++)
            {
                mCurrentLexeme = "";
                mCurrentIndex = 0;
                mPeek = 1;
                found_inst = false;

                for (int c = 0; c < mLines[i].Length; c++)
                {
                    mCurrentLexeme += (mLines[i][c] + "");

                    if (IsInstruction(mCurrentLexeme))
                    {
                        if (mPeek != mLines[i].Length)
                            if (IsInstruction(mCurrentLexeme + (mLines[i][mPeek] + "")))
                            {
                                parseLine(mCurrentLexeme + (mLines[i][mPeek] + ""), 
                                    mLines[i].Substring(mCurrentLexeme.Length + 1, mLines[i].Length - mCurrentLexeme.Length + 1), i);
                                if (mError)
                                    return;
                                found_inst = true;
                                break;
                            }

                        parseLine(mCurrentLexeme, 
                            mLines[i].Substring(mCurrentLexeme.Length, mLines[i].Length - mCurrentLexeme.Length), i);
                        if (mError)
                            return;
                        found_inst = true;
                        break;
                    }

                    mCurrentIndex++;
                    mPeek++;
                }

                if (!found_inst)
                {
                    MessageBox.Show("Unknown instruction found at line " + (i + 1) + ": " + mLines[i], "Unknown Instruction",
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
            }
        }

        private void parseLine(string iname, string line, int num)
        {
            Tokens[] op_types = new Tokens[3];
            op_types[0] = Tokens.UNKNOWN;
            op_types[1] = Tokens.UNKNOWN;
            op_types[2] = Tokens.UNKNOWN;

            string[] operands = line.Split(",".ToCharArray());
            Instruction[] insts = GetInstructionByName(iname);

            if (operands.Length > 3)
            {
                MessageBox.Show("Too many operands found at line " + (num + 1) + ": " + mLines[num], "Too Many Operands",
    MessageBoxButtons.OK, MessageBoxIcon.Error);
                mError = true;
                return;
            }

            Instruction inst;
            inst.mOpcode = "";
            inst.mName = "";
            inst.mOperandTypes = null;
            inst.mNumberOfOperands = -1;

            if (operands[0].Length != 0)
            {
                for (int i = 0; i < operands.Length; i++)
                {
                    if (operands[i][0] == 'V')
                    {
                        op_types[i] = Tokens.OP_REGISTER;

                        if (operands[i].Length != 2)
                        {
                            MessageBox.Show("Invalid Register found at line " + (num + 1) + ": " + mLines[num], "Invalid Register",
MessageBoxButtons.OK, MessageBoxIcon.Error);
                            mError = true;
                            return;
                        }

                        operands[i] = operands[i][1] + "";

                        int value = Convert.ToInt32(operands[i], 16);

                        if (value < 0 || value > 15)
                        {
                            MessageBox.Show("Invalid Register found at line " + (num + 1) + " (Must be 0-F): " + mLines[num], "Invalid Register",
MessageBoxButtons.OK, MessageBoxIcon.Error);
                            mError = true;
                            return;
                        }

                        if (iname.Equals("HPS") || iname.Equals("HPL"))
                            if (value < 0 || value > 7)
                            {
                                MessageBox.Show("Invalid HP Register found at line " + (num + 1) + " (Must be 0-7): " + mLines[num], "Invalid HP Register",
MessageBoxButtons.OK, MessageBoxIcon.Error);
                                mError = true;
                                return;
                            }
                    }
                    else if (operands[i][0] == '#')
                    {
                        op_types[i] = Tokens.OP_LITERAL;
                        operands[i] = operands[i].Substring(1, operands[i].Length - 1);
                    }
                    else if (operands[i].Equals("DT"))
                        op_types[i] = Tokens.OP_DT;
                    else if (operands[i].Equals("ST"))
                        op_types[i] = Tokens.OP_ST;
                    else if (operands[i].Equals("I"))
                        op_types[i] = Tokens.OP_I;
                    else if (operands[i].Equals("K"))
                        op_types[i] = Tokens.OP_KEY;
                    else if (operands[i].Equals("[I]"))
                        op_types[i] = Tokens.OP_MEM;
                    else if (operands[i].Equals("F"))
                        op_types[i] = Tokens.OP_FONT;
                    else if (operands[i].Equals("HF"))
                        op_types[i] = Tokens.OP_HFONT;
                    else
                        op_types[i] = Tokens.OP_LABEL;
                }
            }

            if (operands[0].Length == 0)
                inst = insts[0];
            else
            {
                for (int i = 0; i < insts.Length; i++)
                {
                    if (insts[i].mNumberOfOperands != operands.Length)
                        continue;

                    switch (insts[i].mNumberOfOperands)
                    {
                        case 1:
                            if (insts[i].mOperandTypes[0] == op_types[0])
                                inst = insts[i];
                            break;

                        case 2:
                            if (insts[i].mOperandTypes[0] == op_types[0] &&
                               insts[i].mOperandTypes[1] == op_types[1])
                                inst = insts[i];
                            break;

                        case 3:
                            if (insts[i].mOperandTypes[0] == op_types[0] &&
                                insts[i].mOperandTypes[1] == op_types[1] &&
                                insts[i].mOperandTypes[2] == op_types[2])
                                    inst = insts[i];
                            break;
                    }

                    if (inst.mNumberOfOperands != -1)
                        break;
                }
            }

            if (inst.mNumberOfOperands == -1)
            {
                MessageBox.Show("Incorrect operands for instruction " + iname + " found at line " + (num + 1) + ": " + mLines[num], "Incorrect Operands",
MessageBoxButtons.OK, MessageBoxIcon.Error);
                mError = true;
                return;
            }

            parseOpcode(inst, operands, num);
        }

        private void parseOpcode(Instruction inst, string[] operands, int num)
        {
            if (inst.mNumberOfOperands == 0)
                emit(inst.mOpcode);
            else if (inst.mName.Equals("DW"))
                emit(operands[0]);
            else if (inst.mOpcode.Contains("NNN"))
            {
                string key;

                if (inst.mOperandTypes[0] == Tokens.OP_I)
                    key = operands[1];
                else
                    key = operands[0];

                if (mSymbolTable.ContainsKey(key))
                {
                    Symbol s;

                    if (inst.mOperandTypes[0] == Tokens.OP_I)
                        s = (Symbol)mSymbolTable[operands[1]];
                    else
                        s = (Symbol)mSymbolTable[operands[0]];

                    int address = s.mLine + mStartingAddress;

                    if (address > 4095 || address < 0)
                    {
                        MessageBox.Show("Out of range address " + operands[0] + " found at line " + (num + 1) + ": " + mLines[num], "Out of range address",
MessageBoxButtons.OK, MessageBoxIcon.Error);
                        mError = true;
                        return;
                    }

                    string target = Convert.ToString(address, 16).ToUpper();

                    emit(inst.mOpcode[0] + target);
                }
                else
                {
                    MessageBox.Show("Unknown label " + operands[0] + " found at line " + (num + 1) + ": " + mLines[num], "Unknown label",
MessageBoxButtons.OK, MessageBoxIcon.Error);
                    mError = true;
                    return;
                }
            }
            else
            {
                string oc = inst.mOpcode;

                if (oc[1] == 'X')
                {
                    if (inst.mOpcode.Equals("FX55") || inst.mOpcode.Equals("FX1E") || 
                        inst.mOpcode.Equals("FX30") || inst.mOpcode.Equals("FX29"))
                        oc = oc.Replace('X', operands[1].ToCharArray()[0]);
                    else
                        oc = oc.Replace('X', operands[0].ToCharArray()[0]);
                }

                if (oc[2] == 'Y')
                    oc = oc.Replace('Y', operands[1].ToCharArray()[0]);

                if (oc.Contains("KK"))
                    oc = oc.Replace("KK", operands[1].PadLeft(2, '0'));

                if (oc[3] == 'K')
                {
                    if (inst.mNumberOfOperands == 1)
                        oc = oc.Replace('K', operands[0].ToCharArray()[0]);
                    else
                        oc = oc.Replace('K', operands[2].ToCharArray()[0]);
                }

                emit(oc);
            }
        }

        private void emit(string opcode)
        {
            mStream.WriteByte(Convert.ToByte(opcode.Substring(0, 2), 16));
            mStream.WriteByte(Convert.ToByte(opcode.Substring(2, 2), 16));

            mBytes += 2;
        }
#endregion

        #region Utility Methods

        private bool IsInstruction(string name)
        {
            for (int i = 0; i < mInstructionTable.Count; i++)
                if (mInstructionTable[i].mName.Equals(name))
                    return true;

            return false;
        }

        private Instruction[] GetInstructionByName(string name)
        {
            List<Instruction> ret = new List<Instruction>();

            for (int i = 0; i < mInstructionTable.Count; i++)
                if (mInstructionTable[i].mName.Equals(name))
                    ret.Add(mInstructionTable[i]);

            if (ret.Count == 0)
                return null;

            return ret.ToArray();
        }

        private void fillSymbolTable()
        {
            List<string> repl = new List<string>();

            int l = 0;

            for(int i = 0; i < mLines.Count; i++)
            {
                if (mLines[i].EndsWith(":"))
                {
                    Symbol new_symbol;
                    new_symbol.mLine = (l - (mSymbolTable.Count * 2));
                    new_symbol.mName = mLines[i].TrimEnd(":".ToCharArray());

                    mSymbolTable.Add(new_symbol.mName, new_symbol);
                }
                else
                    repl.Add(mLines[i]);

                l += 2;
            }

            mLines = repl;
        }

        private void fillLines(string code)
        {
            string[] lineAry = code.Split("\n".ToCharArray());

            foreach (string s in lineAry)
                if (!s.StartsWith(";"))
                    mLines.Add(s);
        }

        private void removeComments()
        {

            for (int i = 0; i < mLines.Count; i++)
            {
                string replace = "";

                for (int c = 0; c < mLines[i].Length; c++)
                {
                    if (mLines[i][c] == ';')
                    {
                        mLines[i] = replace;
                        break;
                    }

                    replace += (mLines[i][c] + "");
                }
            }

        }

        private void removeWhitespace()
        {
            for (int i = 0; i < mLines.Count; i++)
            {
                string replace = "";

                for (int c = 0; c < mLines[i].Length; c++)
                    if (!char.IsWhiteSpace(mLines[i][c]))
                        replace += (mLines[i][c] + "");

                mLines[i] = replace.ToUpper();
            }

            List<string> repl = new List<string>();

            for (int i = 0; i < mLines.Count; i++)
                if (mLines[i].Length != 0)
                    repl.Add(mLines[i]);

            mLines = repl;
        }

        #endregion

        #region Instruction Definition

        private void defineInstructions()
        {
            Instruction High;
            High.mName = "HIGH";
            High.mNumberOfOperands = 0;
            High.mOperandTypes = null;
            High.mOpcode = "00FF";
            mInstructionTable.Add(High);

            Instruction Scld;
            Scld.mName = "SCLD";
            Scld.mNumberOfOperands = 1;
            Scld.mOperandTypes = new Tokens[Scld.mNumberOfOperands];
            Scld.mOperandTypes[0] = Tokens.OP_LITERAL;
            Scld.mOpcode = "00CK";
            mInstructionTable.Add(Scld);

            Instruction Draw;
            Draw.mName = "DRW";
            Draw.mNumberOfOperands = 3;
            Draw.mOperandTypes = new Tokens[Draw.mNumberOfOperands];
            Draw.mOperandTypes[0] = Tokens.OP_REGISTER;
            Draw.mOperandTypes[1] = Tokens.OP_REGISTER;
            Draw.mOperandTypes[2] = Tokens.OP_LITERAL;
            Draw.mOpcode = "DXYK";
            mInstructionTable.Add(Draw);

            Instruction DW;
            DW.mName = "DW";
            DW.mNumberOfOperands = 1;
            DW.mOperandTypes = new Tokens[DW.mNumberOfOperands];
            DW.mOperandTypes[0] = Tokens.OP_LITERAL;
            DW.mOpcode = "NNNN";
            mInstructionTable.Add(DW);

            Instruction SELit;
            SELit.mName = "SE";
            SELit.mNumberOfOperands = 2;
            SELit.mOperandTypes = new Tokens[SELit.mNumberOfOperands];
            SELit.mOperandTypes[0] = Tokens.OP_REGISTER;
            SELit.mOperandTypes[1] = Tokens.OP_LITERAL;
            SELit.mOpcode = "3XKK";
            mInstructionTable.Add(SELit);

            Instruction SEReg;
            SEReg.mName = "SE";
            SEReg.mNumberOfOperands = 2;
            SEReg.mOperandTypes = new Tokens[SEReg.mNumberOfOperands];
            SEReg.mOperandTypes[0] = Tokens.OP_REGISTER;
            SEReg.mOperandTypes[1] = Tokens.OP_REGISTER;
            SEReg.mOpcode = "5XY0";
            mInstructionTable.Add(SEReg);

            Instruction SNELit;
            SNELit.mName = "SNE";
            SNELit.mNumberOfOperands = 2;
            SNELit.mOperandTypes = new Tokens[SNELit.mNumberOfOperands];
            SNELit.mOperandTypes[0] = Tokens.OP_REGISTER;
            SNELit.mOperandTypes[1] = Tokens.OP_LITERAL;
            SNELit.mOpcode = "4XKK";
            mInstructionTable.Add(SNELit);

            Instruction SNEReg;
            SNEReg.mName = "SNE";
            SNEReg.mNumberOfOperands = 2;
            SNEReg.mOperandTypes = new Tokens[SNEReg.mNumberOfOperands];
            SNEReg.mOperandTypes[0] = Tokens.OP_REGISTER;
            SNEReg.mOperandTypes[1] = Tokens.OP_REGISTER;
            SNEReg.mOpcode = "9XY0";
            mInstructionTable.Add(SNEReg);

            Instruction Call;
            Call.mName = "CALL";
            Call.mNumberOfOperands = 1;
            Call.mOperandTypes = new Tokens[Call.mNumberOfOperands];
            Call.mOperandTypes[0] = Tokens.OP_LABEL;
            Call.mOpcode = "2NNN";
            mInstructionTable.Add(Call);

            Instruction Jump;
            Jump.mName = "JP";
            Jump.mNumberOfOperands = 1;
            Jump.mOperandTypes = new Tokens[Jump.mNumberOfOperands];
            Jump.mOperandTypes[0] = Tokens.OP_LABEL;
            Jump.mOpcode = "1NNN";
            mInstructionTable.Add(Jump);

            Instruction JumpAdd;
            JumpAdd.mName = "JP";
            JumpAdd.mNumberOfOperands = 2;
            JumpAdd.mOperandTypes = new Tokens[JumpAdd.mNumberOfOperands];
            JumpAdd.mOperandTypes[0] = Tokens.OP_LABEL;
            JumpAdd.mOperandTypes[1] = Tokens.OP_REGISTER;
            JumpAdd.mOpcode = "BNNN";
            mInstructionTable.Add(JumpAdd);

            Instruction SKP;
            SKP.mName = "SKP";
            SKP.mNumberOfOperands = 1;
            SKP.mOperandTypes = new Tokens[SKP.mNumberOfOperands];
            SKP.mOperandTypes[0] = Tokens.OP_REGISTER;
            SKP.mOpcode = "EX9E";
            mInstructionTable.Add(SKP);

            Instruction SKNP;
            SKNP.mName = "SKNP";
            SKNP.mNumberOfOperands = 1;
            SKNP.mOperandTypes = new Tokens[SKNP.mNumberOfOperands];
            SKNP.mOperandTypes[0] = Tokens.OP_REGISTER;
            SKNP.mOpcode = "EXA1";
            mInstructionTable.Add(SKNP);

            Instruction Rand;
            Rand.mName = "RND";
            Rand.mNumberOfOperands = 2;
            Rand.mOperandTypes = new Tokens[Rand.mNumberOfOperands];
            Rand.mOperandTypes[0] = Tokens.OP_REGISTER;
            Rand.mOperandTypes[1] = Tokens.OP_LITERAL;
            Rand.mOpcode = "CXKK";
            mInstructionTable.Add(Rand);

            Instruction AddLit;
            AddLit.mName = "ADD";
            AddLit.mNumberOfOperands = 2;
            AddLit.mOperandTypes = new Tokens[AddLit.mNumberOfOperands];
            AddLit.mOperandTypes[0] = Tokens.OP_REGISTER;
            AddLit.mOperandTypes[1] = Tokens.OP_LITERAL;
            AddLit.mOpcode = "7XKK";
            mInstructionTable.Add(AddLit);

            Instruction AddReg;
            AddReg.mName = "ADD";
            AddReg.mNumberOfOperands = 2;
            AddReg.mOperandTypes = new Tokens[AddReg.mNumberOfOperands];
            AddReg.mOperandTypes[0] = Tokens.OP_REGISTER;
            AddReg.mOperandTypes[1] = Tokens.OP_REGISTER;
            AddReg.mOpcode = "8XY4";
            mInstructionTable.Add(AddReg);

            Instruction AddI;
            AddI.mName = "ADD";
            AddI.mNumberOfOperands = 2;
            AddI.mOperandTypes = new Tokens[AddI.mNumberOfOperands];
            AddI.mOperandTypes[0] = Tokens.OP_I;
            AddI.mOperandTypes[1] = Tokens.OP_REGISTER;
            AddI.mOpcode = "FX1E";
            mInstructionTable.Add(AddI);

            Instruction Sub;
            Sub.mName = "SUB";
            Sub.mNumberOfOperands = 2;
            Sub.mOperandTypes = new Tokens[Sub.mNumberOfOperands];
            Sub.mOperandTypes[0] = Tokens.OP_REGISTER;
            Sub.mOperandTypes[1] = Tokens.OP_REGISTER;
            Sub.mOpcode = "8XY5";
            mInstructionTable.Add(Sub);

            Instruction Subn;
            Subn.mName = "SUBN";
            Subn.mNumberOfOperands = 2;
            Subn.mOperandTypes = new Tokens[Subn.mNumberOfOperands];
            Subn.mOperandTypes[0] = Tokens.OP_REGISTER;
            Subn.mOperandTypes[1] = Tokens.OP_REGISTER;
            Subn.mOpcode = "8XY7";
            mInstructionTable.Add(Subn);

            Instruction Ret;
            Ret.mName = "RET";
            Ret.mNumberOfOperands = 0;
            Ret.mOperandTypes = null;
            Ret.mOpcode = "00EE";
            mInstructionTable.Add(Ret);

            Instruction LdLit;
            LdLit.mName = "LD";
            LdLit.mNumberOfOperands = 2;
            LdLit.mOperandTypes = new Tokens[LdLit.mNumberOfOperands];
            LdLit.mOperandTypes[0] = Tokens.OP_REGISTER;
            LdLit.mOperandTypes[1] = Tokens.OP_LITERAL;
            LdLit.mOpcode = "6XKK";
            mInstructionTable.Add(LdLit);

            Instruction LdReg;
            LdReg.mName = "LD";
            LdReg.mNumberOfOperands = 2;
            LdReg.mOperandTypes = new Tokens[LdReg.mNumberOfOperands];
            LdReg.mOperandTypes[0] = Tokens.OP_REGISTER;
            LdReg.mOperandTypes[1] = Tokens.OP_REGISTER;
            LdReg.mOpcode = "8XY0";
            mInstructionTable.Add(LdReg);

            Instruction LdI;
            LdI.mName = "LD";
            LdI.mNumberOfOperands = 2;
            LdI.mOperandTypes = new Tokens[LdI.mNumberOfOperands];
            LdI.mOperandTypes[0] = Tokens.OP_I;
            LdI.mOperandTypes[1] = Tokens.OP_LABEL;
            LdI.mOpcode = "ANNN";
            mInstructionTable.Add(LdI);

            Instruction LdF;
            LdF.mName = "LD";
            LdF.mNumberOfOperands = 2;
            LdF.mOperandTypes = new Tokens[LdF.mNumberOfOperands];
            LdF.mOperandTypes[0] = Tokens.OP_FONT;
            LdF.mOperandTypes[1] = Tokens.OP_REGISTER;
            LdF.mOpcode = "FX29";
            mInstructionTable.Add(LdF);

            Instruction LdHF;
            LdHF.mName = "LD";
            LdHF.mNumberOfOperands = 2;
            LdHF.mOperandTypes = new Tokens[LdHF.mNumberOfOperands];
            LdHF.mOperandTypes[0] = Tokens.OP_HFONT;
            LdHF.mOperandTypes[1] = Tokens.OP_REGISTER;
            LdHF.mOpcode = "FX30";
            mInstructionTable.Add(LdHF);

            Instruction LdMemFromReg;
            LdMemFromReg.mName = "LD";
            LdMemFromReg.mNumberOfOperands = 2;
            LdMemFromReg.mOperandTypes = new Tokens[LdMemFromReg.mNumberOfOperands];
            LdMemFromReg.mOperandTypes[0] = Tokens.OP_MEM;
            LdMemFromReg.mOperandTypes[1] = Tokens.OP_REGISTER;
            LdMemFromReg.mOpcode = "FX55";
            mInstructionTable.Add(LdMemFromReg);

            Instruction LdRegFromMem;
            LdRegFromMem.mName = "LD";
            LdRegFromMem.mNumberOfOperands = 2;
            LdRegFromMem.mOperandTypes = new Tokens[LdRegFromMem.mNumberOfOperands];
            LdRegFromMem.mOperandTypes[0] = Tokens.OP_REGISTER;
            LdRegFromMem.mOperandTypes[1] = Tokens.OP_MEM;
            LdRegFromMem.mOpcode = "FX65";
            mInstructionTable.Add(LdRegFromMem);

            Instruction LdRegDT;
            LdRegDT.mName = "LD";
            LdRegDT.mNumberOfOperands = 2;
            LdRegDT.mOperandTypes = new Tokens[LdRegDT.mNumberOfOperands];
            LdRegDT.mOperandTypes[0] = Tokens.OP_REGISTER;
            LdRegDT.mOperandTypes[1] = Tokens.OP_DT;
            LdRegDT.mOpcode = "FX07";
            mInstructionTable.Add(LdRegDT);

            Instruction LdDT;
            LdDT.mName = "LD";
            LdDT.mNumberOfOperands = 2;
            LdDT.mOperandTypes = new Tokens[LdDT.mNumberOfOperands];
            LdDT.mOperandTypes[0] = Tokens.OP_DT;
            LdDT.mOperandTypes[1] = Tokens.OP_REGISTER;
            LdDT.mOpcode = "FX15";
            mInstructionTable.Add(LdDT);

            Instruction LdST;
            LdST.mName = "LD";
            LdST.mNumberOfOperands = 2;
            LdST.mOperandTypes = new Tokens[LdST.mNumberOfOperands];
            LdST.mOperandTypes[0] = Tokens.OP_ST;
            LdST.mOperandTypes[1] = Tokens.OP_REGISTER;
            LdST.mOpcode = "FX18";
            mInstructionTable.Add(LdST);

            Instruction LdK;
            LdK.mName = "LD";
            LdK.mNumberOfOperands = 2;
            LdK.mOperandTypes = new Tokens[LdK.mNumberOfOperands];
            LdK.mOperandTypes[0] = Tokens.OP_REGISTER;
            LdK.mOperandTypes[1] = Tokens.OP_KEY;
            LdK.mOpcode = "FX0A";
            mInstructionTable.Add(LdK);

            Instruction SCLR;
            SCLR.mName = "SCLR";
            SCLR.mNumberOfOperands = 0;
            SCLR.mOperandTypes = null;
            SCLR.mOpcode = "00FB";
            mInstructionTable.Add(SCLR);

            Instruction SCLL;
            SCLL.mName = "SCLL";
            SCLL.mNumberOfOperands = 0;
            SCLL.mOperandTypes = null;
            SCLL.mOpcode = "00FC";
            mInstructionTable.Add(SCLL);

            Instruction EXIT;
            EXIT.mName = "EXIT";
            EXIT.mNumberOfOperands = 0;
            EXIT.mOperandTypes = null;
            EXIT.mOpcode = "00FD";
            mInstructionTable.Add(EXIT);

            Instruction LOW;
            LOW.mName = "LOW";
            LOW.mNumberOfOperands = 0;
            LOW.mOperandTypes = null;
            LOW.mOpcode = "00FE";
            mInstructionTable.Add(LOW);

            Instruction CLS;
            CLS.mName = "CLS";
            CLS.mNumberOfOperands = 0;
            CLS.mOperandTypes = null;
            CLS.mOpcode = "00E0";
            mInstructionTable.Add(CLS);

            Instruction OR;
            OR.mName = "OR";
            OR.mNumberOfOperands = 2;
            OR.mOperandTypes = new Tokens[OR.mNumberOfOperands];
            OR.mOperandTypes[0] = Tokens.OP_REGISTER;
            OR.mOperandTypes[1] = Tokens.OP_REGISTER;
            OR.mOpcode = "8XY1";
            mInstructionTable.Add(OR);

            Instruction AND;
            AND.mName = "AND";
            AND.mNumberOfOperands = 2;
            AND.mOperandTypes = new Tokens[AND.mNumberOfOperands];
            AND.mOperandTypes[0] = Tokens.OP_REGISTER;
            AND.mOperandTypes[1] = Tokens.OP_REGISTER;
            AND.mOpcode = "8XY2";
            mInstructionTable.Add(AND);

            Instruction XOR;
            XOR.mName = "XOR";
            XOR.mNumberOfOperands = 2;
            XOR.mOperandTypes = new Tokens[XOR.mNumberOfOperands];
            XOR.mOperandTypes[0] = Tokens.OP_REGISTER;
            XOR.mOperandTypes[1] = Tokens.OP_REGISTER;
            XOR.mOpcode = "8XY3";
            mInstructionTable.Add(XOR);

            Instruction SHL;
            SHL.mName = "SHL";
            SHL.mNumberOfOperands = 1;
            SHL.mOperandTypes = new Tokens[SHL.mNumberOfOperands];
            SHL.mOperandTypes[0] = Tokens.OP_REGISTER;
            SHL.mOpcode = "8X0E";
            mInstructionTable.Add(SHL);

            Instruction SHR;
            SHR.mName = "SHR";
            SHR.mNumberOfOperands = 1;
            SHR.mOperandTypes = new Tokens[SHR.mNumberOfOperands];
            SHR.mOperandTypes[0] = Tokens.OP_REGISTER;
            SHR.mOpcode = "8X06";
            mInstructionTable.Add(SHR);

            Instruction BCD;
            BCD.mName = "BCD";
            BCD.mNumberOfOperands = 1;
            BCD.mOperandTypes = new Tokens[BCD.mNumberOfOperands];
            BCD.mOperandTypes[0] = Tokens.OP_REGISTER;
            BCD.mOpcode = "FX33";
            mInstructionTable.Add(BCD);

            Instruction HPS;
            HPS.mName = "HPS";
            HPS.mNumberOfOperands = 1;
            HPS.mOperandTypes = new Tokens[HPS.mNumberOfOperands];
            HPS.mOperandTypes[0] = Tokens.OP_REGISTER;
            HPS.mOpcode = "FX75";
            mInstructionTable.Add(HPS);

            Instruction HPL;
            HPL.mName = "HPL";
            HPL.mNumberOfOperands = 1;
            HPL.mOperandTypes = new Tokens[HPL.mNumberOfOperands];
            HPL.mOperandTypes[0] = Tokens.OP_REGISTER;
            HPL.mOpcode = "FX85";
            mInstructionTable.Add(HPL);
        }

        #endregion
    }
}
