#include <SDL.h>
#include <math.h>

#define uc unsigned char

SDL_Surface *gradient;

unsigned int colorGradient(float t)
{
	unsigned char *pixels = gradient->pixels;
	float st = pow(t - 1, 201) + 1;
	int offset = (int)(st * (gradient->w - 1)) * 3;
	return (pixels[offset]) + (pixels[offset + 1] << 8) + (pixels[offset + 2] << 16);
}

void setPixel(unsigned int x, unsigned int y, unsigned int c, SDL_Surface *s)
{
	if (x >= 0 && x < s->w && y >= 0 && y < s->h)
	{
		unsigned int *pixels = s->pixels;
		pixels[y * s->w + x] = c;
	}
}

float f1(float x)
{
	return -sin(x);
}

float f2(float x)
{
	return tan(3 * x);
}

float f3(float x)
{
	return -sin(x);
}

float f4(float x)
{
	return tan(3 * x);
}

#define MAX_ITER 1000

#define HISTOGRAM

#define width 1000
#define height 600

int main(int argc, char *argv[])
{
	SDL_Surface *screen;
	int hwidth = width / 2, hheight = height / 2;
	SDL_Event event;
	float dt = 0.05;
	int dx = 1, dy = 1, sx, sy;
	int done = 0;
	float cx, cy, cxn, cyn;
	int x, y, n;
	unsigned short *histogram, maxN = 0;
	FILE *histFile;
	int fileExists = 0;

	float scaleX = 4. / width, scaleY = 2.4 / height, tx = -2., ty = -1.2;

	SDL_Init(SDL_INIT_VIDEO);
	screen = SDL_SetVideoMode(width, height, 32, SDL_HWSURFACE);
	SDL_FillRect(screen, NULL, 0xffffff);
	
	gradient = SDL_LoadBMP("gradient.bmp");
	if (!gradient)
	{
		printf("Couldn't load gradient.bmp\n");
		exit(0);
	}

	SDL_LockSurface(gradient);
	histogram = malloc(width*height*sizeof(unsigned short));
	if (!histogram)
	{
		printf("Couldn't allocate histogram\n");
		exit(0);
	}

	SDL_LockSurface(screen);

	histFile = fopen("histogram.txt", "rb");
	if (!histFile)
	{
		histFile = fopen("histogram.txt", "wb");
		fileExists = 0;
	}
	else
		fileExists = 1;

#ifdef HISTOGRAM
	if (!fileExists)
	{
#endif
		for (y = 0; y < height; y += dy)
		{
			for (x = 0; x < width; x += dx)
			{
				cx = scaleX * x + tx; cy = scaleY * y + ty;
				for (n = 0; n < MAX_ITER; n++)
				{
					cxn = cx + f1(cy + f2(cy)) * dt;
					cyn = cy + f3(cx + f4(cx)) * dt;
					sx = (cxn - tx) / scaleX, sy = (cyn - ty) / scaleY;
#ifdef HISTOGRAM
					if (sx >= 0 && sx < width && sy >= 0 && sy < height)
					{
						histogram[sy * width + sx]++;
						maxN = max(maxN, histogram[sy*width + sx]);
					}
#else
					setPixel(sx, sy, (int)(n * 256 / MAX_ITER) * 0x010100 + 0xff, screen);
#endif
					cx = cxn;
					cy = cyn;
				}
			}
			printf("%d %%\n", y * 100 / height);
		}
#ifdef HISTOGRAM
		fputc(maxN & 0xff, histFile);
		fputc(maxN >> 8, histFile);
		fwrite((void*)histogram, sizeof(unsigned short), width*height, histFile);
	}
	else
	{
		maxN = fgetc(histFile) + (fgetc(histFile) << 8);
		fread((void*)histogram, sizeof(unsigned short), width*height, histFile);
	}
#endif

#ifdef HISTOGRAM
	for (y = 0; y < height; y++)
	{
		for (x = 0; x < width; x++)
		{
			setPixel(x, y, colorGradient((float)histogram[y*width+x] / maxN), screen);
		}
	}
#endif

	free(histogram);
	fclose(histFile);
	SDL_UnlockSurface(gradient);
	SDL_UnlockSurface(screen);
	SDL_Flip(screen);

	while (!done)
	{
		SDL_WaitEvent(&event);
		if (event.type == SDL_QUIT) done = 1;
	}

	SDL_FreeSurface(gradient);
	SDL_Quit();

	return 0;
}
